//
//  CLBoundView.m
//  DateLine
//
//  Created by Alex Clarke on 6/05/05.
//  Copyright 2005 CocoaLab. All rights reserved.
//
// Code courtesy of Apple Computer Inc

#import "CLBoundView.h"
#import "CLDateElement.h"

static void *PropertyObservationContext = (void *)1091;
static void *GraphicsObservationContext = (void *)1092;
static void *SelectionIndexesObservationContext = (void *)1093;

@implementation CLBoundView

+ (void)initialize
{
	[self exposeBinding:@"graphics"];
	[self exposeBinding:@"selectionIndexes"];
}

- (NSArray *)exposedBindings
{
	return [NSArray arrayWithObjects:@"graphics", @"selectedObjects", nil];
}

- (id)initWithFrame:(NSRect)frameRect
{
	if ((self = [super initWithFrame:frameRect]) != nil) 
	{
		[self setTransform:[NSAffineTransform transform]];		
	}
	return self;
}

- (void)startObservingGraphics:(NSArray *)graphics
{
	//NSLog(@"observe graphics");
	if ([graphics isEqual:[NSNull null]])
	{
		return;
	}
	/* Register to observe each of the new graphics, and
	 each of their observable properties -- we need old and new
	 values for drawingBounds to figure out what our dirty rect is */
	NSEnumerator *graphicsEnumerator = [graphics objectEnumerator];
	// Declare newGraphic as NSObject * to get key value observing methods
	// Add Graphic protocol for drawing 
	NSObject <Graphic> *newGraphic;
	// Register as observer for all the drawing-related properties 
	while (newGraphic = [graphicsEnumerator nextObject])
	{
		[newGraphic addObserver:self forKeyPath:@"drawingBounds" options:(NSKeyValueObservingOptionNew | NSKeyValueObservingOptionOld) context:PropertyObservationContext];
		
		NSArray *keys = [[CLDateElement class] keysForNonBoundsProperties];
		NSEnumerator *keyEnumerator = [keys objectEnumerator];
		NSString *key;
		while (key = [keyEnumerator nextObject])
		{
			[newGraphic addObserver:self forKeyPath:key options:(NSKeyValueObservingOptionNew | NSKeyValueObservingOptionOld) context:PropertyObservationContext];
		}
	}
}

- (void)stopObservingGraphics:(NSArray *)graphics
{
	if ([graphics isEqual:[NSNull null]])
	{
		return;
	}
	
	NSEnumerator *graphicsEnumerator = [graphics objectEnumerator];
	
    id oldGraphic;
    while (oldGraphic = [graphicsEnumerator nextObject])
	{
		NSArray *keys = [[oldGraphic class] keysForNonBoundsProperties];
		NSEnumerator *keyEnumerator = [keys objectEnumerator];
		NSString *key;
		while (key = [keyEnumerator nextObject])
		{
			[oldGraphic removeObserver:self forKeyPath:key];
		}
		[oldGraphic removeObserver:self forKeyPath:@"drawingBounds"];
	}
}

- (void)bind:(NSString *)bindingName
	toObject:(id)observableObject
 withKeyPath:(NSString *)observableKeyPath
	 options:(NSDictionary *)options
{
	
    if ([bindingName isEqualToString:@"graphics"])
	{
		
		[self setGraphicsContainer:observableObject];
		[self setGraphicsKeyPath:observableKeyPath];
		[graphicsContainer addObserver:self forKeyPath:graphicsKeyPath options:(NSKeyValueObservingOptionNew | NSKeyValueObservingOptionOld) context:GraphicsObservationContext];
		[self startObservingGraphics:[graphicsContainer valueForKeyPath:graphicsKeyPath]];
		
    }
	else if ([bindingName isEqualToString:@"selectionIndexes"])
	{
		[self setSelectionIndexesContainer:observableObject];
		[self setSelectionIndexesKeyPath:observableKeyPath];
		[selectionIndexesContainer addObserver:self forKeyPath:selectionIndexesKeyPath options:(NSKeyValueObservingOptionNew | NSKeyValueObservingOptionOld) context:SelectionIndexesObservationContext];
    }
	
	[super bind:bindingName
	   toObject:observableObject
	withKeyPath:observableKeyPath
		options:options];
}

- (void)unbind:(NSString *)bindingName {
	
    if ([bindingName isEqualToString:@"graphics"])
	{
		[graphicsContainer removeObserver:self forKeyPath:graphicsKeyPath];
		[self setGraphicsContainer:nil];
		[self setGraphicsKeyPath:nil];
    }
	
	if ([bindingName isEqualToString:@"selectionIndexes"])
	{
		[selectionIndexesContainer removeObserver:self forKeyPath:selectionIndexesKeyPath];
		[self setSelectionIndexesContainer:nil];
		[self setSelectionIndexesKeyPath:nil];
	}
	
	[super unbind:bindingName];
}


- (void)observeValueForKeyPath:(NSString *)keyPath
					  ofObject:(id)object
						change:(NSDictionary *)change
					   context:(void *)context
{
	
    if (context == GraphicsObservationContext)
	{
		/*
		 Should be able to use
		 NSArray *oldGraphics = [change objectForKey:NSKeyValueChangeOldKey];
		 etc. but the dictionary doesn't contain old and new arrays...??
		 */
		NSArray *newGraphics = [object valueForKeyPath:graphicsKeyPath];
		
		NSMutableArray *onlyNew = [newGraphics mutableCopy];
		[onlyNew removeObjectsInArray:oldGraphics];
		[self startObservingGraphics:onlyNew];
		[onlyNew release];
		
		NSMutableArray *removed = [oldGraphics mutableCopy];
		[removed removeObjectsInArray:newGraphics];
		[self stopObservingGraphics:removed];
		[removed release];
		
		[self setOldGraphics:newGraphics];
		
		// could check drawingBounds of old and new, but...
		[self setNeedsDisplay:YES];
		return;
    }
	
	if (context == PropertyObservationContext)
	{
		NSRect updateRect;
		
		// Note: for CLPage, drawingBounds is a dependent key of all the other
		// property keys except color, so we'll get this anyway...
		if ([keyPath isEqualToString:@"drawingBounds"])
		{
			NSRect newBounds = [[change objectForKey:NSKeyValueChangeNewKey] rectValue];
			NSRect oldBounds = [[change objectForKey:NSKeyValueChangeOldKey] rectValue];
			updateRect = NSUnionRect(newBounds,oldBounds);
		}
		else
		{
			updateRect = [(CLDateElement<Graphic> *)object drawingBounds];
		}
		updateRect = NSMakeRect(updateRect.origin.x-1.0,
								updateRect.origin.y-1.0,
								updateRect.size.width+2.0,
								updateRect.size.height+2.0);
		[self setNeedsDisplayInRect:updateRect];
		return;
	}
	
	if (context == SelectionIndexesObservationContext)
	{
		[self setNeedsDisplay:YES];
		return;
	}
}

- (void)drawRect:(NSRect)rect
{
	[super drawRect:rect];
}

- (void)mouseDown:(NSEvent *)event
{
	[super mouseDown:event];
}

- (void) setFrameSize:(NSSize)newSize 
{
    [super setFrameSize:newSize];
    // A change in size has required the view to be invalidated.
    if ([self inLiveResize]) 
    {
        NSRect rects[4];
        int count;
        [self getRectsExposedDuringLiveResize:rects count:&count];
        while (count-- > 0) 
        {
            [self setNeedsDisplayInRect:rects[count]];
        }
    } 
    else 
    {
        [self setNeedsDisplay:YES];
    }
}

- (NSAffineTransform *) transform  
{
	return transform; 
}
- (void) setTransform:(NSAffineTransform *) aTransform
{
	[aTransform retain];
	[transform release];
	transform = aTransform;
	[self setNeedsDisplay:YES];
}

- (BOOL)preservesContentDuringLiveResize
{
	return YES;
}

-(BOOL)isOpaque
{
	return YES;
}


-(void) setElementColor:(NSColor *)elementColor
{
	NSArray * graphicsArray = [self graphics];
	
	NSEnumerator *graphicsEnumerator = [graphicsArray objectEnumerator];
	id graphic;
	while (graphic = [graphicsEnumerator nextObject])
	{
		[graphic setColor:elementColor];
	}
}

-(void) setTextColor:(NSColor *)textColor
{
	NSArray * graphicsArray = [self graphics];
	
	NSEnumerator *graphicsEnumerator = [graphicsArray objectEnumerator];
	id graphic;
	while (graphic = [graphicsEnumerator nextObject])
	{
		[graphic setTextColor:textColor];
	}
}

-(void) setTodayTextColor:(NSColor *)todayTextColor
{
	NSArray * graphicsArray = [self graphics];
	
	NSEnumerator *graphicsEnumerator = [graphicsArray objectEnumerator];
	id graphic;
	while (graphic = [graphicsEnumerator nextObject])
	{
		[graphic setTodayTextColor:todayTextColor];
	}
}

-(void) setTodayMarkColor:(NSColor *)todayMarkColor
{
	NSArray * graphicsArray = [self graphics];
	
	NSEnumerator *graphicsEnumerator = [graphicsArray objectEnumerator];
	id graphic;
	while (graphic = [graphicsEnumerator nextObject])
	{
		[graphic setTodayMarkColor:todayMarkColor];
	}
}

-(void) setDayMarkColor:(NSColor *)dayMarkColor
{
	NSArray * graphicsArray = [self graphics];
	
	NSEnumerator *graphicsEnumerator = [graphicsArray objectEnumerator];
	id graphic;
	while (graphic = [graphicsEnumerator nextObject])
	{
		[graphic setDayMarkColor:dayMarkColor];
	}
}

#pragma mark Accessors

- (NSArray *)graphics
{	
    return [graphicsContainer valueForKeyPath:graphicsKeyPath];	
}

- (NSIndexSet *)selectionIndexes
{
	return [selectionIndexesContainer valueForKeyPath:selectionIndexesKeyPath];
}

- (NSArray *)oldGraphics { return oldGraphics; }

- (void)setOldGraphics:(NSArray *)anOldGraphics
{
    if (oldGraphics != anOldGraphics) {
        [oldGraphics release];
        oldGraphics = [anOldGraphics mutableCopy];
    }
}

- (NSObject *)graphicsContainer
{
    return graphicsContainer; 
}
- (void)setGraphicsContainer:(NSObject *)aGraphicsContainer
{
    if (graphicsContainer != aGraphicsContainer) {
        [graphicsContainer release];
        graphicsContainer = [aGraphicsContainer retain];
    }
}

- (NSObject *)selectionIndexesContainer
{
    return selectionIndexesContainer; 
}
- (void)setSelectionIndexesContainer:(NSObject *)aSelectionIndexesContainer
{
    if (selectionIndexesContainer != aSelectionIndexesContainer) {
        [selectionIndexesContainer release];
        selectionIndexesContainer = [aSelectionIndexesContainer retain];
    }
}

- (NSString *)graphicsKeyPath
{
    return graphicsKeyPath; 
}
- (void)setGraphicsKeyPath:(NSString *)aGraphicsKeyPath
{
    if (graphicsKeyPath != aGraphicsKeyPath) {
        [graphicsKeyPath release];
        graphicsKeyPath = [aGraphicsKeyPath copy];
    }
}


- (NSString *)selectionIndexesKeyPath
{
    return selectionIndexesKeyPath; 
}
- (void)setSelectionIndexesKeyPath:(NSString *)aSelectionIndexesKeyPath
{
    if (selectionIndexesKeyPath != aSelectionIndexesKeyPath) {
        [selectionIndexesKeyPath release];
        selectionIndexesKeyPath = [aSelectionIndexesKeyPath copy];
    }
}

+ (Class)class
{
	return [super class];
}

-(void) dealloc
{
	[self unbind:@"selectionIndexes"];
	[self unbind:@"graphics"];
	[super dealloc];	
}


@end
