/**********************************************************************
Copyright (c) 2002 Kelly Grizzle (TJDO) and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2003 Erik Bengtson - removed ununsed imports
2003 Andy Jefferson - coding standards
2004 Andy Jefferson - renamed from SetLiteral
    ...
**********************************************************************/
package org.datanucleus.store.mapped.expression;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import org.datanucleus.store.mapped.mapping.JavaTypeMapping;

/**
 * An SQL expression that will test if a column of a table falls within the
 * given Collection of values.  This is used for Querys where a transient
 * Collection is passed in as a parameter.
 */
public class CollectionLiteral extends ScalarExpression
{
    private final boolean isEmpty;
    private final boolean containsNull;
    /** ScalarExpressions for all elements in the Collection **/ 
    private List scalarExpressions;

    /**
     * Constructor.
     * @param qs     The QueryStatement the CollectionLiteral will be used in.
     * @param value  The transient Collection that is the value.
     * @param mapping The mapping to the Collection
     */
    public CollectionLiteral(QueryExpression qs, JavaTypeMapping mapping, Collection value)
    {
        super(qs);

        this.mapping = mapping;
        containsNull = (value != null) && value.contains(null);

        /*
         * We'll consider the Collection to be empty if it is null, is really
         * empty, or only contains null. 
         * If it contains null we need a special case when creating the SQL.
         */
        isEmpty =
            (value == null) ||
            (value.isEmpty()) ||
            (value.size() == 1 && containsNull);

        // If the Collection is empty, don't build the list of SQLExpressions.
        if (!isEmpty)
        {
            scalarExpressions = new ArrayList();
            st.append("(");

            boolean hadPrev = false;

            for (Iterator it=value.iterator(); it.hasNext();)
            {
                Object current = it.next();
                if (current != null)
                {
                    JavaTypeMapping m = qs.getStoreManager().getMappingManager().getMappingWithDatastoreMapping(
                        current.getClass(), false, false, qs.getClassLoaderResolver());
                    ScalarExpression expr = m.newLiteral(qs,current);

                    // Append the SQLExpression (should be a literal) for the
                    // current element.
                    st.append(hadPrev ? "," : "");
                    st.append(expr);
                    scalarExpressions.add(expr);

                    hadPrev = true;
                }
            }

            st.append(")");
        }
    }

    /**
     * Method to check the containing of an element.
     * Return the BooleanExpression that results from 
     * CollectionLiteral.contains(SQLExpression).
     * 
     * @param expr The SQLExpression that is checked for membership in the
     *             Collection
     * @return The BooleanExpression that results from 
     *         CollectionLiteral.contains(SQLExpression).
     */
    public BooleanExpression containsMethod(ScalarExpression expr)
    {
        if( isEmpty )
        {
            return new BooleanLiteral(qs, mapping, false);
        }
        BooleanExpression bExpr = null;
        for( int i=0; i<scalarExpressions.size(); i++)
        {
            if( bExpr == null )
            {
                bExpr = ((ScalarExpression)scalarExpressions.get(i)).eq(expr); 
            }
            else
            {
                bExpr = bExpr.ior(((ScalarExpression)scalarExpressions.get(i)).eq(expr)); 
            }
        }
        bExpr.encloseWithInParentheses();
        return bExpr;
    }

    /**
     * Method to check for emptiness of the collection.
     * @return The BooleanExpression.
     **/
    public BooleanExpression isEmptyMethod()
    {
        return new BooleanLiteral(qs, mapping, isEmpty);
    }
}