/**********************************************************************
Copyright (c) 2002 Mike Martin (TJDO) and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2003 Andy Jefferson - coding standards
    ...
**********************************************************************/
package org.datanucleus.store.mapped.expression;

import java.util.ArrayList;
import java.util.Iterator;

import org.datanucleus.store.mapped.DatastoreAdapter;
import org.datanucleus.store.mapped.DatastoreClass;
import org.datanucleus.store.mapped.DatastoreContainerObject;
import org.datanucleus.store.mapped.DatastoreField;
import org.datanucleus.store.mapped.DatastoreIdentifier;

/**
 * A SQL table expression that joins superclass tables by constructing a parenthesized set of subjoins.
 * Used by Informix.
 * @see QueryExpression
 */
public class TableExprAsSubjoins extends LogicSetExpression
{
    protected final ArrayList supertables = new ArrayList();

    /**
     * 
     * @param qs the QueryExpression
     * @param mainTable the main table in the query
     * @param mainRangeVar The alias/identifier of the table in the query
     */
    public TableExprAsSubjoins(QueryExpression qs, DatastoreContainerObject mainTable, DatastoreIdentifier mainRangeVar)
    {
        super(qs, mainTable, mainRangeVar);
    }

    public String referenceColumn(DatastoreField col)
    {
        assertNotFrozen();

        DatastoreContainerObject table = col.getDatastoreContainerObject();

        if (!table.equals(mainTable))
        {
            if (!(mainTable instanceof DatastoreClass) || !(table instanceof DatastoreClass))
            {
                throw new TableMismatchException(col, mainTable);
            }

            /*
             * Since both tables are ClassTables we assume that the column
             * is a superclass field, meaning 'table' is a supertable of
             * 'mainTable'.  We add it to the list of supertables that will be
             * joined in when the statement text is constructed.
             */
            if (!(supertables.contains(table)))
            {
                supertables.add(table);
            }
        }

        if( mainAlias.toString().length() > 0)
        {
            return col.applySelectFunction(mainAlias + "." + col.getIdentifier());
        }
        else
        {
            return col.applySelectFunction(col.getIdentifier().toString());
        }
    }

    public String toString()
    {
        if (sqlText == null)
        {
            StringBuffer sb = new StringBuffer();
            DatastoreIdentifier mainTableName = mainTable.getIdentifier();

            Iterator i = supertables.iterator();
            if (i.hasNext())
            {
                sb.append('(');
                if (this.getMainTable().getStoreManager().getDatastoreAdapter().supportsOption(DatastoreAdapter.PROJECTION_IN_TABLE_REFERENCE_JOINS))
                {
                    sb.append("SELECT * FROM ");
                }
                sb.append(mainTable.toString());
                while (i.hasNext())
                {
                    DatastoreClass supertable = (DatastoreClass)i.next();

					sb.append(" INNER JOIN ").append(supertable.toString());
					sb.append(" ON ");
					for (int j=0; j<((DatastoreClass) mainTable).getIDMapping().getNumberOfDatastoreFields(); j++)
					{
						DatastoreIdentifier mainTableIDColumnName = ((DatastoreClass)mainTable).getIDMapping().getDataStoreMapping(j).getDatastoreField().getIdentifier();
						if (j > 0)
						{
							sb.append(" AND ");
						}
						sb.append(mainTable.toString()).append('.').append(mainTableIDColumnName);
						sb.append(" = ");
						sb.append(supertable.toString()).append('.').append(supertable.getIDMapping().getDataStoreMapping(j).getDatastoreField().getIdentifier());
					}
                }

                sb.append(") ").append(mainAlias);
            }
            else
            {
                sb.append(mainTable.toString());
                if (!mainAlias.equals(mainTableName))
                {
                    sb.append(' ').append(mainAlias);
                }
            }

            sqlText = sb.toString();
        }

        return sqlText;
    }
}
