/**********************************************************************
 Copyright (c) 2005 Andy Jefferson and others. All rights reserved.
 Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. 
 

 Contributors:
 ...
 **********************************************************************/
package org.datanucleus.store.mapped.query;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.ObjectManagerFactoryImpl;
import org.datanucleus.metadata.AbstractClassMetaData;
import org.datanucleus.store.StoreManager;
import org.datanucleus.store.mapped.DatastoreAdapter;
import org.datanucleus.store.mapped.DatastoreClass;
import org.datanucleus.store.mapped.DatastoreIdentifier;
import org.datanucleus.store.mapped.MappedStoreManager;
import org.datanucleus.store.mapped.expression.QueryExpression;
import org.datanucleus.util.Localiser;

/**
 * Abstract representation of a statement that is used for iterating through a list of
 * objects of a class, possibly including subclasses.
 */
public abstract class AbstractIteratorStatement
{
    /** Localisation or messages. */
    protected static final Localiser LOCALISER = Localiser.getInstance(
        "org.datanucleus.Localisation", ObjectManagerFactoryImpl.class.getClassLoader());

    /** Whether to include iteration through subclasses of the candidate. */
    protected final boolean includeSubclasses;

    /** Manager for the Store. */
    protected final MappedStoreManager storeMgr;

    /** Datastore adapter */
    protected final DatastoreAdapter dba;
    
    /** full class name for the candidate class **/
    protected String candidateFullClassName;

    /** Table where the candidate objects are stored. */
    protected DatastoreClass candidateTable;

    /** ClassLoader resolver. */
    protected final ClassLoaderResolver clr;

    /**
     * Constructor.
     * @param type Class that we are querying for as our base.
     * @param clr ClassLoaderResolver
     * @param subclasses Whether to include subclasses in the iteration.
     * @param storeMgr Manager for the store
     */
    public AbstractIteratorStatement(Class type, ClassLoaderResolver clr, boolean subclasses, StoreManager storeMgr)
    {
        this.includeSubclasses = subclasses;
        this.storeMgr = (MappedStoreManager)storeMgr;
        this.dba = ((MappedStoreManager)storeMgr).getDatastoreAdapter();
        this.clr = clr;
        AbstractClassMetaData cmd = storeMgr.getOMFContext().getMetaDataManager().getMetaDataForClass(type, clr);
        if (cmd == null)
        {
            candidateFullClassName = type.getName();
        }
        else
        {
            candidateFullClassName = cmd.getFullClassName();
        }

        if (!this.storeMgr.getMappedTypeManager().isSupportedMappedType(candidateFullClassName))
        {
            candidateTable = this.storeMgr.getDatastoreClass(candidateFullClassName, clr);
        }
    }

    /**
     * Accessor for the QueryStatement that will return the objects of the candidate
     * type and its subclasses (if required).
     * @param candidateAlias Alias for the candidate
     * @return The QueryExpression
     */
    public abstract QueryExpression getQueryStatement(DatastoreIdentifier candidateAlias);
}