/**********************************************************************
Copyright (c) 2006 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. 

Contributors:
    ...
**********************************************************************/
package org.datanucleus.enhancer;

import java.io.File;
import java.io.FileOutputStream;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.metadata.AbstractClassMetaData;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.metadata.ClassMetaData;
import org.datanucleus.metadata.InterfaceMetaData;

/**
 * Abstract representation of a generator of implementations of abstract-classes/interfaces.
 */
public abstract class AbstractImplementationGenerator
{
    /** Meta data for the abstract-class/interface **/
    protected final AbstractClassMetaData inputCmd;

    /** Class name of the implementation. **/
    protected final String className;

    /** Fully-qualified class name (including package) of the implementation. **/
    protected final String fullClassName;

    /** Class name for the superclass. **/
    protected String fullSuperclassName = "java.lang.Object";

    /** bytes for the implementation class. **/
    protected byte[] bytes;

    /**
     * Constructor for an implementation of an interface.
     * @param interfaceMetaData The MetaData for the interface
     * @param implClassName Name of the implementation class to generate (omitting packages)
     */
    public AbstractImplementationGenerator(InterfaceMetaData interfaceMetaData, String implClassName)
    {
        this.className = implClassName;
        this.fullClassName = interfaceMetaData.getPackageName() + '.' + className;
        this.inputCmd = interfaceMetaData;
    }

    /**
     * Constructor for an implementation of an abstract class.
     * @param classMetaData The MetaData for the abstract class
     * @param implClassName Name of the implementation class to generate (omitting packages)
     */
    public AbstractImplementationGenerator(ClassMetaData classMetaData, String implClassName)
    {
        this.className = implClassName;
        this.fullClassName = classMetaData.getPackageName() + '.' + className;
        this.inputCmd = classMetaData;
    }

    /**
     * Enhance the implementation for use in the persistence process.
     * @param clr ClassLoader Resolver 
     */
    public abstract void enhance(ClassLoaderResolver clr);

    /**
     * Accessor for the byte representation of the generated class.
     * @return the byte representation of the class
     */
    public byte[] getBytes()
    {
        return bytes;
    }

    /**
     * Creates fields for the properties of this class and super classes.
     */
    protected void createPropertyFields()
    {
        AbstractClassMetaData acmd = inputCmd;
        do
        {
            createPropertyFields(acmd);
            acmd = acmd.getSuperAbstractClassMetaData();
        }
        while (acmd != null);
    }

    /**
     * Creates fields for the properties of the specified class/interface.
     * @param acmd MetaData for the class/interface 
     */
    protected abstract void createPropertyFields(AbstractClassMetaData acmd);

    /**
     * Create getters and setters methods for this class and super classes
     */
    protected void createPropertyMethods()
    {
        AbstractClassMetaData acmd = inputCmd;
        do
        {
            createPropertyMethods(acmd);
            acmd = acmd.getSuperAbstractClassMetaData();
        }
        while (acmd != null);
    }  

    /**
     * Create getters and setters methods.
     * @param acmd AbstractClassMetaData
     */
    protected void createPropertyMethods(AbstractClassMetaData acmd)
    {
        if (acmd == null)
        {
            return;
        }

        AbstractMemberMetaData[] memberMetaData = acmd.getManagedMembers();
        for (int i=0; i<memberMetaData.length; i++)
        {
            createGetter(memberMetaData[i]);
            createSetter(memberMetaData[i]);
        }
    }

    /**
     * Create a getter method for the specified property.
     * @param mmd MetaData for the property
     */
    protected abstract void createGetter(AbstractMemberMetaData mmd);

    /**
     * Create a setter method for the specified property.
     * @param mmd MetaData for the property
     */
    protected abstract void createSetter(AbstractMemberMetaData mmd);

    /**
     * Convenience method to dump the generated class to the specified file.
     * @param filename
     */
    public void dumpToFile(String filename)
    {
        FileOutputStream out = null;
        try
        {
            out = new FileOutputStream(new File(filename));
            out.write(getBytes());
            DataNucleusEnhancer.LOGGER.info("Generated class for " + fullClassName + " dumped to " + filename);
        }
        catch (Exception e)
        {
            DataNucleusEnhancer.LOGGER.error("Failure to dump generated class to file", e);
        }
        finally
        {
            try
            {
                out.close();
                out = null;
            }
            catch (Exception ignore)
            {
                // ignore exception in closing the stream
            }
        }
    }
}