/**********************************************************************
Copyright (c) 2002 Kelly Grizzle (TJDO) and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2002 Mike Martin (TJDO)
2003 Andy Jefferson - coding standards
    ...
**********************************************************************/
package org.datanucleus.store.mapped.expression;

import org.datanucleus.store.mapped.mapping.JavaTypeMapping;

/**
 * Representation of BooleanBit column expression in a Query.
 *
 * @version $Revision: 1.10 $
 **/
public class BooleanBitColumnExpression extends BooleanExpression
{
    private boolean truthTest;
    private StatementText booleanCondition = new StatementText();

    /**
     * Constructor. Perform a positive truth test("<i>this</i> = 1").
     * @param qs the QueryExpression
     * @param mapping the mapping associated to this expression
     * @param te the TableExpression where this expression refers to
     */
    public BooleanBitColumnExpression(QueryExpression qs, JavaTypeMapping mapping, LogicSetExpression te)
    {
        this(qs, mapping, te, true);
    }

    /**
     * Constructor. Perform a truth test
     * @param qs the QueryExpression
     * @param mapping the mapping associated to this expression
     * @param te the TableExpression where this expression refers to
     * @param truthTest true for positive ("<i>this</i> = 1"), 0 for negative ("<i>this</i> = 0")
     */
    public BooleanBitColumnExpression(QueryExpression qs, JavaTypeMapping mapping, LogicSetExpression te, boolean truthTest)
    {
        super(qs, mapping, te);

        this.truthTest = truthTest;
        booleanCondition.append(" = ").append(new BooleanBitColumnLiteral(qs, mapping, truthTest));

        lowestOperator = OP_EQ;
    }

    public BooleanExpression not()
    {
        return new BooleanBitColumnExpression(qs, mapping, te, !truthTest);
    }

    public BooleanExpression eq(ScalarExpression expr)
    {
        if (expr instanceof NullLiteral)
        {
            return expr.eq(new NumericExpression(qs, mapping, te));
        }
        else if (expr instanceof BooleanBitColumnExpression)
        {
            return new BooleanExpression(new NumericExpression(qs, mapping, te),
                                         OP_EQ,
                                         new NumericExpression(qs, expr.mapping, expr.te));
        }
        else
        {
            return super.eq(expr);
        }
    }

    public BooleanExpression noteq(ScalarExpression expr)
    {
        if (expr instanceof NullLiteral)
        {
            return expr.noteq(new NumericExpression(qs, mapping, te));
        }
        else if (expr instanceof BooleanBitColumnExpression)
        {
            return new BooleanExpression(new NumericExpression(qs, mapping, te),
                                         OP_NOTEQ,
                                         new NumericExpression(qs, expr.mapping, expr.te));
        }
        else
        {
            return super.noteq(expr);
        }
    }

    public StatementText toStatementText(int mode)
    {
        StatementText st = super.toStatementText(mode);
        if (mode == ScalarExpression.FILTER)
        {
            st.append(booleanCondition,mode);
        }
        return st;
    }
}