/**********************************************************************
Copyright (c) 2005 Erik Bengtson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.mapped.expression;

/**
 * Every expression written in the Java programming language has a type that can
 * be deduced from the structure of the expression and the types of the
 * literals, variables, and methods mentioned in the expression. It is possible,
 * however, to write an expression in a context where the type of the expression
 * is not appropriate. In some cases, this leads to an error at compile time. In
 * other cases, the context may be able to accept a type that is related to the
 * type of the expression; as a convenience, rather than requiring the
 * programmer to indicate a type conversion explicitly, the language performs an
 * implicit conversion from the type of the expression to a type acceptable for
 * its surrounding context.
 * <p>
 * Supported conversion types:
 * <ul>
 * <li>Widening conversion</li>
 * <li>String conversion</li>
 * </ul>
 * </p>
 * <h4>Widening conversion</h4>
 * <p>
 * Widening conversions do not lose information about the overall magnitude of a
 * numeric value.
 * </p>
 * <h4>String conversion</h4>
 * <p>
 * If only one operand expression is of type String, then string conversion is
 * performed on the other operand to produce a string at run time. The result is
 * a reference to a String object (newly created, unless the expression is a
 * compile-time constant expression (15.28))that is the concatenation of the
 * two operand strings. The characters of the left-hand operand precede the
 * characters of the right-hand operand in the newly created string. If an
 * operand of type String is null, then the string "null" is used instead of
 * that operand.
 * </p>
 * @version $Revision: 1.2 $
 */
public interface ExpressionConversionAdapter
{
    /**
     * A Widening conversion that returns the appropriate expression for the
     * <code>(int)'A'</code> expression. In SQL, it should compile something
     * like:
     * <p>
     * <blockquote>
     * <pre>
     * ASCII('A')
     * </pre>
     * </blockquote>
     * </p>
     * @param expr The CharacterExpression
     * @return The NumericExpression
     */
    NumericExpression toNumericExpression(CharacterExpression expr);

    /**
     * A String conversion that converts a numeric expression to string.
     * If the expr argument represents a Literal value, converts to a Literal string.
     * In SQL, it should compile something like:
     * <p>
     * <blockquote>
     * <pre>
     * CAST(999999 AS VARCHAR(4000))
     * </pre>
     * </blockquote>
     * </p>
     * @param expr The NumericExpression
     * @return the StringExpression
     */
    StringExpression toStringExpression(NumericExpression expr);
    
    /**
     * A String conversion that converts a String literal to String expression. It will allow
     * the String to be evaluated only at runtime.
     * In SQL, it should compile something like:
     * <p>
     * <blockquote>
     * <pre>
     * CAST('value' AS VARCHAR(4000))
     * </pre>
     * </blockquote>
     * </p>
     * @param expr The StringLiteral
     * @return the StringExpression
     */
    StringExpression toStringExpression(StringLiteral expr);    
}