/**********************************************************************
Copyright (c) 2004 Erik Bengtson and others. All rights reserved. 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
 

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.mapped.expression;

import java.util.Map;

import org.datanucleus.store.mapped.mapping.JavaTypeMapping;

/**
 * An SQL expression that will test if a column of a table falls within the
 * given Map. This is used for Querys where a Map is passed in as a parameter.
 * @version $Revision: 1.6 $
 */
public class MapLiteral extends ScalarExpression
{
    private final boolean isEmpty;
    private final boolean containsNull;
    private final MapValueLiteral mapValueLiteral;
    private final MapKeyLiteral mapKeyLiteral; 

    /**
     * Constructor.
     * @param qs  The QueryStatement the MapLiteral will be used in.
     * @param mapping The mapping to the Map
     * @param map The Map that is the value.
     */
    public MapLiteral(QueryExpression qs, JavaTypeMapping mapping, Map map)
    {
        super(qs);
        this.mapping = mapping;
        containsNull = (map != null) && map.containsValue(null);
        mapValueLiteral = new MapValueLiteral(qs,mapping,map);
        mapKeyLiteral = new MapKeyLiteral(qs,mapping,map);

        // We'll consider the Map to be empty if it is null, is really
        // empty, or only contains null. 
        // If it contains null we need a special case when creating the SQL.
        isEmpty =
            (map == null) ||
            (map.isEmpty()) ||
            (map.size() == 1 && containsNull);
    }

    /**
     * Method to check the containing of a value in the Map.
     * Return the BooleanExpression that results from 
     * MapValueLiteral.contains(SQLExpression).
     * @param expr The SQLExpression that is checked for membership in the Map.
     * @return The BooleanExpression that results from 
     *         MapValueLiteral.contains(SQLExpression).
     */
    public BooleanExpression containsMethod(ScalarExpression expr)
    {
        return mapValueLiteral.containsMethod(expr);
    }
    
    /**
     * Method to check the containing of a value in the Map.
     * Return the BooleanExpression that results from 
     * MapValueLiteral.contains(SQLExpression).
     * @param expr The SQLExpression that is checked for membership in the Map.
     * @return The BooleanExpression that results from 
     *         MapValueLiteral.contains(SQLExpression).
     */
    public BooleanExpression containsValueMethod(ScalarExpression expr)
    {
        return mapValueLiteral.containsMethod(expr);
    }

    /**
     * Method to check the containing of a key in the Map.
     * Return the BooleanExpression that results from 
     * MapKeyLiteral.contains(SQLExpression).
     * @param expr The SQLExpression that is checked for membership in the Map.
     * @return The BooleanExpression that results from 
     *         MapKeyLiteral.contains(SQLExpression).
     */
    public BooleanExpression containsKeyMethod(ScalarExpression expr)
    {
        return mapKeyLiteral.containsMethod(expr);
    }
    
    /**
     * Method to check the containing of an entry in the Map.
     * Return the BooleanExpression that results from 
     * MapValueLiteral.contains(SQLExpression) and MapKeyLiteral.contains(SQLExpression).
     * @param expr The SQLExpression that is checked for membership in the Map.
     * @return The BooleanExpression that results from 
     *         MapValueLiteral.contains(SQLExpression) and MapKeyLiteral.contains(SQLExpression).
     */
    public BooleanExpression containsEntryMethod(ScalarExpression expr)
    {
        return mapKeyLiteral.containsMethod(expr).and(mapValueLiteral.containsMethod(expr));
    }
    
    /**
     * Method to get a value from the Map for a key
     * @param expr The key argument expression
     * @return The statement
     **/
    public ScalarExpression getMethod(ScalarExpression expr)
    {
        return mapKeyLiteral.getMethod(expr);
    }
    
    /**
     * Method to check for emptiness of the collection.
     * @return The BooleanExpression.
     **/
    public BooleanExpression isEmptyMethod()
    {
        return new BooleanLiteral(qs, mapping, isEmpty);
    }
}
