// (c) Microsoft Corporation 2005-2007. 
#light

namespace Microsoft.FSharp.Control

open Microsoft.FSharp.Core
open Microsoft.FSharp.Core.LanguagePrimitives
open Microsoft.FSharp.Core.LanguagePrimitives.IntrinsicOperators
open Microsoft.FSharp.Core.Operators
open Microsoft.FSharp.Collections
open Microsoft.FSharp.Control
open Microsoft.FSharp.Primitives.Basics

[<CompilationRepresentation(CompilationRepresentationFlags.ModuleSuffix)>]
module IEvent = 

    let create() = 
        let listeners = ref []
        (fun x -> List.iter (fun f -> f x) !listeners), 
        { new IEvent<'a> with Add(f) = lock listeners (fun () -> listeners := (!listeners)@[f]) }

    let create1() = 
        let listeners = ref None
        (fun x -> Option.iter (fun f -> f x) !listeners), 
        { new IEvent<'a> with Add(f) = lock listeners (fun () -> listeners := Some f) }

    let create_HandlerEvent<'sender,'a>() : ('sender * 'a -> unit) * IHandlerEvent<'a> = 
        let listeners = ref [] 
        let add d = lock listeners (fun () -> listeners := (!listeners)@[d]) 
        let remove d1 = lock listeners (fun () -> listeners := List.filter (fun d2 -> not (PhysicalEquality d1 d2)) !listeners) 
        let ev = { new IHandlerEvent<_> 
                     with Add(f) = add (new Handler<'a>(fun _ x -> f x))
                     and  AddHandler(h) = add h
                     and  RemoveHandler(h) = remove h }
        let fire (obj,x) =  (!listeners) |> List.iter (fun d -> ignore(d.Invoke(obj,x))) 
        fire, ev

    #if CLI_AT_LEAST_2_0
    [<Experimental("It is recommended to use 'create_HanlderEvent' when possible")>]
    let create_DelegateEvent<'handler when 'handler :> System.Delegate>() : (obj list -> unit) * IPrimitiveDelegateEvent<'handler> =
        let multicast = ref (None : System.Delegate option)
        let invoke (args) = 
            let args = (List.to_array args)
            match !multicast with 
            | None -> ()
            | Some d -> d.DynamicInvoke(args) |> ignore
        let get () = match !multicast with None -> null | Some d -> d
        let add d = multicast := Some(System.Delegate.Combine(get(), (d :> System.Delegate)))
        let remove d = multicast := Some(System.Delegate.Remove(get(), d))
        let delType = (type 'handler)
        let ev = { new IPrimitiveDelegateEvent<'handler> 
                     with AddHandler(h) = add h
                     and  RemoveHandler(h) = remove h }
        (invoke, ev)
    #endif 
    
    let create_public<'sender,'a>() = create_HandlerEvent<'sender,'a>()
    let createCompat() = 
        let fire,ev = create_public()
        (fun x -> fire ((null:obj), x)), ev


    let map f (w: #IEvent<'a>) =
        let outw,oute = create()
        w.Add(fun x -> outw(f x));
        oute

    let filter f (w: #IEvent<'a>) =
        let outw,oute = create()
        w.Add(fun x -> if f x then outw x);
        oute

    let partition f (w: #IEvent<'a>) =
        let outw1,oute1 = create()
        let outw2,oute2 = create()
        w.Add(fun x -> if f x then outw1 x else outw2 x);
        oute1,oute2

    let choose f (w: #IEvent<'a>) =
        let outw,oute = create()
        w.Add(fun x -> match f x with None -> () | Some r -> outw r);
        oute

    let fold f z (w: #IEvent<'a>) =
        let state = ref z
        let ow,oe = create() 
        w.Add(fun msg ->
             let z = !state
             let z = f z msg
             state := z; 
             ow(z));
        oe

    let fold_and_emit f  z (w: #IEvent<'a>) =
        let state = ref z
        let ow,oe = create() 
        w.Add(fun msg ->
             let z = !state
             let z,res = f z msg
             state := z; 
             ow(res));
        oe
    let listen f (w: #IEvent<'a>) = w.Add(f)

    let pairwise (inp : #IEvent<'a>) : IEvent<'a * 'a> = 
        let fire,ev = create()
        let lastArgs = ref None
        inp |> listen(fun e2 -> 
            (match !lastArgs with 
             | None -> () 
             | Some e1 -> fire(e1,e2));
            lastArgs := Some e2); 

        ev

    let merge (w1: #IEvent<'a>) (w2: #IEvent<'a>) =
        let wf,we = create() 
        w1.Add(fun x -> wf(x));
        w2.Add(fun x -> wf(x));
        we

    let split (f : 'a -> Choice<'b,'c>) (w: #IEvent<'a>) =
        let w1,e1 = create() 
        let w2,e2 = create() 
        w.Add(fun x -> match f x with Choice2_1 y -> w1(y) | Choice2_2 z -> w2(z));
        e1,e2

