// (c) Microsoft Corporation 2005-2007.

#light

namespace Microsoft.FSharp.Compatibility.OCaml

open Microsoft.FSharp.Core
open Microsoft.FSharp.Core.LanguagePrimitives.IntrinsicOperators
open Microsoft.FSharp.Core.Operators
open Microsoft.FSharp.Collections
#if CLI_AT_MOST_1_1
open Microsoft.FSharp.Compatibility
#else
open System.Collections.Generic
#endif

module Set = 


    // Functor
    type Provider<'a,'tag> when 'tag :> IComparer<'a> =
       { //type t = Tagged.Set<'a,'tag>
         empty    : Tagged.Set<'a,'tag>;
         is_empty : Tagged.Set<'a,'tag> -> bool;
         mem      : 'a -> Tagged.Set<'a,'tag> -> bool;
         add      : 'a -> Tagged.Set<'a,'tag> -> Tagged.Set<'a,'tag>;
         singleton: 'a -> Tagged.Set<'a,'tag>;
         remove   : 'a -> Tagged.Set<'a,'tag> -> Tagged.Set<'a,'tag>;
         union    : Tagged.Set<'a,'tag> -> Tagged.Set<'a,'tag> -> Tagged.Set<'a,'tag>;
         inter    : Tagged.Set<'a,'tag> -> Tagged.Set<'a,'tag> -> Tagged.Set<'a,'tag>;
         diff     : Tagged.Set<'a,'tag> -> Tagged.Set<'a,'tag> -> Tagged.Set<'a,'tag>;
         iter     : ('a -> unit) -> Tagged.Set<'a,'tag> -> unit;
         elements : Tagged.Set<'a,'tag> -> 'a list;
         equal    : Tagged.Set<'a,'tag> -> Tagged.Set<'a,'tag> -> bool;
         subset   : Tagged.Set<'a,'tag> -> Tagged.Set<'a,'tag> -> bool;
         compare  : Tagged.Set<'a,'tag> -> Tagged.Set<'a,'tag> -> int;
         for_all  : ('a -> bool) -> Tagged.Set<'a,'tag> -> bool;
         exists   : ('a -> bool) -> Tagged.Set<'a,'tag> -> bool;
         filter   : ('a -> bool) -> Tagged.Set<'a,'tag> -> Tagged.Set<'a,'tag>;
         partition: ('a -> bool) -> Tagged.Set<'a,'tag> -> Tagged.Set<'a,'tag> * Tagged.Set<'a,'tag>;
         fold     : 'b. ('a -> 'b -> 'b) -> Tagged.Set<'a,'tag> -> 'b -> 'b;
         cardinal : Tagged.Set<'a,'tag> -> int;
         min_elt  : Tagged.Set<'a,'tag> -> 'a;
         max_elt  : Tagged.Set<'a,'tag> -> 'a;
         choose   : Tagged.Set<'a,'tag> -> 'a }

    let gen_is_empty (s : Tagged.Set<_,_>) = s.IsEmpty
    let gen_mem x (s : Tagged.Set<_,_>) = s.Contains(x)
    let gen_add x (s : Tagged.Set<_,_>) = s.Add(x)
    let gen_remove x (s : Tagged.Set<_,_>) = s.Remove(x)
    let gen_iter f (s : Tagged.Set<_,_>)  = s.Iterate(f)
    let gen_elements (s : Tagged.Set<_,_>) = s.ToList()
    let gen_subset (s1 : Tagged.Set<_,_>)  (s2 : Tagged.Set<_,_>)  = s1.IsSubsetOf(s2)
    let gen_for_all f (s : Tagged.Set<_,_>) = s.ForAll f
    let gen_exists f (s : Tagged.Set<_,_>) = s.Exists f
    let gen_filter f (s : Tagged.Set<_,_>) = s.Filter f
    let gen_partition f (s : Tagged.Set<_,_>) = s.Partition f 
    let gen_fold f (s : Tagged.Set<_,_>) acc = s.Fold f acc
    let gen_cardinal (s : Tagged.Set<_,_>) = s.Count
    let gen_size (s : Tagged.Set<_,_>) = s.Count
    let gen_min_elt (s : Tagged.Set<_,_>) = s.MinimumElement
    let gen_max_elt (s : Tagged.Set<_,_>) = s.MaximumElement
    let gen_choose (s : Tagged.Set<_,_>) = s.Choose

    let MakeTagged (cf :  ('tag :> IComparer<'a>)) : Provider<'a,'tag> = 
       { empty = Tagged.Set<'a,'tag>.Empty(cf);
         is_empty = gen_is_empty;
         mem = gen_mem;
         add = gen_add;
         singleton = (fun x -> Tagged.Set<'a,'tag>.Singleton(cf,x));
         remove = gen_remove;
         union = (fun s1 s2 -> Tagged.Set<'a,'tag>.Union(s1,s2));
         inter = (fun s1 s2 -> Tagged.Set<'a,'tag>.Intersection(s1,s2));
         diff = (fun s1 s2 -> Tagged.Set<'a,'tag>.Difference(s1,s2));
         iter = gen_iter;
         elements = gen_elements;
         equal = (fun s1 s2 -> Tagged.Set<'a,'tag>.Equality(s1,s2));
         subset = gen_subset;
         compare = (fun s1 s2 -> Tagged.Set<'a,'tag>.Compare(s1,s2));
         for_all = gen_for_all;
         fold= gen_fold;
         exists = gen_exists;
         filter = gen_filter;
         partition = gen_partition;
         cardinal = gen_cardinal;
         min_elt = gen_min_elt;
         max_elt = gen_max_elt;
         choose = gen_choose; }

    type Provider<'a> = Provider<'a, IComparer<'a>>
    let Make cf  = MakeTagged (ComparisonIdentity.FromFunction cf)

    type CSet<'a> = Tagged.Set<'a>
    type CSetOps<'a> = Provider<'a>

