// Copyright (c) Microsoft Corporation 2005-2007.
// This sample code is provided "as is" without warranty of any kind. 
// We disclaim all warranties, either express or implied, including the 
// warranties of merchantability and fitness for a particular purpose. 
//

#light

open System
open System.Collections
open System.Collections.Generic
open System.IO
open System.Windows.Forms
open System.Xml.Linq

open Microsoft.FSharp.Linq
open Microsoft.FSharp.Linq.SequenceOps
open Microsoft.FSharp.Xml.Linq
open Microsoft.FSharp.Xml.Linq.SequenceOps
open Sample.Support

//type decimal = System.Decimal
type Customer = 
    { CustomerID:string;
      CompanyName:string;
      Address:string;
      City:string;
      Region:string;
      PostalCode:string;
      Country:string;
      Phone:string;
      Fax:string;
      Orders: Order[]; }

and Order =
    { OrderID:int;
      OrderDate:DateTime;
      Total:Decimal; }

and Product =
    { ProductID:int; 
      ProductName:string;
      Category:string;
      UnitPrice:Decimal;
      UnitsInStock:int; }

let dataPath = Path.Combine(Environment.GetFolderPath(Environment.SpecialFolder.ProgramFiles), @"LINQ Preview\Data\") 

let products =
     [ { ProductID = 1; ProductName = "Chai"; Category = "Beverages"; UnitPrice = Convert.ToDecimal(18.0000); UnitsInStock = 39 };
       { ProductID = 2; ProductName = "Chang"; Category = "Beverages"; UnitPrice = Convert.ToDecimal(19.0000); UnitsInStock = 17 };
       { ProductID = 3; ProductName = "Aniseed Syrup"; Category = "Condiments"; UnitPrice = Convert.ToDecimal(10.0000); UnitsInStock = 13 };
       { ProductID = 4; ProductName = "Chef Anton's Cajun Seasoning"; Category = "Condiments"; UnitPrice = Convert.ToDecimal(22.0000); UnitsInStock = 53 };
       { ProductID = 5; ProductName = "Chef Anton's Gumbo Mix"; Category = "Condiments"; UnitPrice = Convert.ToDecimal(21.3500); UnitsInStock = 0 };
       { ProductID = 6; ProductName = "Grandma's Boysenberry Spread"; Category = "Condiments"; UnitPrice = Convert.ToDecimal(25.0000); UnitsInStock = 120 };
       { ProductID = 7; ProductName = "Uncle Bob's Organic Dried Pears"; Category = "Produce"; UnitPrice = Convert.ToDecimal(30.0000); UnitsInStock = 15 };
       { ProductID = 8; ProductName = "Northwoods Cranberry Sauce"; Category = "Condiments"; UnitPrice = Convert.ToDecimal(40.0000); UnitsInStock = 6 };
       { ProductID = 9; ProductName = "Mishi Kobe Niku"; Category = "Meat/Poultry"; UnitPrice = Convert.ToDecimal(97.0000); UnitsInStock = 29 };
       { ProductID = 10; ProductName = "Ikura"; Category = "Seafood"; UnitPrice = Convert.ToDecimal(31.0000); UnitsInStock = 31 };
       { ProductID = 11; ProductName = "Queso Cabrales"; Category = "Dairy Products"; UnitPrice = Convert.ToDecimal(21.0000); UnitsInStock = 22 };
       { ProductID = 12; ProductName = "Queso Manchego La Pastora"; Category = "Dairy Products"; UnitPrice = Convert.ToDecimal(38.0000); UnitsInStock = 86 };
       { ProductID = 13; ProductName = "Konbu"; Category = "Seafood"; UnitPrice = Convert.ToDecimal(6.0000); UnitsInStock = 24 };
       { ProductID = 14; ProductName = "Tofu"; Category = "Produce"; UnitPrice = Convert.ToDecimal(23.2500); UnitsInStock = 35 };
       { ProductID = 15; ProductName = "Genen Shouyu"; Category = "Condiments"; UnitPrice = Convert.ToDecimal(15.5000); UnitsInStock = 39 };
       { ProductID = 16; ProductName = "Pavlova"; Category = "Confections"; UnitPrice = Convert.ToDecimal(17.4500); UnitsInStock = 29 };
       { ProductID = 17; ProductName = "Alice Mutton"; Category = "Meat/Poultry"; UnitPrice = Convert.ToDecimal(39.0000); UnitsInStock = 0 };
       { ProductID = 18; ProductName = "Carnarvon Tigers"; Category = "Seafood"; UnitPrice = Convert.ToDecimal(62.5000); UnitsInStock = 42 };
       { ProductID = 19; ProductName = "Teatime Chocolate Biscuits"; Category = "Confections"; UnitPrice = Convert.ToDecimal(9.2000); UnitsInStock = 25 };
       { ProductID = 20; ProductName = "Sir Rodney's Marmalade"; Category = "Confections"; UnitPrice = Convert.ToDecimal(81.0000); UnitsInStock = 40 };
       { ProductID = 21; ProductName = "Sir Rodney's Scones"; Category = "Confections"; UnitPrice = Convert.ToDecimal(10.0000); UnitsInStock = 3 };
       { ProductID = 22; ProductName = "Gustaf's Knckebrd"; Category = "Grains/Cereals"; UnitPrice = Convert.ToDecimal(21.0000); UnitsInStock = 104 };
       { ProductID = 23; ProductName = "Tunnbrd"; Category = "Grains/Cereals"; UnitPrice = Convert.ToDecimal(9.0000); UnitsInStock = 61 };
       { ProductID = 24; ProductName = "Guaran Fantstica"; Category = "Beverages"; UnitPrice = Convert.ToDecimal(4.5000); UnitsInStock = 20 };
       { ProductID = 25; ProductName = "NuNuCa Nu-Nougat-Creme"; Category = "Confections"; UnitPrice = Convert.ToDecimal(14.0000); UnitsInStock = 76 };
       { ProductID = 26; ProductName = "Gumbr Gummibrchen"; Category = "Confections"; UnitPrice = Convert.ToDecimal(31.2300); UnitsInStock = 15 };
       { ProductID = 27; ProductName = "Schoggi Schokolade"; Category = "Confections"; UnitPrice = Convert.ToDecimal(43.9000); UnitsInStock = 49 };
       { ProductID = 28; ProductName = "Rssle Sauerkraut"; Category = "Produce"; UnitPrice = Convert.ToDecimal(45.6000); UnitsInStock = 26 };
       { ProductID = 29; ProductName = "Thringer Rostbratwurst"; Category = "Meat/Poultry"; UnitPrice = Convert.ToDecimal(123.7900); UnitsInStock = 0 };
       { ProductID = 30; ProductName = "Nord-Ost Matjeshering"; Category = "Seafood"; UnitPrice = Convert.ToDecimal(25.8900); UnitsInStock = 10 };
       { ProductID = 31; ProductName = "Gorgonzola Telino"; Category = "Dairy Products"; UnitPrice = Convert.ToDecimal(12.5000); UnitsInStock = 0 };
       { ProductID = 32; ProductName = "Mascarpone Fabioli"; Category = "Dairy Products"; UnitPrice = Convert.ToDecimal(32.0000); UnitsInStock = 9 };
       { ProductID = 33; ProductName = "Geitost"; Category = "Dairy Products"; UnitPrice = Convert.ToDecimal(2.5000); UnitsInStock = 112 };
       { ProductID = 34; ProductName = "Sasquatch Ale"; Category = "Beverages"; UnitPrice = Convert.ToDecimal(14.0000); UnitsInStock = 111 };
       { ProductID = 35; ProductName = "Steeleye Stout"; Category = "Beverages"; UnitPrice = Convert.ToDecimal(18.0000); UnitsInStock = 20 };
       { ProductID = 36; ProductName = "Inlagd Sill"; Category = "Seafood"; UnitPrice = Convert.ToDecimal(19.0000); UnitsInStock = 112 };
       { ProductID = 37; ProductName = "Gravad lax"; Category = "Seafood"; UnitPrice = Convert.ToDecimal(26.0000); UnitsInStock = 11 };
       { ProductID = 38; ProductName = "Cte de Blaye"; Category = "Beverages"; UnitPrice = Convert.ToDecimal(263.5000); UnitsInStock = 17 };
       { ProductID = 39; ProductName = "Chartreuse verte"; Category = "Beverages"; UnitPrice = Convert.ToDecimal(18.0000); UnitsInStock = 69 };
       { ProductID = 40; ProductName = "Boston Crab Meat"; Category = "Seafood"; UnitPrice = Convert.ToDecimal(18.4000); UnitsInStock = 123 };
       { ProductID = 41; ProductName = "Jack's New England Clam Chowder"; Category = "Seafood"; UnitPrice = Convert.ToDecimal(9.6500); UnitsInStock = 85 };
       { ProductID = 42; ProductName = "Singaporean Hokkien Fried Mee"; Category = "Grains/Cereals"; UnitPrice = Convert.ToDecimal(14.0000); UnitsInStock = 26 };
       { ProductID = 43; ProductName = "Ipoh Coffee"; Category = "Beverages"; UnitPrice = Convert.ToDecimal(46.0000); UnitsInStock = 17 };
       { ProductID = 44; ProductName = "Gula Malacca"; Category = "Condiments"; UnitPrice = Convert.ToDecimal(19.4500); UnitsInStock = 27 };
       { ProductID = 45; ProductName = "Rogede sild"; Category = "Seafood"; UnitPrice = Convert.ToDecimal(9.5000); UnitsInStock = 5 };
       { ProductID = 46; ProductName = "Spegesild"; Category = "Seafood"; UnitPrice = Convert.ToDecimal(12.0000); UnitsInStock = 95 };
       { ProductID = 47; ProductName = "Zaanse koeken"; Category = "Confections"; UnitPrice = Convert.ToDecimal(9.5000); UnitsInStock = 36 };
       { ProductID = 48; ProductName = "Chocolade"; Category = "Confections"; UnitPrice = Convert.ToDecimal(12.7500); UnitsInStock = 15 };
       { ProductID = 49; ProductName = "Maxilaku"; Category = "Confections"; UnitPrice = Convert.ToDecimal(20.0000); UnitsInStock = 10 };
       { ProductID = 50; ProductName = "Valkoinen suklaa"; Category = "Confections"; UnitPrice = Convert.ToDecimal(16.2500); UnitsInStock = 65 };
       { ProductID = 51; ProductName = "Manjimup Dried Apples"; Category = "Produce"; UnitPrice = Convert.ToDecimal(53.0000); UnitsInStock = 20 };
       { ProductID = 52; ProductName = "Filo Mix"; Category = "Grains/Cereals"; UnitPrice = Convert.ToDecimal(7.0000); UnitsInStock = 38 };
       { ProductID = 53; ProductName = "Perth Pasties"; Category = "Meat/Poultry"; UnitPrice = Convert.ToDecimal(32.8000); UnitsInStock = 0 };
       { ProductID = 54; ProductName = "Tourtire"; Category = "Meat/Poultry"; UnitPrice = Convert.ToDecimal(7.4500); UnitsInStock = 21 };
       { ProductID = 55; ProductName = "Pt chinois"; Category = "Meat/Poultry"; UnitPrice = Convert.ToDecimal(24.0000); UnitsInStock = 115 };
       { ProductID = 56; ProductName = "Gnocchi di nonna Alice"; Category = "Grains/Cereals"; UnitPrice = Convert.ToDecimal(38.0000); UnitsInStock = 21 };
       { ProductID = 57; ProductName = "Ravioli Angelo"; Category = "Grains/Cereals"; UnitPrice = Convert.ToDecimal(19.5000); UnitsInStock = 36 };
       { ProductID = 58; ProductName = "Escargots de Bourgogne"; Category = "Seafood"; UnitPrice = Convert.ToDecimal(13.2500); UnitsInStock = 62 };
       { ProductID = 59; ProductName = "Raclette Courdavault"; Category = "Dairy Products"; UnitPrice = Convert.ToDecimal(55.0000); UnitsInStock = 79 };
       { ProductID = 60; ProductName = "Camembert Pierrot"; Category = "Dairy Products"; UnitPrice = Convert.ToDecimal(34.0000); UnitsInStock = 19 };
       { ProductID = 61; ProductName = "Sirop d'rable"; Category = "Condiments"; UnitPrice = Convert.ToDecimal(28.5000); UnitsInStock = 113 };
       { ProductID = 62; ProductName = "Tarte au sucre"; Category = "Confections"; UnitPrice = Convert.ToDecimal(49.3000); UnitsInStock = 17 };
       { ProductID = 63; ProductName = "Vegie-spread"; Category = "Condiments"; UnitPrice = Convert.ToDecimal(43.9000); UnitsInStock = 24 };
       { ProductID = 64; ProductName = "Wimmers gute Semmelkndel"; Category = "Grains/Cereals"; UnitPrice = Convert.ToDecimal(33.2500); UnitsInStock = 22 };
       { ProductID = 65; ProductName = "Louisiana Fiery Hot Pepper Sauce"; Category = "Condiments"; UnitPrice = Convert.ToDecimal(21.0500); UnitsInStock = 76 };
       { ProductID = 66; ProductName = "Louisiana Hot Spiced Okra"; Category = "Condiments"; UnitPrice = Convert.ToDecimal(17.0000); UnitsInStock = 4 };
       { ProductID = 67; ProductName = "Laughing Lumberjack Lager"; Category = "Beverages"; UnitPrice = Convert.ToDecimal(14.0000); UnitsInStock = 52 };
       { ProductID = 68; ProductName = "Scottish Longbreads"; Category = "Confections"; UnitPrice = Convert.ToDecimal(12.5000); UnitsInStock = 6 };
       { ProductID = 69; ProductName = "Gudbrandsdalsost"; Category = "Dairy Products"; UnitPrice = Convert.ToDecimal(36.0000); UnitsInStock = 26 };
       { ProductID = 70; ProductName = "Outback Lager"; Category = "Beverages"; UnitPrice = Convert.ToDecimal(15.0000); UnitsInStock = 15 };
       { ProductID = 71; ProductName = "Flotemysost"; Category = "Dairy Products"; UnitPrice = Convert.ToDecimal(21.5000); UnitsInStock = 26 };
       { ProductID = 72; ProductName = "Mozzarella di Giovanni"; Category = "Dairy Products"; UnitPrice = Convert.ToDecimal(34.8000); UnitsInStock = 14 };
       { ProductID = 73; ProductName = "Rd Kaviar"; Category = "Seafood"; UnitPrice = Convert.ToDecimal(15.0000); UnitsInStock = 101 };
       { ProductID = 74; ProductName = "Longlife Tofu"; Category = "Produce"; UnitPrice = Convert.ToDecimal(10.0000); UnitsInStock = 4 };
       { ProductID = 75; ProductName = "Rhnbru Klosterbier"; Category = "Beverages"; UnitPrice = Convert.ToDecimal(7.7500); UnitsInStock = 125 };
       { ProductID = 76; ProductName = "Lakkalikri"; Category = "Beverages"; UnitPrice = Convert.ToDecimal(18.0000); UnitsInStock = 57 };
       { ProductID = 77; ProductName = "Original Frankfurter grne Soe"; Category = "Condiments"; UnitPrice = Convert.ToDecimal(13.0000); UnitsInStock = 32 }
     ]
        
    
            // Customer/order data read into memory from XML file open XLinq:
let customerListPath = Path.Combine(__SOURCE_DIRECTORY__, @"customers.xml")
let customers =
    (XDocument.Load(customerListPath)).Root.Elements(xname "customer")
    |> select (fun e -> 
         { CustomerID  = XElement.op_Explicit (e.Element(xname "id"));
           CompanyName = XElement.op_Explicit (e.Element(xname "name"));
           Address     = XElement.op_Explicit (e.Element(xname "address"));
           City        = XElement.op_Explicit (e.Element(xname "city"));
           Region      = XElement.op_Explicit (e.Element(xname "region"));
           PostalCode  = XElement.op_Explicit (e.Element(xname "postalcode"));
           Country     = XElement.op_Explicit (e.Element(xname "country"));
           Phone       = XElement.op_Explicit (e.Element(xname "phone"));
           Fax         = XElement.op_Explicit (e.Element(xname "fax"));
           Orders = 
             e.Elements(xname "orders")
             |> elements(xname "order")
             |> select (fun o -> 
                 { OrderID   = XElement.op_Explicit (o.Element(xname "id"));
                   OrderDate = XElement.op_Explicit (o.Element(xname "orderdate"));
                   Total     = XElement.op_Explicit (o.Element(xname "total")) }) 
             |> to_Array })

[<Category("Restriction Operators");
  Title("Where - Simple 1");
  Description("This sample uses where to find all elements of an array less than 5.")>]
let Linq1() =
    let numbers = [ 5; 4; 1; 3; 9; 8; 6; 7; 2; 0 ] |> List.to_seq

    let lowNums =
        numbers
        |> where (fun n -> n < 5)

    Console.WriteLine("Numbers < 5: {0}", any_to_string (to_list lowNums))


[<Category("Restriction Operators");
  Title("Where - Simple 2");
  Description("This sample uses where to find all products that are out of stock.")>]
let Linq2() = 
  
    let soldOutProducts =
         products
         |> where (fun p -> p.UnitsInStock = 0) 
              
    Console.WriteLine("Sold out products: {0}", any_to_string (to_list soldOutProducts))
        
[<Category("Restriction Operators");
  Title("Where - Simple 3");
  Description("This sample uses where to find all products that are stock and \
              cost more than 3.00 per unit.")>]
let Linq3() = 

    let expensiveInStockProducts =
         products
         |> where (fun p -> p.UnitsInStock > 0 && p.UnitPrice > Convert.ToDecimal(3.00)) 
         |> select (fun p -> p.ProductName)
            
    Console.WriteLine("In-stock products that cost more than 3.00: {0}", any_to_string (to_list expensiveInStockProducts))
        
[<Category("Restriction Operators");
  Title("Where - Drilldown");
  Description("This sample uses where to find all customers Washington \
              and then uses the resulting sequence to drill down into their \
              orders.")>]

let Linq4() = 

    let waCustomers =
        customers 
        |> where (fun c -> match c.Region with | null -> false | s -> s = "WA")
              
    let waCustomersAndOrders = 
        waCustomers 
        |> select (fun c -> c.CustomerID, c.CompanyName, c.Orders)

    Console.WriteLine("Customers from Washington and their orders: {0}", any_to_string (to_list waCustomersAndOrders))
  
       
[<Category("Restriction Operators");
  Title("Where - Indexed");
  Description("This sample demonstrates an indexed Where clause that returns digits whose name is \
              shorter than their value.")>]
let Linq5() = 
    let digits = [| "zero"; "one"; "two"; "three"; "four"; "five"; "six"; "seven"; "eight"; "nine" |]

    let shortDigits = 
        digits
        |> wherei (fun digit index -> digit.Length < index)
          
    Console.WriteLine("Short digits: {0}", any_to_string (to_list shortDigits))



[<Category("Projection Operators");
  Title("Select - Simple 1");
  Description("This sample uses select to produce a sequence of ints one higher than \
              those an existing array of ints.")>]
let Linq6() = 
    let numbers = [ 5; 4; 1; 3; 9; 8; 6; 7; 2; 0 ] |> List.to_seq

    let numsPlusOne =
        numbers 
        |> select (fun n -> n + 1)
              
    Console.WriteLine("Numbers + 1: {0}", any_to_string (to_list numsPlusOne))


        
[<Category("Projection Operators");
  Title("Select - Simple 2");
  Description("This sample uses select to return a sequence of just the names of a list of products.")>]
let Linq7() = 

  let productNames =
     products
     |>  select (fun p -> p.ProductName) 
            
  Console.WriteLine("Product Names: {0}", any_to_string (to_list productNames))



[<Category("Projection Operators");
  Title("Select - Transformation");
  Description("This sample uses select to produce a sequence of strings representing \ 
              the text version of a sequence of ints.")>]

let Linq8() = 
  let numbers = [ 5; 4; 1; 3; 9; 8; 6; 7; 2; 0 ] |> List.to_seq
  let strings = [| "zero"; "one"; "two"; "three"; "four"; "five"; "six"; "seven"; "eight"; "nine" |]

  let textNums = 
    numbers 
    |> select (fun n -> strings.(n))
            
  Console.WriteLine("Number strings: {0}", any_to_string (to_list textNums))


[<Category("Projection Operators");
  Title("Select - Anonymous Types 1");
  Description("This sample uses select to produce a sequence of the uppercase \
               and lowercase versions of each word the original array.")>]
let Linq9() = 
  let words = [| "aPPLE"; "BlUeBeRrY"; "cHeRry" |]

  let upperLowerWords =
      words
      |> select (fun w -> w.ToUpper(), w.ToLower())

  upperLowerWords |> iter (fun (u,l) -> 
    Console.WriteLine("Uppercase: {0}, Lowercase: {1}", u, l)
  )

[<Category("Projection Operators");
  Title("Select - Indexed");
  Description("This sample uses an indexed Select clause to determine if the value of ints \
               an array match their position the array.")>]

let Linq12() = 
    let numbers = [ 5; 4; 1; 3; 9; 8; 6; 7; 2; 0 ] |> List.to_seq

    let numsInPlace = 
        numbers 
        |> selecti (fun num index -> (num, (num = index)))
          
    Console.WriteLine("Number: In-place?");
    numsInPlace |> iter (fun (n,p) -> Console.WriteLine("{0}: {1}",n,p))


[<Category("Projection Operators");
  Title("SelectMany - Compound from 2");
  Description("This sample uses a compound from clause to select all orders where the \
               order total is less than 500.00.")>]

let Linq15() = 
    let infos = 
        customers
        |> selectMany (fun c -> 
            c.Orders
            |> where (fun o -> o.Total < Convert.ToDecimal(500.00))
            |> select(fun o -> (c.CustomerID, o.OrderID, o.Total)))
    Console.WriteLine("orders =  {0}", any_to_string (to_list infos))


[<Category("Ordering Operators");
  Title("OrderBy - Simple 1");
  Description("This sample uses orderBy to sort a list of words alphabetically.")>]
let Linq28() = 
    let words = [| "cherry"; "apple"; "blueberry" |]
    let sortedWords =
        words
        |> orderBy it 
    Console.WriteLine("The sorted list of words: {0}", any_to_string (to_list sortedWords))


let LinqSamples = 
  "101 LINQ Query Samples",
  "Linq",
  Linq1
