
(* (c) Microsoft Corporation. All rights reserved *)
(*----------------------------------------------------------------------------
 * Loading initial context, reporting errors etc.
 *--------------------------------------------------------------------------*)

(*F# 
open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 
open Microsoft.FSharp.Compiler 
module Il = Microsoft.Research.AbstractIL.IL 
F#*) 

open Range
open Ast
open Tast
open Tastops
open Lib
open Infos
open Tc
open Env

(*----------------------------------------------------------------------------
 * File suffixes
 *--------------------------------------------------------------------------*)

val sigSuffixes : string list
val implSuffixes : string list

(*----------------------------------------------------------------------------
 * Parsing inputs
 *--------------------------------------------------------------------------*)
  
val qname_of_unique_path : range -> string list -> Ast.qualifiedNameOfFile

val prepend_path_to_input : Ast.ident list -> Ast.input -> Ast.input

val parseInput : (Lexing.lexbuf -> Pars.token) -> Lexing.lexbuf -> string option -> string -> Ast.input

(*----------------------------------------------------------------------------
 * Errors
 *--------------------------------------------------------------------------*)

val mark_of_err : exn -> range option
val errorNumber : exn -> int
val splitRelatedErrors : exn -> exn * exn list
val output_exn : Buffer.t -> exn -> unit
val output_err : bool -> Buffer.t -> exn -> unit
val output_err_context : string -> (string -> int -> string) -> Buffer.t -> exn -> unit


(*----------------------------------------------------------------------------
 * Options and configuration
 *--------------------------------------------------------------------------*)

type warningOptions 
val newWarningOptions : unit -> warningOptions

type tcConfig =
  {mutable mscorlib_assembly_name: string;
   mutable sscli: bool;
   mutable autoResolveOpenDirectivesToDlls: bool;
   mutable noFeedback: bool;
   mutable implicitIncludeDir: string;
   mutable openBinariesInMemory: bool;
   mutable openDebugInformationForLaterStaticLinking: bool;
   mutable fsharpBinariesDir: string;
   mutable compilingFslib: bool;
   mutable compilingMllib: bool;
   mutable includes: string list;
   mutable implicitOpens: string list;
   mutable useMLLib: bool;
   mutable useFsiAuxLib: bool;
   mutable framework: bool;
   mutable light: bool;
   mutable ifdefs: string list;
   mutable fullGenericsSupported: bool;
   mutable retargetable: string list;
   mutable desiredCLILibraryVersionOpt: Il.version_info option;
   mutable defaultNamespace: string option;
   mutable referencedDLLs: (range * string) list;
   mutable copyLocalFiles: (range * string) list;
   mutable clrRoot: string option;
   mutable manager: Il.manager option;
   mutable optimizeForMemory: bool;
   mutable inputCodePage: int option;
   warningOptions: warningOptions;}

val newTcConfig : string * warningOptions * bool * string -> tcConfig

val clrRootInclude : tcConfig -> string list

val compilerProcessRunsMono10 : unit -> bool
val latestCLRVersionOnMachine_or_PretendNET20ForMono10_or_CLRVersionAlreadyLoaded : unit -> Il.version_info

val configureIlxDefaultsForCLRVersion : Il.version_info -> unit

val turnWarnOff : tcConfig -> range -> string -> unit
val requireIncludePath : tcConfig -> range -> string -> unit
val requireDLLReference : tcConfig -> range -> string -> unit

(*----------------------------------------------------------------------------
 * Tables of referenced DLLs 
 *--------------------------------------------------------------------------*)

type dllinfo = {dllinfo_filename: string;
                dllinfo_isFsharp: bool;
                dllinfo_il_modul: Il.modul;
                dllinfo_il_scoref: Il.scope_ref;}

type ccuinfo = {ccuinfo_il_scoref: Il.scope_ref;
                ccuinfo_ccu: Tast.ccu;
                ccuinfo_optdata: Opt.modul_info option Lazy.t;}

type tcImports 

val dllinfosOfTcImports : tcImports -> dllinfo list
val ccuinfosOfTcImports : tcImports -> ccuinfo list
val disposeTcImports : tcImports -> unit
val ccusOfTcImportsInDeclOrder : tcImports -> Tast.ccu list

(*----------------------------------------------------------------------------
 * Special resources in DLLs
 *--------------------------------------------------------------------------*)

val is_IntfDataResource : Il.resource -> bool
val is_OptDataResource : Il.resource -> bool
val is_PickledDefinitionsResource : Il.resource -> bool

val write_IntfData : tcConfig -> tcGlobals -> Tastops.expr_remap -> ccu -> Il.resource

val write_OptData :  Tast.ccu * Opt.modul_info -> Il.resource

(*----------------------------------------------------------------------------
 * Finding files
 *--------------------------------------------------------------------------*)

val makeAbsolute : tcConfig -> string -> string
val fileExists : string -> bool
exception FileNameNotResolved of exn
val tryResolveLibFile : tcConfig -> range -> string -> string
val resolveLibFile : tcConfig -> range -> string -> string
val resolveSourceFile : tcConfig -> range -> string -> string
val hasSuffixCaseInsensitive : string -> string -> bool
val isDll : string -> bool


(*----------------------------------------------------------------------------
 * Finding files
 *--------------------------------------------------------------------------*)


val name_of_scoref : Il.scope_ref -> string
val name_of_module : Il.modul -> Il.assembly_name

val fslib : unit -> string
val mllib : unit -> string
val fsiaux : unit -> string
val fsiAuxSettingsModulePath : string

(*----------------------------------------------------------------------------
 * Finding and requiring DLLs
 *--------------------------------------------------------------------------*)

val findDllInfo : tcImports -> range -> string -> dllinfo
val findCcu : tcConfig * tcImports * tcGlobals -> range -> string -> Tast.ccu

val requireDLL : tcConfig -> tcImports -> tcGlobals -> tcEnv -> range -> string -> tcEnv * (dllinfo list * ccuinfo list)

val autoModuleResolver : tcConfig -> tcGlobals -> tcImports -> Ast.ident -> Tast.modul_ref option


(*----------------------------------------------------------------------------
 * Processing # commands
 *--------------------------------------------------------------------------*)

val processMetaCommandsFromInput : ('a -> range -> string -> 'a) * ('a -> range -> string -> unit) -> tcConfig -> Ast.input -> 'a -> 'a
val processMetaCommandsFromInputAsCommandLineFlags : tcConfig -> Ast.input -> unit

(*----------------------------------------------------------------------------
 * Loading the default library sets
 *--------------------------------------------------------------------------*)
                
val core_framework : string list
val extended_framework : string list
val all_framework : string list

val buildFoundationalTcImports : tcConfig -> Tast.ccu * tcImports
val buildFrameworkTcImports :  tcConfig -> tcImports -> ccu -> tcGlobals * tcImports
val buildReferencedDllTcImports : tcConfig -> tcGlobals -> tcImports -> tcImports
val buildTcImports : tcConfig -> tcGlobals * tcImports

(*----------------------------------------------------------------------------
 * Type checking and querying the type checking state
 *--------------------------------------------------------------------------*)

val implicitOpen : autoModuleResolver -> range -> tcEnv -> string -> tcEnv
val initTcEnv : range -> tcConfig -> tcImports -> tcGlobals -> tcEnv

                
type topRootedSigs = (qualifiedNameOfFile, Tast.modul_typ) Zmap.t
type tcSigsAndImpls

type tcState = {tcsCcu: Tast.ccu;
                tcsCcuType: Tast.modul_spec;
                tcsNiceNameGen: Ast.niceNameGenerator;
                tcsTcSigEnv: tcEnv;
                tcsTcImplEnv: tcEnv;
                tcsTopSigsAndImpls: tcSigsAndImpls;}

val typecheckInitialState : range -> string -> tcConfig -> tcGlobals -> Ast.niceNameGenerator -> tcEnv -> tcState
val tcEnvOfTcState : tcState -> tcEnv
val nngOfTcState : tcState -> Ast.niceNameGenerator
val topRootedSigsOfTcState : tcState -> topRootedSigs

val typecheckMultipleInputs :
  (unit -> bool) -> tcConfig -> tcImports -> tcGlobals ->
  Ast.long_ident option -> tcState -> Ast.input list ->
  (tcEnv * topAttribs * Tast.typedImplFile list) * tcState
  
val typecheckClosedInputSet :
  (unit -> bool) -> tcConfig -> tcImports -> tcGlobals -> 
  Ast.long_ident option -> tcState -> Ast.input list ->
  tcState * topAttribs * Tast.typedAssembly * tcEnv


(*----------------------------------------------------------------------------
 * Type checking and querying the type checking state
 *--------------------------------------------------------------------------*)
  
val parseArgs : (string -> unit) -> (string * Arg.spec * string) list -> string list -> unit
val reportWarning : warningOptions -> exn -> bool
val reportWarningAsError : warningOptions -> exn -> bool
val frontEndArgSpecs : tcConfig -> (string * Arg.spec * string) list * (string * Arg.spec * string) list
