// (c) Microsoft Corporation 2005-2007. 
#light

namespace Microsoft.FSharp.Collections

open Microsoft.FSharp.Core
open Microsoft.FSharp.Collections
open System.Collections
open System

#if CLI_AT_MOST_1_1
open Microsoft.FSharp.Compatibility
#else
open System.Collections.Generic
#endif
type seq<'a> = IEnumerable<'a>   

/// Basic operations on IEnumerables.  
module Seq = 

    /// Create an empty IEnumerable 
    [<GeneralizableValueAttribute>]
    val empty<'a> : seq<'a>

    /// Return the length of the IEnumerable
    val length: #seq<'a> -> int

    /// Return an IEnumerable that yields one item only.
    val singleton: 'a -> seq<'a>

    /// Return an IEnumerable that when iterated yields
    /// the given item followed by the items in the given sequence
    val cons: 'a -> #seq<'a> -> seq<'a>

    /// Return an IEnumerable that when iterated yields
    /// the items in the given sequence followed by the given item
    val tail_cons: #seq<'a> -> 'a -> seq<'a>

    /// Return the first element of the IEnumerable.  Raise (Invalid_argument "hd") if undefined.
    val hd: #seq<'a> -> 'a

    /// Return the first N elements of the IEnumerable.  Raise (Invalid_argument "take") if undefined.
    val take: int -> #seq<'a> -> 'a list

    /// Return an IEnumerable that when enumerated returns at most N elements.
    val truncate: int -> #seq<'a> -> seq<'a>

    /// Return true if the IEnumerable is not empty.
    val nonempty: #seq<'a> -> bool

    /// Apply a function to each element of the collection, threading an 'accumulator' argument
    /// through the computation. If the elements are "i0...iN" then computes "f (... (f s i0)...) iN"
    val fold: ('b -> 'a -> 'b) -> 'b -> #seq<'a> -> 'b

    /// Apply a function to each element of the sequence, threading an 'accumulator' argument
    /// through the computation. Begin by applying the function to the first two elements.
    /// Then feed this result into the function along with the third element and so on.  
    /// Return the final result.
    /// Raises [[InvalidArgumentException]] if the sequence has no elements.
    val fold1: ('a -> 'a -> 'a) -> #seq<'a> -> 'a

    /// Like fold, but compute on-demand and return the sequence of intermediary and final results
    val scan : ('b -> 'a -> 'b) -> 'b -> #seq<'a> -> seq<'b>

    /// Like fold1, but compute on-demand and return the sequence of intermediary and final results
    val scan1 : ('a -> 'a -> 'a) -> #seq<'a> -> seq<'a>


    /// Return an IEnumerable that contains the elements generated by the given computation.
    /// The given initial 'state' argument is passed to the element generator.
    /// For each IEnumerator elements in the stream are generated on-demand by applying the element
    /// generator, until a None value is returned by the element generator. Each call to the element
    /// generator returns a new residual 'state'.
    ///
    /// Note the stream will be recomputed each time an IEnumerator is requested and iterated for the IEnumerable.
    ///
    /// The returned IEnumerable may be passed between threads safely. However, 
    /// individual IEnumerator values generated from the returned IEnumerable should not be accessed from
    /// multiple threads simultaneously.
    val unfold   : generator:('b -> ('a * 'b) option) -> 'b -> seq<'a>


    /// Return an IEnumerable that is built from the given delayed specification of an
    /// IEnumerable. The input function is evaluated each time an IEnumerator for the IEnumerable 
    /// is requested.
    val delay   : (unit -> #seq<'a>) -> seq<'a>

    /// Imperative generation of an IEnumerable from a cursor handle (e.g. from a file handle
    /// or database connection).
    /// 
    /// A cursor is created by calling the 'open' function, which will typically
    /// open a file, a database connection or some other resource.
    /// 
    /// One cursor is created each time an IEnumerator is generated for the IEnumerable, hence
    /// you can use this funciton to create IEnumerables for a logical structure (e.g. a file) that has
    /// multiple active independent readers (e.g. open file handles).  Individual elements are
    /// read from the file using the element generator. The element generator is executed on demand 
    /// as an enumeration proceeds over the IEnumerator, until a None value is returned by the element generator.
    /// 
    /// The 'close' function is called when the None condition is reached or when the IEnumerator object
    /// is explicitly disposed.  The close function is called at most once per active 
    /// enumeration (i.e. once per IEnumerator object created for this IEnumerable). The close function 
    /// is not called during finalization, since the enumerator is not itself the immediate owner of
    /// any resources.
    /// 
    /// Note a fresh call to 'opener' will be used and the stream will be recomputed each time 
    /// an IEnumerator is requested for the IEnumerable.  
    ///
    /// The returned IEnumerable may be passed between threads safely.  However, 
    /// individual IEnumerator values generated from it should not be accessed from
    /// multiple threads simultaneously. The 'close' function may also be called from the finalization thread.
    val generate   : opener:(unit -> 'b) -> generator:('b -> 'a option) -> closer:('b -> unit) -> seq<'a>

    /// Similar to generate, except the 'close' function is implicit from the IDisposable
    /// interface implemented by the state value
    ///
    /// Equivalent to "generate opener generator (fun s -> s.Dispose())"
    val generate_using   : opener:(unit -> ('a :> IDisposable)) -> generator:('a -> 'b option) -> seq<'b>

    /// Generate a sequence by repeatedly calling the given function and
    /// concatenating the results (similar to a while loop)
    val generated   : (unit -> bool) -> #seq<'a> -> seq<'a>

    /// Call the given function every time an enumerator for the returned sequence is disposed.
    val do_finally :  #seq<'a> -> (unit -> unit) -> seq<'a>
    
    /// Generate an IEnumerable based on a function that generates an arbitrarily typed enumerator,
    /// two functions that are effectively the same as IEnumerator.MoveNext and IEnumerator.Current
    val of_functions: (unit -> 'a) -> ('a -> bool) -> ('a -> 'b) -> seq<'b>

    /// Test if any element of the collection satisfies the given predicate.
    /// If the elements are "i0...iN" then computes "p i0 or ... or p iN".
    val exists: ('a -> bool) -> #seq<'a> -> bool

    /// Test if all elements of the collection satisfy the given predicate.
    /// If the elements are "i0...iN" then computes "p i0 &amp;&amp; ... &amp;&amp; p iN".
    val for_all: ('a -> bool) -> #seq<'a> -> bool

    /// Return a new collection containing only the elements of the collection
    /// for which the given predicate returns "true"
    ///
    /// The returned IEnumerable may be passed between threads safely. However, 
    /// individual IEnumerator values generated from the returned IEnumerable should not be accessed from
    /// multiple threads simultaneously.
    ///
    /// Remember IEnumerable is lazy, effects are delayed until it is enumerated.      
    val filter: ('a -> bool) -> #seq<'a> -> seq<'a>

    /// Apply the given function to each element of the collection. 
    val iter: ('a -> unit) -> #seq<'a> -> unit

    /// Build a new collection whose elements are the results of applying the given function
    /// to each of the elements of the collection.  The function will be applied
    /// as elements are demanded using the 'Current' property on iterators retrieved from the
    /// object.  Generating multiple iterators or calling the 'Current' property multiple
    /// times may result in the function being called multiple times.
    ///
    /// The returned IEnumerable may be passed between threads safely. However, 
    /// individual IEnumerator values generated from the returned IEnumerable should not be accessed from
    /// multiple threads simultaneously.
    ///
    /// Remember IEnumerable is lazy, effects are delayed until it is enumerated.
    val map  : ('a -> 'b) -> #seq<'a> -> seq<'b>

    /// Apply the given function to each element of the list. Return
    /// the list comprised of the results "x" for each element where
    /// the function returns Some(x)
    ///
    /// The returned IEnumerable may be passed between threads safely. However, 
    /// individual IEnumerator values generated from the returned IEnumerable should not be accessed from
    /// multiple threads simultaneously.
    ///
    /// Remember IEnumerable is lazy, effects are delayed until it is enumerated.
    val choose: ('a -> 'b option) -> #seq<'a> -> seq<'b>

    /// Apply the given function to successive elements, returning the first
    /// result where function returns "Some(x)" for some x.
    val first: ('a -> 'b option) -> #seq<'a> -> 'b option

    /// Return the first element for which the given function returns "true".
    /// Raise 'Not_found' if no such element exists.
    val find: ('a -> bool) -> #seq<'a> -> 'a

    /// Return the index of the first element in the sequence of pairs
    /// that satisfies the given predicate. Raise 'Not_found' if no such element exists.
    val find_index : ('v -> bool) -> #seq<'k * 'v> -> 'k
  
    /// Return the index of the first element in the sequence of pairs
    /// that satisfies the given predicate. Raise 'Not_found' if no such element exists.
    val find_indexi : ('k -> 'v -> bool) -> #seq<'k * 'v> -> 'k
  

    /// Return the first element for which the given function returns "true".
    /// Return None if no such element exists.
    val tryfind: ('a -> bool) -> #seq<'a> -> 'a option

    /// Return the index of the first element in the sequence of pairs
    /// that satisfies the given predicate. Return 'None' if no such element exists.
    val tryfind_index : ('v -> bool) -> #seq<'k * 'v> -> 'k option
  
    /// Return the index of the first element in the sequence of pairs
    /// that satisfies the given predicate. Return 'None' if no such element exists.
    val tryfind_indexi : ('k -> 'v -> bool) -> #seq<'k * 'v> -> 'k option
  
    /// Compare two IEnumerable's using the given comparison function.  Both
    /// input IEnumerables are assumed to be in a canonical order (i.e. sorted)
    val compare: ('a -> 'a -> int) -> #seq<'a> -> #seq<'a> -> int

    //--------------------------------------------------------------------------
    // The following functions relate IEnumerables to the most common other F# 
    // collection types.
    //--------------------------------------------------------------------------

    /// Build a collection from the given array
    val of_array: 'a array -> seq<'a>

    /// Build an array from the given collection
    val to_array: #seq<'a> -> 'a array

    /// Build a collection from the given array
    val of_list: 'a list -> seq<'a>

    /// Build a list from the given collection
    val to_list: #seq<'a> -> 'a list

    /// Wrap a loosely-typed System.Collections IEnumerable as a typed 
    /// System.Collections.Generic.IEnumerable.
    ///
    /// The use of this function usually requires a type annotation.
    /// An incorrect type annotation may result in runtime type
    /// errors.
    ///
    /// Individual IEnumerator values generated from the returned IEnumerable should not be accessed from
    /// multiple threads simultaneously.
    val untyped_to_typed: #IEnumerable -> seq<'a>

    /// Transform a loosely-typed System.Collections IEnumerable
    /// to a new collection whose elements are the results of applying the given function
    /// to each of the elements of the collection.
    ///
    /// The use of this function usually requires a type annotation.
    /// An incorrect type annotation may result in runtime type
    /// errors.
    ///
    /// The returned IEnumerable may be passed between threads safely. However, 
    /// individual IEnumerator values generated from the returned IEnumerable should not be accessed from
    /// multiple threads simultaneously.
    ///
    /// Remember IEnumerable is lazy, effects are delayed until it is enumerated.
    val map_to_typed: ('a -> 'b) -> #IEnumerable -> seq<'b>

    /// Transform a loosely-typed System.Collections IEnumerable
    /// to a new collection whose elements are the results of applying the given function
    /// to each of the elements of the collection.
    ///
    /// The use of this function usually requires a type annotation.
    /// An incorrect type annotation may result in runtime type
    /// errors.
    ///
    /// The returned IEnumerable may be passed between threads safely. However, 
    /// individual IEnumerator values generated from the returned IEnumerable should not be accessed from
    /// multiple threads simultaneously.
    ///
    /// Remember IEnumerable is lazy, effects are delayed until it is enumerated.
    val map_with_type: ('a -> 'b) -> #IEnumerable -> seq<'b>

    /// Wrap the given enumeration-of-enumerations as a single concatenated
    /// enumeration.
    ///
    /// The returned IEnumerable may be passed between threads safely. However, 
    /// individual IEnumerator values generated from the returned IEnumerable should not be accessed from
    /// multiple threads simultaneously.
    val concat: #seq<#seq<'a>> -> seq<'a> 

    /// For each element of the enumeration apply the given function and concatenate all the results.
    ///
    /// Remember IEnumerable is lazy, effects are delayed until it is enumerated.    
    val map_concat: ('a -> #seq<'b>) -> #seq<'a> -> seq<'b> 

    /// Wrap the two given enumeration-of-enumerations as a single concatenated
    /// enumeration.
    ///
    /// The returned IEnumerable may be passed between threads safely. However, 
    /// individual IEnumerator values generated from the returned IEnumerable should not be accessed from
    /// multiple threads simultaneously.
    val append: #seq<'a>  -> #seq<'a> -> seq<'a> 

    /// Generate a new IEnumerable which, when iterated, will return successive
    /// elements by calling the given function, up to the given count.  The results of calling the function
    /// will not be saved, i.e. the function will be reapplied as necessary to
    /// regenerate the elements.  The function is passed the index of the item being
    /// generated.
    ///
    /// The returned IEnumerable may be passed between threads safely. However, 
    /// individual IEnumerator values generated from the returned IEnumerable should not be accessed from
    /// multiple threads simultaneously.
    val init_finite: int -> (int -> 'a) -> seq<'a>

    /// Generate a new IEnumerable which, when iterated, will return successive
    /// elements by calling the given function.  The results of calling the function
    /// will not be saved, i.e. the function will be reapplied as necessary to
    /// regenerate the elements.  The function is passed the index of the item being
    /// generated
    ///
    /// The returned IEnumerable may be passed between threads safely. However, 
    /// individual IEnumerator values generated from the returned IEnumerable should not be accessed from
    /// multiple threads simultaneously.
    val init_infinite: (int -> 'a) -> seq<'a>

    /// Apply the given function to two collections simultaneously. The
    /// collections must have identical size.
    val iter2: ('a -> 'b -> unit) -> #seq<'a> -> #seq<'b> -> unit

    /// Compute the nth element in the collection.
    val nth: int -> #seq<'a> -> 'a

    /// Build a new collection whose elements are the results of applying the given function
    /// to the corresponding elements of the two collections pairwise.
    ///
    /// Remember IEnumerable is lazy, effects are delayed until it is enumerated.        
    val map2: ('a -> 'b -> 'c) -> #seq<'a> -> #seq<'b> -> seq<'c>

    /// Combine the two sequences into a list of pairs. The two sequences need not have equal lengths:
    /// when one sequence is exhausted any remaining elements in the other
    /// sequence are ignored.
    val zip: #seq<'a> -> #seq<'b> -> seq<'a * 'b>

    /// Use the given function to create a resource, then passs the resource
    /// to the second function. An instance of the resource will be created each
    /// time an IEnumerator for the sequence is created, and disposed when the IEnumerator 
    /// is disposed.
    val using : 'a -> ('a -> #seq<'b>) -> seq<'b> when 'a :> IDisposable

    /// A synonym for Seq.zip
    val combine: #seq<'a> -> #seq<'b> -> seq<'a * 'b>

    /// Build a new collection whose elements are the results of applying the given function
    /// to each of the elements of the collection. The integer index passed to the
    /// function indicates the index (from 0) of element being transformed.
    ///
    /// Remember IEnumerable is lazy, effects are delayed until it is enumerated.            
    val mapi: (int -> 'a -> 'b) -> #seq<'a> -> seq<'b>


    /// Apply the given function to each element of the collection. The integer passed to the
    /// function indicates the index of element.
    val iteri: (int -> 'a -> unit) -> #seq<'a> -> unit

    /// Return a sequence of each element in the input sequence and its predecessor, with the
    /// exception of the first element which is only returned as the predecessor of the second element.
    val pairwise: #seq<'a> -> seq<'a * 'a>

    //--------------------------------------------------------------------------
    // The following functions work over the System.Collections.IEnumerable type
    //--------------------------------------------------------------------------

    /// Apply a function to each element of the collection, threading an 'accumulator' argument
    /// through the computation. If the elements are "i0...iN" then computes "f (... (f s i0)...) iN"
    ///
    /// The use of this function usually requires a type annotation.
    /// An incorrect type annotation may result in runtime type
    /// errors.
    val untyped_fold: ('b -> 'a -> 'b) -> 'b -> #IEnumerable -> 'b

    /// Apply the given function to each element of the collection. 
    ///
    /// The use of this function usually requires a type annotation.
    /// An incorrect type annotation may result in runtime type
    /// errors.
    val untyped_iter: ('a -> unit) -> #IEnumerable -> unit

    /// Build a new collection whose elements are the results of applying the given function
    /// to each of the elements of the collection.
    ///
    /// The use of this function usually requires a type annotation.
    /// An incorrect type annotation may result in runtime type
    /// errors.
    val untyped_map: ('a -> 'b) -> #IEnumerable -> IEnumerable

    /// Return a new collection containing only the elements of the collection
    /// for which the given predicate returns "true"
    ///
    /// The use of this function usually requires a type annotation.
    /// An incorrect type annotation may result in runtime type
    /// errors.
    ///
    /// Remember IEnumerable is lazy, effects are delayed until it is enumerated.
    val untyped_filter: ('a -> bool) -> #IEnumerable -> IEnumerable

    /// Transform a loosely-typed System.Collections IEnumerable to a strongly typed F# list.
    ///
    /// The use of this function usually requires a type annotation.
    /// An incorrect type annotation may result in runtime type
    /// errors.
    val untyped_to_list: #IEnumerable -> 'a list

   
#if CLI_AT_MOST_1_1
#else

    /// The sequence returned by this function caches the computation of the input
    /// sequence. Enumerators for the returned sequence fill a lookaside table
    /// of entries computed so far.  This table is consulted whenever an entry is required
    /// up to the limit of the computed portion of the table. The lookaside table 
    /// is associated with the returned sequence and is used by all enumerators
    /// for the returned sequence. 
    ///
    /// Enumerators of the returned sequence initially generate elements by each using
    /// an enumerator for the underlying sequence. This happens even if some lookaside elements
    /// are available, e.g. from partial or incomplete enumerations of the returned seqeuence. 
    /// However once at least one enumerator manages to reach completion (i.e. IEnumerator.MoveNext returns
    /// false) then all the future enumerators of the returned sequence will will 
    /// simply return the results contained in the lookaside table.
    ///
    /// The sequence must be stable, i.e. each enumerator for the sequence must return 
    /// identical results.
    ///
    /// Generated IEnumerator values may be used simultaneously from different threads (accesses to 
    /// the internal lookaside table are thread safe). Each individual IEnumerator
    /// is not thread safe and should not be accessed from multiple threads simultaneously.
    [<Obsolete("Consider using Seq.cache instead. This has a slightly different behaviour to Seq.cache_all but it is probably the one you want")>]
    val cache_all: #seq<'a> -> seq<'a>

    /// A CacheResult is the value returned by Seq.cache. It is a sequence object that also is IDisposable.
    type CacheResult<'a> with
        interface IDisposable
        interface System.Collections.Generic.IEnumerable<'a>
        interface System.Collections.IEnumerable
        /// Clear any data or partial enumeration stored in the cache. Subsequent enumerations will start again
        /// from the first element with a fresh enumerator.
        member Clear: unit -> unit
        
    /// Return a CacheResult for a sequence that corresponds to a cached version of the input sequence.
    /// This result sequence will have the same elements as the input sequence.
    /// The result can be enumerated multiple times.
    /// The input sequence will be enumerated at most once and only as far as is necessary (it's enumeration is cached).
    ///
    /// Enumeration of the result sequence is thread safe in the sense that multiple independent IEnumerator
    /// values may be used simultaneously from different threads (accesses to 
    /// the internal lookaside table are thread safe). Each individual IEnumerator
    /// is not typically thread safe and should not be accessed from multiple threads simultaneously.
    ///
    /// Note, once enumeration of the input sequence has started,
    /// it's enumerator will be held on to by this object until the enumeration has completed.
    /// At that point, the enumerator will be disposed. 
    ///
    /// The enumerator and underlying cache storage may be released by disposing or clearing the CacheResult. 
    val cache: #seq<'a> -> CacheResult<'a>


    /// Apply a key-generating function to each element of a sequence and returns a sequence of 
    /// unique keys. Each unique key has also contains a sequence of all elements that match 
    /// to this key. 
    /// 
    /// Note that this function digests the whole initial sequence and should not be used with 
    /// large or infinite sequences. The function makes no assumption on the ordering of the original 
    /// sequence.
    val groupBy : ('a -> 'key) -> #seq<'a> -> seq<'key * seq<'a>>

    /// Apply a key-generating function to each element of a sequence and returns a sequence order
    /// by keys.  
    /// 
    /// Note that this function digests the whole initial sequence and should not be used with 
    /// large or infinite sequences. The function makes no assumption on the ordering of the original 
    /// sequence.
    val orderBy : ('a -> 'key) -> #seq<'a> -> seq<'a>

    /// Apply a key-generating function to each element of a sequence and returns a sequence of unique
    /// keys and their number of occurences in the original sequence.  
    /// 
    /// Note that this function digests the whole initial sequence and should not be used with 
    /// large or infinite sequences. The function makes no assumption on the ordering of the original 
    /// sequence.
    val countBy : ('a -> 'key) -> #seq<'a> -> seq<'key * int>

    /// Return the sum of the results generated by applying the function to each element of the array.
    val sumByInt     : ('a -> int)     -> #seq<'a> -> int
    /// Return the sum of the results generated by applying the function to each element of the array.
    val sumByFloat   : ('a -> float)   -> #seq<'a> -> float
    /// Return the sum of the results generated by applying the function to each element of the array.
    val sumByFloat32 : ('a -> float32) -> #seq<'a> -> float32
    /// Return the sum of the results generated by applying the function to each element of the array.
    val sumByInt64   : ('a -> int64)   -> #seq<'a> -> int64

    /// Build a new sequence object that delegates to the given sequence object. This ensures 
    /// the original sequence can't be rediscovered and mutated by a type cast. For example, 
    /// if given an array the returned sequence will return the elements of the array, but
    /// you can't cast the returned sequence object to an array.
    val readonly : #seq<'a> -> seq<'a>
        
#endif

[<Obsolete("The 'IEnumerable' module is obsolete. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
module IEnumerable = 

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val empty : unit -> seq<'a>

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val length: #seq<'a> -> int

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val singleton: 'a -> seq<'a>

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val hd: #seq<'a> -> 'a

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val take: int -> #seq<'a> -> 'a list

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val truncate: int -> #seq<'a> -> seq<'a>

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val nonempty: #seq<'a> -> bool

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val fold: ('b -> 'a -> 'b) -> 'b -> #seq<'a> -> 'b

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val unfold   : generator:('b -> ('a * 'b) option) -> 'b -> seq<'a>


    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val delay   : (unit -> #seq<'a>) -> seq<'a>

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val generate   : opener:(unit -> 'b) -> generator:('b -> 'a option) -> closer:('b -> unit) -> seq<'a>

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val generate_using   : opener:(unit -> ('a :> IDisposable)) -> generator:('a -> 'b option) -> seq<'b>

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val of_functions: (unit -> 'a) -> ('a -> bool) -> ('a -> 'b) -> seq<'b>

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val exists: ('a -> bool) -> #seq<'a> -> bool

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val for_all: ('a -> bool) -> #seq<'a> -> bool

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val filter: ('a -> bool) -> #seq<'a> -> seq<'a>

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val iter: ('a -> unit) -> #seq<'a> -> unit

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val map  : ('a -> 'b) -> #seq<'a> -> seq<'b>

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val choose: ('a -> 'b option) -> #seq<'a> -> seq<'b>

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val first: ('a -> 'b option) -> #seq<'a> -> 'b option

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val find: ('a -> bool) -> #seq<'a> -> 'a

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val tryfind: ('a -> bool) -> #seq<'a> -> 'a option

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val compare: ('a -> 'a -> int) -> #seq<'a> -> #seq<'a> -> int

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val of_array: 'a array -> seq<'a>

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val to_array: #seq<'a> -> 'a array

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val of_list: 'a list -> seq<'a>

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val to_list: #seq<'a> -> 'a list
    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val untyped_to_typed: #IEnumerable -> seq<'a>

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val map_to_typed: ('a -> 'b) -> #IEnumerable -> seq<'b>
    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val map_with_type: ('a -> 'b) -> #IEnumerable -> seq<'b>

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val concat: #seq<#seq<'a>> -> seq<'a> 

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val map_concat: ('a -> #seq<'b>) -> #seq<'a> -> seq<'b> 

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val append: #seq<'a>  -> #seq<'a> -> seq<'a> 

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val init_finite: int -> (int -> 'a) -> seq<'a>

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val init_infinite: (int -> 'a) -> seq<'a>

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val iter2: ('a -> 'b -> unit) -> #seq<'a> -> #seq<'b> -> unit

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val nth: int -> #seq<'a> -> 'a

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val map2: ('a -> 'b -> 'c) -> #seq<'a> -> #seq<'b> -> seq<'c>

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val combine: #seq<'a> -> #seq<'b> -> seq<'a * 'b>

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val mapi: (int -> 'a -> 'b) -> #seq<'a> -> seq<'b>

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val iteri: (int -> 'a -> unit) -> #seq<'a> -> unit

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val untyped_fold: ('b -> 'a -> 'b) -> 'b -> #IEnumerable -> 'b

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val untyped_iter: ('a -> unit) -> #IEnumerable -> unit

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val untyped_map: ('a -> 'b) -> #IEnumerable -> IEnumerable

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val untyped_filter: ('a -> bool) -> #IEnumerable -> IEnumerable

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val untyped_to_list: #IEnumerable -> 'a list

    [<Obsolete("The 'IEnumerable' module has been renamed to Seq. Replace uses of the 'IEnumerable' module with uses of the 'Seq' module")>]
    val pairwise: #IEnumerable<'a> -> IEnumerable<'a * 'a>



