// (c) Microsoft Corporation. All rights reserved 

#light

namespace Microsoft.FSharp.Math

    open Microsoft.FSharp.Collections
    open Microsoft.FSharp.Core
    open Microsoft.FSharp.Math
    open Microsoft.FSharp.Core.LanguagePrimitives.IntrinsicOperators
    open Microsoft.FSharp.Primitives.Basics
    open Microsoft.FSharp.Math.Primitives
    open Microsoft.FSharp.Core.Operators
    open System
    open System.Globalization

    module Z = Microsoft.FSharp.Math.BigInt
    type BigNumLarge = 
        | Q of BigInt * BigInt // invariants: (p,q) in lowest form, q >= 0 

        static member ToString(Q(p,q)) =
            if Z.isOne q then Z.to_string p 
            else Z.to_string p + "/" + Z.to_string q

        static member Hash (Q(ap,aq) as q) = 
            // This hash code must be identical to the hash for BigInt when the numbers coincide.
            if Z.isOne aq then Z.hash(ap) else (Z.hash(ap) <<< 3) + Z.hash(aq)
        
        static member Equals(Q(ap,aq), Q(bp,bq)) = ap=bp && aq=bq   // normal form, so structural equality 
        
        static member LessThan(Q(ap,aq), Q(bp,bq)) = Z.lt  (ap * bq) (bp * aq)
        
        // note: performance improvement possible here
        static member Compare(p,q) = 
            if BigNumLarge.LessThan(p,q) then -1 else
            if BigNumLarge.LessThan(q,p)then  1 else 0 

        interface System.IComparable with 
           member this.CompareTo(that:obj) = 
            match that with 
            | :? BigNumLarge as that -> BigNumLarge.Compare(this,that)
            | _ -> invalid_arg "BigNumLarge.CompareTo: that"

        override this.Equals(that:obj) = 
            match that with 
            | :? BigNumLarge as that -> BigNumLarge.Equals(this,that)
            | _ -> false

        override x.ToString()               = BigNumLarge.ToString(x)

        override x.GetHashCode()            = BigNumLarge.Hash(x)

        interface IStructuralHash with 
            member x.GetStructuralHashCode(_) = BigNumLarge.Hash(x)

    [<CompilationRepresentation(CompilationRepresentationFlags.ModuleSuffix)>]
    module BigNumLarge = 
    
        let normalize p q =
            if Z.isZero q then
                if   Z.negative p then Q(-Z.one,Z.zero)      // -1/0 
                elif Z.positive p then Q(Z.one,Z.zero)       // +1/0 
                else                   Q(Z.zero,Z.zero)      //  0/0 
            elif Z.isOne q then
                Q(p,q)
            else
                let k = Z.hcf p q 
                let p = p / k 
                let q = q / k 
                if Z.negative q then Q(-p,-q) else Q(p,q)

        let rational  p q = normalize (Z.of_int p) (Z.of_int q)
        let rationalz p q = normalize p q
       
        let add (Q(ap,aq)) (Q(bp,bq)) = normalize ((ap * bq) + (bp * aq)) (aq * bq)
        let sub (Q(ap,aq)) (Q(bp,bq)) = normalize ((ap * bq) - (bp * aq)) (aq * bq)
        let mul (Q(ap,aq)) (Q(bp,bq)) = normalize (ap * bp) (aq * bq)
        let div (Q(ap,aq)) (Q(bp,bq)) = normalize (ap * bq) (aq * bp)
          
        let sqr    (Q(ap,aq)) = Q(ap * ap,aq * aq) // still coprime, aq^2 >= 0 
        let inv    (Q(ap,aq)) = normalize aq ap    
        let neg (Q(bp,bq))    = Q(-bp,bq)          // still coprime, bq >= 0 

        let powi (Q(p,q)) n = Q(Z.powi p n,Z.powi q n)     // p,q powers still coprime 
        let pow  (Q(p,q)) z = Q(Z.pow  p z,Z.pow  q z)     // p,q powers still coprime 
        
        let equal (Q(ap,aq)) (Q(bp,bq)) = ap=bp && aq=bq   // normal form, so structural equality 
        let lt    a b = BigNumLarge.LessThan(a,b)
        let gt    a b = BigNumLarge.LessThan(b,a)
        let lte   (Q(ap,aq)) (Q(bp,bq)) = Z.lte (ap * bq) (bp * aq)
        let gte   (Q(ap,aq)) (Q(bp,bq)) = Z.gte (ap * bq) (bp * aq)

        let compare p q = // improvement possible 
            if lt p q then -1 else
            if gt p q then  1 else 0 
        
        let negative (Q(ap,aq)) = Z.negative ap  
        let positive (Q(ap,aq)) = Z.positive ap
        let integral (Q(ap,aq)) = Z.isOne aq
        
        let hash q = BigNumLarge.Hash(q)
        
        let max x y = if lt x y then y else x
        let min x y = if lt x y then x else y
        let sign (Q(p,q)) =
            if   Z.negative p then -1 
            elif Z.positive p then 1  
            else 0  
        
        let to_float (Q(p,q)) = Z.to_float p / Z.to_float q
        
        let of_bigint   z = rationalz z Z.one 
        let of_int n = rational  n 1
       
        let to_string n = BigNumLarge.ToString(n)
        
        let of_string (str:string) =
          let len = str.Length 
          if len=0 then invalid_arg "of_string: empty string";
          let j = str.IndexOf '/' 
          if j >= 0 then 
              let p = Z.of_string (str.Substring(0,j)) 
              let q = Z.of_string (str.Substring(j+1,len-j-1)) 
              rationalz p q
          else
              let p = Z.of_string str 
              rationalz p Z.one
        
        // integer and fraction parts 
        let integer (Q(p,q)) =
            let d,r = Z.divmod p q          // have p = d.q + r, |r| < |q| 
            if Z.negative r 
            then d - Z.one                 // p = (d-1).q + (r+q) 
            else d                             // p =     d.q + r 
        
        let fraction (Q(p,q)) =
            let d,r = Z.divmod p q          // have p = d.q + r 
            if Z.negative r then
                let r = r + q               // p = (d-1).q + (r+q) 
                rationalz r q
            else
                rationalz r q                 // p =     d.q + r 

        let two = Z.of_int 2
        let round (Q(p,q)) =
            let d,r = Z.divmod p q             // have p = d.q + r  for  |r|<q 
            if Z.positive r then
                if Z.lt (q / two) r 
                then Z.one + d                 // have q/2 < r < q   , round up 
                else d                         // have 0   < r <= q/2, note 1/2 rounds down 
            elif Z.negative r then
              if Z.lte (q / two) (-r) 
              then  Z.one - d                  // have -1 < r <= q/2 , round down. note 1/2 rounds down again 
              else d                           // have -q/2 < r < 0  , rounds up 
            else d                             // have r=0 
        
        let floor (Q(p,q)) =
            let d,r = Z.divmod p q          // p = d.q + r 
            if Z.negative r 
            then d - Z.one                  // r -ve, so round down 
            else d
        
        let ceiling (Q(p,q)) =
          let d,r = Z.divmod p q          // p = d.q + r 
          if Z.positive r 
          then d + Z.one                 // when r +ve, adjust up 
          else d
      
    type BigNumLarge with 
        static member ( +  )(n1,n2) = BigNumLarge.add n1 n2
        static member ( *  )(n1,n2) = BigNumLarge.mul n1 n2
        static member ( -  )(n1,n2) = BigNumLarge.sub n1 n2 
        static member ( /  )(n1,n2) = BigNumLarge.div n1 n2
        static member ( ~- )(n1) = BigNumLarge.neg n1
        static member ( ~+ )(n1:BigNumLarge) = n1
        
    //----------------------------------------------------------------------------
    // BigNum
    //--------------------------------------------------------------------------

    type BigNum =
        //   | I of int32 
        | Z of BigInt
        | Q of BigNumLarge
        interface System.IComparable
        interface IStructuralHash
        override this.Equals(that:obj) = ((this :> System.IComparable).CompareTo(that) = 0)

    type BigRational = BigNum

    [<CompilationRepresentation(CompilationRepresentationFlags.ModuleSuffix)>]
    module BigNum = 

        module Z = Microsoft.FSharp.Math.BigInt
        module Q = BigNumLarge

        let of_int      x = Z (Z.of_int x)
        let of_bigint   x = Z x

        let zero = (of_int 0) 
        let one = (of_int 1) 
        let inline bop (zop,qop) n nn =
            match n,nn with
            | Z z ,Z zz -> (zop z zz)
            | Q q ,Q qq -> (qop q qq)
            | Z z ,Q qq -> (qop (Q.of_bigint z) qq)
            | Q q ,Z zz -> (qop q (Q.of_bigint zz))

        let add n nn = bop ((fun x y -> Z (x + y)),       (fun x y -> Q (x + y)) ) n nn
        let sub n nn = bop ((fun x y -> Z (x - y)),       (fun x y -> Q (x - y)) ) n nn
        let mul n nn = bop ((fun x y -> Z (x * y)),       (fun x y -> Q (x * y)) ) n nn
        let div n nn = bop ((fun x y -> Q (Q.rationalz x y)), (fun x y -> Q (x / y)) ) n nn

        let neg = function
            | Z z -> Z (-z)
            | Q q -> Q (-q)

        let powi n i =
            match n with
            | Z z -> Z (Z.powi z i)
            | Q q -> Q (Q.powi q i)

        let equal n nn = bop (Z.equal,Q.equal) n nn
        let lt    n nn = bop (Z.lt   ,Q.lt   )    n nn
        let gt    n nn = bop (Z.gt   ,Q.gt   )    n nn
        let lte   n nn = bop (Z.lte  ,Q.lte  )   n nn
        let gte   n nn = bop (Z.gte  ,Q.gte  )   n nn
        let compare n nn = if lt n nn then -1 elif equal n nn then 0 else 1
       
        let hash = function 
            | Z z -> Z.hash(z)
            | Q q -> Q.hash(q) // nb. Q and Z hash codes must match up - see note in q.fs

        let negative = function
            | Z z -> Z.negative z
            | Q q -> Q.negative q

        let positive = function
            | Z z -> Z.positive z
            | Q q -> Q.positive q
            
        let max x y = if lt x y then y else x
        let min x y = if lt x y then x else y
        let sign x =
            if negative x then -1 else 
            if positive x then  1 else 0
        let abs x =
            if negative x then neg x else x

        let to_float = function
            | Z z -> Z.to_float z
            | Q q -> Q.to_float q

        let to_bigint = function
            | Z z -> z
            | Q q -> Q.integer q 

        let to_int = function
            | Z z -> Z.to_int(z)
            | Q q -> Z.to_int(Q.integer q )

        let to_string = function
            | Z z -> Z.to_string(z)
            | Q q -> Q.to_string(q)
       
        let of_string n = Q (Q.of_string n)

    type BigNum with 
        static member ( + )(n1,n2) = BigNum.add n1 n2
        static member ( * )(n1,n2) = BigNum.mul n1 n2
        static member ( - )(n1,n2) = BigNum.sub n1 n2 
        static member ( / )(n1,n2) = BigNum.div n1 n2
        static member ( ~- )(n1) = BigNum.neg n1
        static member ( ~+ )(n1:BigNum) = n1
        static member (..) (n1:BigNum,n2:BigNum) = StandardRanges.generate BigNum.zero BigNum.add n1 BigNum.one n2
        static member (.. ..) (n1:BigNum,step:BigNum,n2:BigNum) = StandardRanges.generate BigNum.zero BigNum.add n1 step n2
        member x.Details = 
          match x with
          | Z z -> BigNumLarge.of_bigint z
          | Q q -> q

        override x.ToString() = BigNum.to_string(x)
        override x.GetHashCode() = BigNum.hash(x)

        interface System.IComparable with 
           member x.CompareTo(y:obj) = BigNum.compare x (y :?> BigNum)

        interface IStructuralHash with 
           member x.GetStructuralHashCode(nodesRemaining) = BigNum.hash(x) // ignore nodesRemaining

namespace Microsoft.FSharp.Core

    type bignum = Microsoft.FSharp.Math.BigNum

namespace Microsoft.FSharp.Math.Types

    type BigRational = Microsoft.FSharp.Math.BigNum
    type BigNum = Microsoft.FSharp.Math.BigNum
