(*-------------------------------------------------------------------------
 * Apply default values to unresolved type variables throughout an expression
 *------------------------------------------------------------------------- *)

(*F#
module Microsoft.FSharp.Compiler.FindUnsolved
open Microsoft.Research.AbstractIL
open Microsoft.Research.AbstractIL.Internal
open Microsoft.FSharp.Compiler

module Il = Microsoft.Research.AbstractIL.IL
module Ildiag = Microsoft.Research.AbstractIL.Diagnostics
F#*)

open Ildiag
open List
open Range
open Ast
open Tast
open Tastops
open Env
open Lib
open Layout
open Il
open Typrelns
open Infos

type env = Nix

type cenv = { g: tcGlobals; amap: Import.importMap; denv: displayEnv; mutable unsolved: local_typar_ref list }
let mk_cenv  g amap denv =  { g =g ; amap=amap; denv=denv; unsolved = [] }

(*--------------------------------------------------------------------------
!* eliminate internal uninstantiated type variables
 *--------------------------------------------------------------------------*)

let acc_ty cenv env ty =
    (free_in_type ty).free_loctypars |> Zset.iter (fun tp -> 
            if (rigid_of_tpref tp <> TyparRigid) then 
                cenv.unsolved <- tp :: cenv.unsolved) 

let acc_tinst cenv env tyargs =
  tyargs |> List.iter (acc_ty cenv env)

(*--------------------------------------------------------------------------
!* check exprs etc
 *--------------------------------------------------------------------------*)
  
let rec acc_expr   (cenv:cenv) (env:env) expr =     
    let expr = strip_expr expr in
    match expr with
    | TExpr_seq (e1,e2,flag,m) -> 
        acc_expr cenv env e1; 
        acc_expr cenv env e2
    | TExpr_let (bind,body,m,_) ->  
        acc_bind cenv env bind ; 
        acc_expr cenv env body
    | TExpr_const (_,_,ty) -> 
        acc_ty cenv env ty 
    
    | TExpr_val (v,vFlags,m) -> ()
    | TExpr_hole (m,ty) -> 
          acc_ty cenv env ty 
    | TExpr_quote(raw,ast,m,ty) -> 
          acc_expr cenv env ast;
          acc_ty cenv env ty;
    | TExpr_obj (_,typ,basev,basecall,overrides,iimpls,m,_) -> 
          acc_expr cenv env basecall;
          acc_methods cenv env basev overrides ;
          acc_iimpls cenv env basev iimpls;
    | TExpr_op (c,tyargs,args,m) ->
          acc_op cenv env (c,tyargs,args,m) 
    | TExpr_app(f,fty,tyargs,argsl,m) ->
          acc_ty cenv env fty;
          acc_tinst cenv env tyargs;
          acc_expr cenv env f;
          acc_exprs cenv env argsl
    (* REVIEW: fold the next two cases together *)
    | TExpr_lambda(lambda_id,basevopt,argvs,body,m,rty,_) -> 
        let arity_info = TopValInfo (0,[argvs |> map (fun _ -> TopValData.unnamedTopArg1)],TopValData.unnamedRetVal) in 
        let ty = mk_multi_lambda_ty argvs rty in 
        acc_lambdas cenv env arity_info expr ty
    | TExpr_tlambda(lambda_id,tps,body,m,rty,_)  -> 
        let arity_info = TopValInfo (length tps,[],TopValData.unnamedRetVal) in
        acc_ty cenv env rty;
        let ty = try_mk_forall_ty tps rty in 
        acc_lambdas cenv env arity_info expr ty
    | TExpr_tchoose(tps,e1,m)  -> 
        acc_expr cenv env e1 
    | TExpr_match(exprm,dtree,targets,m,ty,_) -> 
        acc_ty cenv env ty;
        acc_dtree cenv env dtree;
        acc_targets cenv env m ty targets;
    | TExpr_letrec (binds,e,m,_) ->  
        acc_binds cenv env binds;
        acc_expr cenv env e
    | TExpr_static_optimization (constraints,e2,e3,m) -> 
        acc_expr cenv env e2;
        acc_expr cenv env e3;
        constraints |> iter (fun (TTyconEqualsTycon(ty1,ty2)) -> 
            acc_ty cenv env ty1;
            acc_ty cenv env ty2)
    | TExpr_link eref -> failwith "unexpected reclink"

and acc_methods cenv env basevopt l = List.iter (acc_method cenv env basevopt) l
and acc_method cenv env basevopt (TMethod(slotsig,tps,vs,e,m) as tmethod) = 
    vs |> iter (acc_val cenv env);
    acc_expr cenv env e

and acc_iimpls cenv env basevopt l = List.iter (acc_iimpl cenv env basevopt) l
and acc_iimpl cenv env basevopt (ty,overrides) = acc_methods cenv env basevopt overrides 

and acc_op cenv env (op,tyargs,args,m) =
    (* Special cases *)
    acc_tinst cenv env tyargs;
    acc_exprs cenv env args;
    match op with 
    (* Handle these as special cases since mutables are allowed inside their bodies *)
    | TOp_ilcall ((virt,protect,valu,newobj,superInit,prop,isDllImport,boxthis,mref),enclTypeArgs,methTypeArgs,tys) ->
        acc_tinst cenv env enclTypeArgs;
        acc_tinst cenv env methTypeArgs;
        acc_tinst cenv env tys
    | TOp_asm (_,tys) ->
        acc_tinst cenv env tys
    | _ ->    ()

and acc_lambdas cenv env arity_info e ety =
  (* The arity_info here says we are _guaranteeing_ to compile a function value *)
  (* as a .NET method with precisely the corresponding argument counts. *)
  match e with
  | TExpr_tchoose(tps,e1,m)  -> acc_lambdas cenv env arity_info e1 ety      
  | TExpr_lambda (lambda_id,_,_,_,m,_,_)  
  | TExpr_tlambda(lambda_id,_,_,m,_,_) ->
      let tps,basevopt,vsl,body,bodyty = dest_top_lambda_upto cenv.g cenv.amap arity_info (e, ety) in
      acc_ty cenv env bodyty;
      vsl |> iter (iter (acc_val cenv env));
      basevopt |> Option.iter (acc_val cenv env);
      acc_expr cenv env body;
  | _ -> 
      acc_expr cenv env e

and acc_exprs            cenv env exprs = iter (acc_expr cenv env) exprs
and acc_targets cenv env m ty targets = Array.iter (acc_target cenv env m ty) targets

and acc_target cenv env m ty (TTarget(vs,e)) = acc_expr cenv env e;

and acc_dtree cenv env x =
  match x with 
  | TDSuccess (es,n) -> acc_exprs cenv env es;
  | TDBind(bind,rest) -> acc_bind cenv env bind; acc_dtree cenv env rest 
  | TDSwitch (e,cases,dflt,m) -> acc_switch cenv env (e,cases,dflt,m)

and acc_switch cenv env (e,cases,dflt,m) =
  acc_expr cenv env e;
  iter (fun (TCase(discrim,e)) -> acc_discrim cenv env discrim; acc_dtree cenv env e) cases;
  Option.iter (acc_dtree cenv env) dflt

and acc_discrim cenv env d =
  match d with 
  | TTest_unionconstr(ucref,tinst) -> acc_tinst cenv env tinst 
  | TTest_array_length(_,ty) -> acc_ty cenv env ty
  | TTest_const _
  | TTest_isnull -> ()
  | TTest_isinst (srcty,tgty) -> acc_ty cenv env srcty; acc_ty cenv env tgty
  | TTest_query (exp, tys, vref, idx, apinfo) -> 
      acc_expr cenv env exp;
      acc_tinst cenv env tys

and acc_attrib cenv env (Attrib(k,args,props)) = 
  acc_exprs cenv env args;
  props |> iter (fun (nm,ty,flg,expr) -> acc_expr cenv env expr)
  
and acc_attribs cenv env attribs = iter (acc_attrib cenv env) attribs

and acc_topValInfo cenv env (TopValInfo(_,args,ret)) =
    args |> List.iter (List.iter (acc_topArgInfo cenv env));
    ret |> acc_topArgInfo cenv env;

and acc_topArgInfo cenv env (TopArgData(attribs,_)) = 
    acc_attribs cenv env attribs

and acc_val cenv env v =
    v |> attribs_of_val |> acc_attribs cenv env;
    v |> arity_of_val |> Option.iter (acc_topValInfo cenv env);
    v |> type_of_val |> acc_ty cenv env 

and acc_bind cenv env (TBind(v,e) as bind) =
    acc_val cenv env v;    
    let arity_info  = match chosen_arity_of_bind bind with Some info -> info | _ -> TopValData.emptyTopValData in
    acc_lambdas cenv env arity_info e (type_of_val v);

and acc_binds cenv env xs = iter (acc_bind cenv env) xs

let modul_rights cpath = Infos.AccessibleFrom (cpath,None)

(*--------------------------------------------------------------------------
!* check tycons
 *--------------------------------------------------------------------------*)
  
let acc_tycon_rfield cenv env tycon (rfield:recdfield_spec) = 
  acc_attribs cenv env (pattribs_of_rfield rfield);
  acc_attribs cenv env (fattribs_of_rfield rfield)

let acc_tycon cenv env tycon =
  acc_attribs cenv env (attribs_of_tycon tycon);
  tycon |> rfields_array_of_tycon |> Array.iter (acc_tycon_rfield cenv env tycon);
  if is_union_tycon tycon then (                            (* This covers finite unions. *)
    uconstrs_of_tycon tycon |> List.iter (fun uc ->
        acc_attribs cenv env (attribs_of_uconstr uc);
        uc |> rfields_of_uconstr |> List.iter (acc_tycon_rfield cenv env tycon))
  )

let acc_tycons cenv env tycons = List.iter (acc_tycon cenv env) tycons

(*--------------------------------------------------------------------------
!* check modules
 *--------------------------------------------------------------------------*)

let rec acc_mexpr cenv env x = 
    match x with  
    | TMTyped(mty,def,m) -> acc_mdef cenv env def
    
and acc_mdefs cenv env x = iter (acc_mdef cenv env) x

and acc_mdef cenv env x = 
    match x with 
    | TMDefRec(tycons,binds,m) -> acc_tycons cenv env tycons; acc_binds cenv env binds 
    | TMDefLet(bind,m)  -> acc_bind cenv env bind 
    | TMAbstract(def)  -> acc_mexpr cenv env def
    | TMDefModul(TMBind(tycon, rhs)) -> acc_tycon cenv env tycon; acc_mdef cenv env rhs 
    | TMDefs(defs) -> acc_mdefs cenv env defs 

let unsolved_typars_of_mdef g amap denv mdef =
   let cenv = mk_cenv  g amap denv in 
   acc_mdef cenv Nix mdef;
   List.rev cenv.unsolved


