(* (c) Microsoft Corporation 2005-2006.  *)

open Lexing
open Fslexast
open Fslexpars
(*F# module Unilex = Microsoft.FSharp.Compiler.UnicodeLexing F#*)

let input = ref None
let out = ref None
let unicode = ref false
let inputCodePage = ref None

let usage =
  [ "-o", Arg.String (fun s -> out := Some s), "Name the output file."; 
(*F# "--codepage", Arg.Int (fun i -> inputCodePage := Some i), "Assume input lexer specification file is encoded with the given codepage.";  F#*)
(*    "--unicode", Arg.Set unicode, "Produce a lexer for use with 16-bit unicode characters.";  *) ]

let _ = Arg.parse usage (fun x -> match !input with Some _ -> failwith "more than one input given" | None -> input := Some x) "fslex <filename>"

let output_int os n = output_string os (string_of_int n)

(* nb. using printf is a little slow here *)
let output_trigraph os n = 
  output_string os "\\";
  output_int os (n/100);
  output_int os ((n mod 100)/10);
  output_int os (n mod 10)

let output_coded_u16 os n = 
  output_trigraph os (n/256);
  output_trigraph os (n mod 256)

let sentinel = 255 * 256 + 255 

let main() = 
  try 
    let filename = (match !input with Some x -> x | None -> failwith "no input given") in 
    let domain = if !unicode then "Unicode" else "Ascii" in 
    let spec = 
      Unilex.usingUnicodeFileAsUTF8Lexbuf filename !inputCodePage (fun lexbuf -> 
        try 
          (*IF-FSHARP Lexing.lexbuf_set_curr_p lexbuf ENDIF-FSHARP*)
          (*IF-OCAML*) lexbuf.Lexing.lex_curr_p <- (*ENDIF-OCAML*)
             {(lexeme_end_p lexbuf) with pos_fname = filename; pos_cnum=0; pos_lnum=1 };
          Fslexpars.spec Fslexlex.token lexbuf 
        with e -> 
          Printf.eprintf "%s(%d,%d): error: %s\n" filename (lexeme_start_p lexbuf).pos_lnum ((lexeme_start_p lexbuf).pos_cnum -  (lexeme_start_p lexbuf).pos_bol) (match e with Failure s -> s | _ -> Printexc.to_string e);
          exit 1
       ) in 
    Printf.printf "compiling to dfas (can take a while...)\n"; flush stdout;
    let perRuleData, dfaNodes = Fslexast.compile spec in 
    let dfaNodes = List.sort (fun n1 n2 -> compare n1.id n2.id) dfaNodes in
    Printf.printf "%d NFA nodes\n" (Fslexast.newNfaNodeId());
    Printf.printf "%d states\n" (List.length dfaNodes); flush stdout;
    Printf.printf "writing output\n"; flush stdout;
    let output = match !out with Some x -> x | _ -> Filename.chop_extension filename ^ ".fs" in
    let os = open_out output in
    let (code,pos) = spec.topcode in 
    Printf.fprintf os "# %d \"%s\"\n" pos.pos_lnum pos.pos_fname;
    Printf.fprintf os "%s\n" code;
    Printf.fprintf os "# 1000000 \"%s\"\n" output;
    
    Printf.fprintf os "let trans : byte[] array = [| \n";
    List.iter 
      (fun state -> 
        Printf.fprintf os "   (* State %d *)\n" state.id;
        Printf.fprintf os " \"";
        let emit n = 
          if List.mem_assoc n state.tr then 
            let s2 = List.assoc n state.tr in
            output_coded_u16 os s2.id 
          else
            output_coded_u16 os sentinel in
        for i = 0 to 255 do 
          let c = Char.chr i in 
          emit (Some (LChar c));
        done;
        emit (Some LEof);
        Printf.fprintf os "\"B;\n")
      dfaNodes;
    Printf.fprintf os "|] \n";
    
    Printf.fprintf os "let actions : byte[] = \"";
    List.iter 
      (fun state -> 
        if List.length state.ac > 0 then 
          output_coded_u16 os (snd (List.hd state.ac))
        else
        output_coded_u16 os sentinel)
      dfaNodes;
    Printf.fprintf os "\"B\n";
    Printf.fprintf os "let __fslex_tables = Microsoft.FSharp.Tools.FsLex.%sTables.Create(trans,actions)\n" domain;
    
    Printf.fprintf os "let rec __fslex_dummy () = __fslex_dummy() \n";
    List.iter2 
      (fun (startNode, actions) (ident,args,_) -> 
        Printf.fprintf os "(* Rule %s *)\n" ident;
        Printf.fprintf os "and %s %s (lexbuf : Microsoft.FSharp.Tools.FsLex.LexBuffer<_,_>) = __fslex_%s %s %d lexbuf\n" ident (String.concat " " args) ident (String.concat " " args) startNode.id;
        Printf.fprintf os "and __fslex_%s %s __fslex_state lexbuf =\n" ident (String.concat " " args);
        Printf.fprintf os "  match __fslex_tables.Interpret(__fslex_state,lexbuf) with\n" ;
        list_iteri
          (fun i (code,pos) -> 
            Printf.fprintf os "  | %d -> ( \n" i;
            Printf.fprintf os "# %d \"%s\"\n" pos.pos_lnum pos.pos_fname;
            Printf.fprintf os "%s%s\n" (String.make (pos.pos_cnum - pos.pos_bol) ' ') code;
            Printf.fprintf os "# 2000000 \"%s\"\n" output;
            Printf.fprintf os ")\n")
          actions;
        Printf.fprintf os "  | _ -> failwith \"%s\"\n" ident)
      perRuleData
      spec.rules;
    
    let (code,pos) = spec.botcode in 
    Printf.fprintf os "\n";
    Printf.fprintf os "# %d \"%s\"\n" pos.pos_lnum pos.pos_fname;
    Printf.fprintf os "%s\n" code;
    Printf.fprintf os "# 3000000 \"%s\"\n" output;
    
    close_out os
  with e -> 
    Printf.eprintf "Error: %s\n" (match e with Failure s -> s | e -> Printexc.to_string e);
    exit 1


let _ = main()
