/* PgSqlClient - ADO.NET Data Provider for PostgreSQL 7.4+
 * Copyright (c) 2003-2004 Carlos Guzman Alvarez
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using System;
using System.Data;
using System.Globalization;
using System.IO;
using System.Text;
using System.Net;

using PostgreSql.Data.PgTypes;

namespace PostgreSql.Data.NPgClient
{
	internal class PgResponsePacket : BinaryReader
	{
		#region Fields

		private char		message;
		private Encoding	encoding;

		#endregion

        #region Properties

		public char Message
		{
			get { return message; }
			set { message = value; }
		}

		public Encoding Encoding
		{
			get { return encoding; }
			set { encoding = value; }
		}

		public long Length
		{
			get { return ((MemoryStream)BaseStream).Length; }
		}

		public long Position
		{
			get { return ((MemoryStream)BaseStream).Position; }
		}

		public bool EOF
		{
			get
			{
				if (Position < Length)
				{
					return false;
				}
				else
				{
					return true;
				}
			}
		}

		public long Pending
		{
			get { return Length - Position; }
		}

		#endregion

		#region Constructors

		public PgResponsePacket() : base(new MemoryStream())
		{
		}

		public PgResponsePacket(byte[] contents) : base(new MemoryStream(contents))
		{
		}

		public PgResponsePacket(char message, byte[] contents) : base(new MemoryStream(contents))
		{
			this.message = message;
		}

		#endregion

		#region Stream Methods

		public void Reset()
		{
			((MemoryStream)BaseStream).SetLength(0);
			((MemoryStream)BaseStream).Position = 0;
		}

		public byte[] ToArray()
		{
			return ((MemoryStream)BaseStream).ToArray();
		}

		#endregion

		#region String Types

		public string ReadNullString()
		{
			StringBuilder cString = new StringBuilder();
			char c;

			while ((c = ReadChar()) != PgCodes.NULL_TERMINATOR )
			{
				cString.Append(c);
			}
			
			return cString.ToString();
		}

		public string ReadString(int length)
		{
			byte[] buffer = new byte[length];
						
			Read(buffer, 0, length);
			
			return encoding.GetString(buffer);
		}

		public override string ReadString()
		{
			int length = this.ReadInt();
			byte[] buffer = new byte[length];
						
			Read(buffer, 0, length);
			
			return encoding.GetString(buffer);
		}

		#endregion

		#region Numeric Types

		public short ReadShort()
		{
			short val = base.ReadInt16();

			return IPAddress.HostToNetworkOrder(val);
		}

		public int ReadInt()
		{
			int val = base.ReadInt32();

			return IPAddress.HostToNetworkOrder(val);
		}

		public long ReadLong()
		{
			return IPAddress.HostToNetworkOrder(base.ReadInt64());
		}

		public override float ReadSingle()
		{			
			FloatLayout floatValue = new FloatLayout();
			floatValue.i0 = IPAddress.HostToNetworkOrder(base.ReadInt32());

			return floatValue.f;
		}

		public float ReadCurrency()
		{
			float val = (float)ReadInt();

			return val/100;
		}
		
		public override double ReadDouble()
		{			
			DoubleLayout doubleValue = new DoubleLayout();
			int temp;			

			doubleValue.d = base.ReadDouble();
			doubleValue.i0 = IPAddress.HostToNetworkOrder(doubleValue.i0);
			doubleValue.i4 = IPAddress.HostToNetworkOrder(doubleValue.i4);

			temp = doubleValue.i0;
			doubleValue.i0 = doubleValue.i4;
			doubleValue.i4 = temp;

			return doubleValue.d;
		}

		#endregion

		#region Date & Time Types

		public DateTime ReadDate()
		{
			int days = ReadInt();
			
			DateTime date = new DateTime(days);
			
			return PgCodes.BASE_DATE.AddDays(days);
		}

		public TimeSpan ReadInterval()
		{
			double	intervalTime	= this.ReadDouble();
			int		intervalMonth	= this.ReadInt();

			TimeSpan interval = TimeSpan.FromSeconds(intervalTime);
			
			return interval.Add(TimeSpan.FromDays(intervalMonth*30));
		}

		public DateTime ReadTime()
		{
			double seconds = ReadDouble();
			
			return PgCodes.BASE_DATE.AddSeconds(seconds);
		}

		public DateTime ReadTimeWithTZ()
		{
			DateTime	time		= this.ReadTime();
			int			timezone	= (-1)*(this.ReadInt()/3600);

			string sbFormat = "{0:D2}{1}{2:D2}{3}{4:D2}{5:D2}";
			if (timezone >= 0)
			{
				sbFormat = "{0:D2}{1}{2:D2}{3}{4:D2}+{5:D2}";
			}

			StringBuilder timeWithTZ = new StringBuilder();
			timeWithTZ.AppendFormat(
				sbFormat,
				time.Hour,
				CultureInfo.CurrentCulture.DateTimeFormat.TimeSeparator,
				time.Minute, 
				CultureInfo.CurrentCulture.DateTimeFormat.TimeSeparator,
				time.Second,
				timezone);
		
			StringBuilder format = new StringBuilder();
			format.AppendFormat(
				"{0}{1}{2}{3}{4}{5}",
				"HH",
				CultureInfo.CurrentCulture.DateTimeFormat.TimeSeparator,
				"mm", 
				CultureInfo.CurrentCulture.DateTimeFormat.TimeSeparator,
				"ss",
				"zz");
		
			return DateTime.ParseExact(
				timeWithTZ.ToString(), 
				format.ToString(),
				CultureInfo.CurrentCulture.DateTimeFormat,
				DateTimeStyles.NoCurrentDateDefault | 
				DateTimeStyles.AllowWhiteSpaces);
		}

		public DateTime ReadTimestamp()
		{
			double seconds = ReadDouble();
			
			return PgCodes.BASE_DATE.AddSeconds(seconds);
		}

		public DateTime ReadTimestampWithTZ()
		{
			return this.ReadTimestamp();
		}

		#endregion

		#region Array & Vector Types

		public Array ReadArray(PgType type, int length)
		{
			if (type.FormatCode == 0)
			{
				return this.readStringArray(type, length);
			}
			else
			{
				int[] lengths;
				int[] lowerBounds;

				// Read number of dimensions
				int dimensions	= this.ReadInt();
				
				// Initialize arrays for lengths and lower bounds
				lengths		= new int[dimensions];
				lowerBounds	= new int[dimensions];

				// Read flags value
				int flags = this.ReadInt();
				if (flags != 0)
				{
					throw new NotSupportedException("Invalid flags value");
				}
				
				// Read array element type
				PgType elementType = PgDbClient.Types[this.ReadInt()];

				// Read array lengths and lower bounds
				for (int i = 0; i < dimensions; i++)
				{
					lengths[i]		= this.ReadInt();
					lowerBounds[i]	= this.ReadInt();
				}

				// Read Array data
				if (type.SystemType.IsPrimitive)
				{
					return this.readPrimitiveArray(elementType, length, 
						dimensions, flags, lengths, lowerBounds);
				}				
				else
				{
					return this.readNonPrimitiveArray(elementType, length, 
						dimensions, flags, lengths, lowerBounds);
				}
			}
		}

		public Array ReadVector(PgType type, int length)
		{
			PgType	elementType = PgDbClient.Types[type.ElementType];
			Array	data		= null;
			
			data = Array.CreateInstance(elementType.SystemType, PgCodes.INDEX_MAX_KEYS);

			for (int i = 0; i < data.Length; i++ )
			{
				object elementValue = ReadValue(elementType, elementType.Size);
				data.SetValue(elementValue, i);
			}

			return data;
		}

		#endregion

		#region Geometric Types

		public PgPoint ReadPoint()
		{
			double x = this.ReadDouble();
			double y = this.ReadDouble();

			return new PgPoint(x, y);
		}

		public PgCircle ReadCircle()
		{
			return new PgCircle(this.ReadPoint(), this.ReadDouble());
		}

		public PgLine ReadLine()
		{
			return new PgLine(this.ReadPoint(), this.ReadPoint());
		}

		public PgLSeg ReadLSeg()
		{
			return new PgLSeg(this.ReadPoint(), this.ReadPoint());
		}

		public PgBox ReadBox()
		{
			PgPoint upperRight	= this.ReadPoint();
			PgPoint lowerLeft	= this.ReadPoint();

			return new PgBox(lowerLeft, upperRight);
		}

		public PgPolygon ReadPolygon()
		{
			PgPoint[] points = new PgPoint[this.ReadInt()];

			for (int i = 0; i < points.Length; i++)
			{
				points[i] = this.ReadPoint();
			}

			return new PgPolygon(points);
		}

		public PgPath ReadPath()
		{
			bool		isClosedPath	= this.ReadBoolean();
			PgPoint[]	points			= new PgPoint[this.ReadInt()];

			for (int i = 0; i < points.Length; i++)
			{
				points[i] = this.ReadPoint();
			}

			return new PgPath(isClosedPath, points);
		}

		#endregion

		#region Common Methods

		public object ReadValue(PgType type, int length)
		{
			switch (type.DataType)
			{
				case PgDataType.Array:
					return this.ReadArray(type, length);

				case PgDataType.Vector:
					return this.ReadVector(type, length);
					
				case PgDataType.Binary:
					return this.ReadBytes(length);

				case PgDataType.Char:
				case PgDataType.VarChar:
					return this.ReadString(length);

				case PgDataType.Boolean:
					return this.ReadBoolean();

				case PgDataType.Byte:
					return this.ReadByte();
					
				case PgDataType.Decimal:
					return Decimal.Parse(
						this.ReadString(length), 
						NumberFormatInfo.InvariantInfo);

				case PgDataType.Currency:
					return this.ReadCurrency();

				case PgDataType.Float:
					return this.ReadSingle();					

				case PgDataType.Double:
					return this.ReadDouble();					

				case PgDataType.Int2:
					return this.ReadShort();					

				case PgDataType.Int4:
					return this.ReadInt();					

				case PgDataType.Int8:
					return this.ReadLong();

				case PgDataType.Interval:
					return this.ReadInterval();

				case PgDataType.Date:
					return this.ReadDate();					

				case PgDataType.Time:
					return this.ReadTime();

				case PgDataType.TimeWithTZ:
					return this.ReadTimeWithTZ();					

				case PgDataType.Timestamp:
					return this.ReadTimestamp();

				case PgDataType.TimestampWithTZ:
					return this.ReadTimestampWithTZ();

				case PgDataType.Point:
					return this.ReadPoint();

				case PgDataType.Circle:
					return this.ReadCircle();

				case PgDataType.Line:
					return this.ReadLine();

				case PgDataType.LSeg:
					return this.ReadLSeg();
				
				case PgDataType.Box:
					return this.ReadBox();

				case PgDataType.Polygon:
					return this.ReadPolygon();

				case PgDataType.Path:
					return this.ReadPath();

				default:
					return ReadBytes(length);
			}
		}

		public object ReadValueFromString(PgType type, int length)
		{
			string stringValue = this.ReadString(length);

			switch (type.DataType)
			{
				case PgDataType.Array:
					return null;

				case PgDataType.Vector:
					return null;
					
				case PgDataType.Binary:
					return null;

				case PgDataType.Char:
				case PgDataType.VarChar:
					return stringValue;

				case PgDataType.Boolean:
					switch (stringValue.ToLower())
					{
						case "t":
						case "true":
						case "y":
						case "yes":
						case "1":
							return true;

						default:
							return false;
					}

				case PgDataType.Byte:
					return Byte.Parse(stringValue);
					
				case PgDataType.Decimal:
					return Decimal.Parse(
						stringValue, 
						NumberFormatInfo.InvariantInfo);

				case PgDataType.Currency:
				case PgDataType.Float:
					return Single.Parse(
						stringValue, 
						NumberFormatInfo.InvariantInfo);
				
				case PgDataType.Double:
					return Double.Parse(
						stringValue, 
						NumberFormatInfo.InvariantInfo);

				case PgDataType.Int2:
					return Int16.Parse(
						stringValue,
						NumberFormatInfo.InvariantInfo);

				case PgDataType.Int4:
					return Int32.Parse(
						stringValue,
						NumberFormatInfo.InvariantInfo);

				case PgDataType.Int8:
					return Int64.Parse(
						stringValue,
						NumberFormatInfo.InvariantInfo);

				case PgDataType.Interval:
					return null;
				
				case PgDataType.Date:				
				case PgDataType.Timestamp:
				case PgDataType.Time:
				case PgDataType.TimeWithTZ:
				case PgDataType.TimestampWithTZ:			
					return DateTime.Parse(stringValue);

				case PgDataType.Point:
					return PgPoint.Parse(stringValue);

				case PgDataType.Circle:
					return PgCircle.Parse(stringValue);

				case PgDataType.Line:
					return PgLine.Parse(stringValue);

				case PgDataType.LSeg:
					return PgLSeg.Parse(stringValue);
				
				case PgDataType.Box:
					return PgBox.Parse(stringValue);

				case PgDataType.Polygon:
					return PgPolygon.Parse(stringValue);

				case PgDataType.Path:
					return PgPath.Parse(stringValue);

				default:
					return ReadBytes(length);
			}
		}

		#endregion

		#region Array Handling Methods

		private Array readPrimitiveArray(PgType elementType, int length, 
			int dimensions, int flags, int[] lengths, int[] lowerBounds)
		{
			Array data = Array.CreateInstance(elementType.SystemType, lengths, lowerBounds);

			// Read array data
			byte[] sourceArray = decodeArrayData(elementType, data.Length, length);
			
			Buffer.BlockCopy(sourceArray, 0, data, 0, sourceArray.Length);

			return data;
		}

		private Array readNonPrimitiveArray(PgType elementType, int length, 
			int dimensions, int flags, int[] lengths, int[] lowerBounds)
		{
			Array data = Array.CreateInstance(elementType.SystemType, lengths, lowerBounds);

			for (int i = data.GetLowerBound(0); i <= data.GetUpperBound(0); i++)
			{
				int	elementLen = this.ReadInt();
				data.SetValue(this.ReadValue(elementType, elementType.Size), i);
			}

			return data;
		}

		private Array readStringArray(PgType type, int length)
		{
			PgType	elementType = PgDbClient.Types[type.ElementType];
			Array	data		= null;

			string contents = ReadString(length);
			contents = contents.Substring(1, contents.Length - 2);

			string[] elements = contents.Split(',');

			data = Array.CreateInstance(elementType.SystemType, elements.Length);

			for (int i = 0; i < elements.Length; i++)
			{
				data.SetValue(elements[i], i);
			}

			return data;
		}

		private byte[] decodeArrayData(PgType type, int elementCount, int length)
		{
			byte[] data = new byte[length];

			int element = 0;
			int index	= 0;
			while (element < elementCount)
			{
				byte[] elementData = null;
				int elementLen = this.ReadInt();

				switch (type.DataType)
				{
					case PgDataType.Boolean:
						elementData = BitConverter.GetBytes(this.ReadBoolean());
						break;
					
					case PgDataType.Float:
						elementData = BitConverter.GetBytes(this.ReadSingle());
						break;

					case PgDataType.Double:
						elementData = BitConverter.GetBytes(this.ReadDouble());
						break;

					case PgDataType.Int2:
						elementData = BitConverter.GetBytes(this.ReadShort());
						break;

					case PgDataType.Int4:
						elementData = BitConverter.GetBytes(this.ReadInt());
						break;

					case PgDataType.Int8:
						elementData = BitConverter.GetBytes(this.ReadLong());
						break;
				}

				// Copy element data to dest array
				elementData.CopyTo(data, index);

				// Increment counters
				element++;
				index += elementData.Length;
			}

			byte[] destArray = new byte[index];

			System.Array.Copy(data, 0, destArray, 0, destArray.Length);

			return destArray;
		}

		#endregion
	}
}
