/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 */
package org.apache.causeway.core.metamodel.facets.properties.validating.dflt;

import jakarta.inject.Inject;

import org.apache.causeway.core.metamodel.context.MetaModelContext;
import org.apache.causeway.core.metamodel.facetapi.FeatureType;
import org.apache.causeway.core.metamodel.facets.FacetFactory;
import org.apache.causeway.core.metamodel.facets.FacetFactoryAbstract;
import org.apache.causeway.core.metamodel.facets.properties.validating.PropertyValidateFacet;
import org.apache.causeway.core.metamodel.progmodel.ProgrammingModel;

/**
 * Simply installs a {@link PropertyValidateFacet} onto all properties.
 *
 * <p>
 * The idea is that this {@link FacetFactory} is included early on in the
 * {@link ProgrammingModel}, but other {@link PropertyValidateFacet}
 * implementations will potentially replace these where the property is
 * annotated or otherwise provides a validation mechanism.
 */
public class PropertyValidateFacetDefaultFactory
extends FacetFactoryAbstract {

    @Inject
    public PropertyValidateFacetDefaultFactory(final MetaModelContext mmc) {
        super(mmc, FeatureType.PROPERTIES_ONLY);
    }

    @Override
    public void process(final ProcessMethodContext processMethodContext) {
        var facetHolder = processMethodContext.getFacetHolder();
        addFacet(
                new PropertyValidateFacetDefault(facetHolder));
    }

    @Override
    public void processParams(final ProcessParameterContext processParameterContext) {
        var facetHolder = processParameterContext.getFacetHolder();
        addFacet(
                new PropertyValidateFacetDefault(facetHolder));
    }

}
