/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.juneau.rest.client;

import static org.apache.juneau.TestUtils.*;
import static org.apache.juneau.commons.utils.CollectionUtils.*;
import static org.junit.jupiter.api.Assertions.*;

import java.io.*;
import java.util.*;

import org.apache.juneau.*;
import org.apache.juneau.MediaType;
import org.apache.juneau.collections.*;
import org.apache.juneau.commons.reflect.*;
import org.apache.juneau.json.*;
import org.apache.juneau.parser.*;
import org.apache.juneau.rest.annotation.*;
import org.apache.juneau.rest.httppart.*;
import org.apache.juneau.rest.mock.*;
import org.apache.juneau.rest.servlet.*;
import org.apache.juneau.swap.*;
import org.junit.jupiter.api.*;

class RestClient_Config_BeanContext_Test extends TestBase {

	@Rest
	public static class A extends BasicRestObject {
		@RestPost
		public Reader echoBody(org.apache.juneau.rest.RestRequest req) throws IOException {
			return req.getContent().getReader();
		}
		@RestGet
		public String[] checkHeader(org.apache.juneau.rest.RestRequest req) {
			return req.getHeaders().getAll(req.getHeaderParam("Check").orElse(null)).stream().map(RequestHeader::getValue).toArray(String[]::new);
		}
		@RestGet
		public Reader checkQuery(org.apache.juneau.rest.RestRequest req) {
			return reader(req.getQueryParams().asQueryString());
		}
		@RestPost
		public Reader checkFormData(org.apache.juneau.rest.RestRequest req) {
			return reader(req.getFormParams().asQueryString());
		}
	}

	protected static class A1 {
		public int f = 1;
		@Override
		public String toString() {
			return "O1";
		}
	}

	@Test void a01_beanClassVisibility() throws Exception {
		var x1 = client().build();
		var x2 = client(A.class).beanClassVisibility(Visibility.PROTECTED).build();
		x1.post("/echoBody",new A1()).run().assertContent("'O1'");
		x2.post("/echoBody",new A1()).run().assertContent("{f:1}");
		x1.get("/checkQuery").queryData("foo",new A1()).run().assertContent("foo=O1");
		x2.get("/checkQuery").queryData("foo",new A1()).run().assertContent("foo=f%3D1").assertContent().asString().asUrlDecode().is("foo=f=1");
		x1.formPost("/checkFormData").formData("foo",new A1()).run().assertContent("foo=O1");
		x2.formPost("/checkFormData").formData("foo",new A1()).run().assertContent("foo=f%3D1").assertContent().asString().asUrlDecode().is("foo=f=1");
		x1.get("/checkHeader").header("foo",new A1()).header("Check","foo").run().assertContent("['O1']");
		x2.get("/checkHeader").header("foo",new A1()).header("Check","foo").run().assertContent("['f=1']");
	}

	public static class A2a {
		private int f;
		protected A2a(int f) {
			this.f = f;
		}
		public int toInt() {
			return f;
		}
	}

	@Rest
	public static class A2b extends BasicRestObject {
		@RestPost
		public Reader test(org.apache.juneau.rest.RestRequest req,org.apache.juneau.rest.RestResponse res) throws IOException {
			res.setHeader("X",req.getHeaderParam("X").orElse(null));
			return req.getContent().getReader();
		}
	}

	@Test void a02_beanConstructorVisibility() throws Exception {
		var x = client(A2b.class)
			.beanConstructorVisibility(Visibility.PROTECTED)
			.build()
			.post("/test",new A2a(1))
			.header("X",new A2a(1))
			.run()
			.cacheContent()
			.assertContent("1")
			.assertHeader("X").is("1");
		assertEquals(1,x.getContent().as(A2a.class).f);
		assertEquals(1,x.getHeader("X").as(A2a.class).get().f);
	}

	public static class A3 {
		public int f1;
		protected int f2;
		static A3 get() {
			var x = new A3();
			x.f1 = 1;
			x.f2 = 2;
			return x;
		}
		@Override
		public String toString() {
			return f1 + "/" + f2;
		}
	}

	@Test void a03_beanFieldVisibility() throws Exception {
		var x = client(A2b.class)
			.beanFieldVisibility(Visibility.PROTECTED)
			.build()
			.post("/test",A3.get())
			.header("X",A3.get())
			.run()
			.cacheContent()
			.assertContent("{f1:1,f2:2}")
			.assertHeader("X").is("f1=1,f2=2");
		assertEquals(2,x.getContent().as(A3.class).f2);
		assertEquals(2,x.getHeader("X").as(A3.class).get().f2);
	}

	public interface A4a {
		int getF3();
		void setF3(int f3);
	}

	public static class A4b implements A4a {
		public int f1, f2;

		private int f3;
		@Override public int getF3() { return f3; }
		@Override public void setF3(int v) { f3 = v; }

		static A4b get() {
			var x = new A4b();
			x.f1 = 1;
			x.f2 = 2;
			x.f3 = 3;
			return x;
		}
		@Override
		public String toString() {
			return f1 + "/" + f2;
		}
	}

	@Test void a04_beanFilters() throws Exception {
		var x = client(A2b.class)
			.beanProperties(A4b.class,"f1")
			.build()
			.post("/test",A4b.get())
			.header("X",A4b.get())
			.run()
			.cacheContent()
			.assertContent()
			.is("{f1:1}")
			.assertHeader("X").is("f1=1");
		assertEquals(0,x.getContent().as(A4b.class).f2);
		assertEquals(0,x.getHeader("X").as(A4b.class).get().f2);

		x = client(A2b.class)
			.beanProperties(A4b.class,"f1")
			.build()
			.post("/test",A4b.get())
			.header("X",A4b.get())
			.run()
			.cacheContent()
			.assertContent("{f1:1}")
			.assertHeader("X").is("f1=1");
		assertEquals(0,x.getContent().as(A4b.class).f2);
		assertEquals(0,x.getHeader("X").as(A4b.class).get().f2);

		x = client(A2b.class)
			.beanProperties(A4b.class,"f1")
			.build()
			.post("/test",A4b.get())
			.header("X",A4b.get())
			.run()
			.cacheContent()
			.assertContent("{f1:1}")
			.assertHeader("X").is("f1=1");
		assertEquals(0,x.getContent().as(A4b.class).f2);
		assertEquals(0,x.getHeader("X").as(A4b.class).get().f2);

		x = client(A2b.class)
			.beanProperties(A4b.class,"f1")
			.build()
			.post("/test",A4b.get())
			.header("X",A4b.get())
			.run()
			.cacheContent()
			.assertContent("{f1:1}")
			.assertHeader("X").is("f1=1");
		assertEquals(0,x.getContent().as(A4b.class).f2);
		assertEquals(0,x.getHeader("X").as(A4b.class).get().f2);

		x = client(A2b.class)
			.interfaces(A4a.class)
			.build()
			.post("/test",A4b.get())
			.header("X",A4b.get())
			.run()
			.cacheContent()
			.assertContent("{f3:3}")
			.assertHeader("X").is("f3=3");
		assertEquals(3,x.getContent().as(A4b.class).f3);
		assertEquals(3,x.getHeader("X").as(A4b.class).get().f3);
	}

	public static class A5  {
		private int f1;
		public int getF1() { return f1; }
		public void setF1(int v) { f1 = v; }

		private int f2;
		protected int getF2() { return f2; }
		protected void setF2(int v) { f2 = v; }

		static A5 get() {
			var x = new A5();
			x.f1 = 1;
			x.f2 = 2;
			return x;
		}

		@Override
		public String toString() {
			return f1 + "/" + f2;
		}
	}

	@Test void a05_beanMethodVisibility() throws Exception {
		var x = client(A2b.class)
			.beanMethodVisibility(Visibility.PROTECTED)
			.build()
			.post("/test",A5.get())
			.header("X",A5.get())
			.run()
			.cacheContent()
			.assertContent("{f1:1,f2:2}")
			.assertHeader("X").is("f1=1,f2=2");
		assertEquals(2,x.getContent().as(A5.class).f2);
		assertEquals(2,x.getHeader("X").as(A5.class).get().f2);
	}

	public static class A6 {}

	@Test void a06_disableBeansRequireSomeProperties() throws Exception {
		client().disableBeansRequireSomeProperties().build().post("/echoBody",new A6()).run().assertContent("{}");
	}

	public static class A7  {
		public String f1;
		public A7(String i) {
			f1 = i;
		}
		@Override
		public String toString() {
			return f1;
		}
	}

	@Test void a07_beansRequireDefaultConstructor() throws Exception {
		client(A2b.class)
			.build()
			.post("/test",new A7("1"))
			.header("X",new A7("1"))
			.run()
			.assertContent("{f1:'1'}")
			.assertHeader("X").is("f1=1");
		client(A2b.class)
			.beansRequireDefaultConstructor()
			.build()
			.post("/test",new A7("1"))
			.header("X",new A7("1"))
			.run()
			.assertContent("'1'")
			.assertHeader("X").is("1");
	}

	@Test void a08_beansRequireSerializable() throws Exception {
		client(A2b.class)
			.build()
			.post("/test",new A7("1"))
			.header("X",new A7("1"))
			.run()
			.assertContent("{f1:'1'}")
			.assertHeader("X").is("f1=1");
		client(A2b.class)
			.beansRequireSerializable()
			.build()
			.post("/test",new A7("1"))
			.header("X",new A7("1"))
			.run()
			.assertContent("'1'")
			.assertHeader("X").is("1");
	}

	public static class A9 {
		private int f1;
		public int getF1() { return f1; }
		public void setF1(int v) { f1 = v; }

		private int f2;
		public int getF2() { return f2; }

		static A9 get() {
			var x = new A9();
			x.f1 = 1;
			x.f2 = 2;
			return x;
		}

		@Override
		public String toString() {
			return f1 + "/" + f2;
		}
	}

	@Test void a09_beansRequireSettersForGetters() throws Exception {
		client(A2b.class)
			.build()
			.post("/test",A9.get())
			.header("X",A9.get())
			.run()
			.assertContent("{f1:1,f2:2}")
			.assertHeader("X").is("f1=1,f2=2");
		client(A2b.class)
			.beansRequireSettersForGetters()
			.build()
			.post("/test",A9.get())
			.header("X",A9.get())
			.run()
			.assertContent("{f1:1}")
			.assertHeader("X").is("f1=1");
	}

	@Test void a10_bpi() throws Exception {
		client(A2b.class)
			.beanProperties(JsonMap.of("A9","f2"))
			.build()
			.post("/test",A9.get())
			.header("X",A9.get())
			.run()
			.assertContent("{f2:2}")
			.assertHeader("X").is("f2=2");
		client(A2b.class)
			.beanProperties(A9.class,"f2")
			.build()
			.post("/test",A9.get())
			.header("X",A9.get())
			.run()
			.assertContent("{f2:2}")
			.assertHeader("X").is("f2=2");
		client(A2b.class)
			.beanProperties("A9","f2")
			.build()
			.post("/test",A9.get())
			.header("X",A9.get())
			.run()
			.assertContent("{f2:2}")
			.assertHeader("X").is("f2=2");
		client(A2b.class)
			.beanProperties(A9.class.getName(),"f2")
			.build()
			.post("/test",A9.get())
			.header("X",A9.get())
			.run()
			.assertContent("{f2:2}")
			.assertHeader("X").is("f2=2");
	}

	@Test void a11_bpro() throws Exception {
		var x = client(A2b.class)
			.beanPropertiesReadOnly(JsonMap.of("09","f2"))
			.build()
			.post("/test",A9.get())
			.header("X",A9.get())
			.run()
			.cacheContent()
			.assertContent("{f1:1,f2:2}")
			.assertHeader("X").is("f1=1,f2=2");
		assertEquals("1/0",x.getContent().as(A9.class).toString());
		assertEquals("1/0",x.getHeader("X").as(A9.class).get().toString());

		x = client(A2b.class)
			.beanPropertiesReadOnly(A9.class,"f2")
			.build()
			.post("/test",A9.get())
			.header("X",A9.get())
			.run()
			.cacheContent()
			.assertContent("{f1:1,f2:2}")
			.assertHeader("X").is("f1=1,f2=2");
		assertEquals("1/0",x.getContent().as(A9.class).toString());
		assertEquals("1/0",x.getHeader("X").as(A9.class).get().toString());

		x = client(A2b.class)
			.beanPropertiesReadOnly("O9","f2")
			.build()
			.post("/test",A9.get())
			.header("X",A9.get())
			.run()
			.cacheContent()
			.assertContent("{f1:1,f2:2}")
			.assertHeader("X").is("f1=1,f2=2");
		assertEquals("1/0",x.getContent().as(A9.class).toString());
		assertEquals("1/0",x.getHeader("X").as(A9.class).get().toString());
	}

	@Test void a12_bpwo() throws Exception {
		var x = client(A2b.class)
			.beanPropertiesWriteOnly(JsonMap.of("A9","f2"))
			.build()
			.post("/test",A9.get())
			.header("X",A9.get())
			.run()
			.cacheContent()
			.assertContent("{f1:1}")
			.assertHeader("X").is("f1=1");
		assertEquals("1/0",x.getContent().as(A9.class).toString());
		assertEquals("1/0",x.getHeader("X").as(A9.class).get().toString());

		x = client(A2b.class)
			.beanPropertiesWriteOnly(A9.class,"f2")
			.build()
			.post("/test",A9.get())
			.header("X",A9.get())
			.run()
			.cacheContent()
			.assertContent("{f1:1}")
			.assertHeader("X").is("f1=1");
		assertEquals("1/0",x.getContent().as(A9.class).toString());
		assertEquals("1/0",x.getHeader("X").as(A9.class).get().toString());

		x = client(A2b.class)
			.beanPropertiesWriteOnly("A9","f2")
			.build()
			.post("/test",A9.get())
			.header("X",A9.get())
			.run()
			.cacheContent()
			.assertContent("{f1:1}")
			.assertHeader("X").is("f1=1");
		assertEquals("1/0",x.getContent().as(A9.class).toString());
		assertEquals("1/0",x.getHeader("X").as(A9.class).get().toString());
	}

	@Test void a13_bpx() throws Exception {
		client(A2b.class)
			.beanPropertiesExcludes(JsonMap.of("A9","f1"))
			.build()
			.post("/test",A9.get())
			.header("X",A9.get()).
			run()
			.assertContent("{f2:2}")
			.assertHeader("X").is("f2=2");
		client(A2b.class)
			.beanPropertiesExcludes(A9.class,"f1")
			.build()
			.post("/test",A9.get())
			.header("X",A9.get())
			.run()
			.assertContent("{f2:2}")
			.assertHeader("X").is("f2=2");
		client(A2b.class)
			.beanPropertiesExcludes("A9","f1")
			.build()
			.post("/test",A9.get())
			.header("X",A9.get())
			.run()
			.assertContent("{f2:2}")
			.assertHeader("X").is("f2=2");
		client(A2b.class)
			.beanPropertiesExcludes(A9.class.getName(),"f1")
			.build()
			.post("/test",A9.get())
			.header("X",A9.get())
			.run()
			.assertContent("{f2:2}")
			.assertHeader("X").is("f2=2");
	}

	public static class A14 {
		public Object f;
	}

	@Test void a14_debug() {
		var x = new A14();
		x.f = x;
		assertThrowsWithMessage(Exception.class, "Recursion occurred", ()->client().debug().build().post("/echo",x).run());
	}

	@org.apache.juneau.annotation.Bean(typeName="foo")
	public static class A15a {
		public String foo;
		static A15a get() {
			var x = new A15a();
			x.foo = "1";
			return x;
		}
	}

	@org.apache.juneau.annotation.Bean(typeName="bar")
	public static class A15b {
		public String foo;
		static A15b get() {
			var x = new A15b();
			x.foo = "2";
			return x;
		}
	}

	public static class A15c {
		public Object foo;
		static A15c get() {
			var x = new A15c();
			x.foo = A15a.get();
			return x;
		}
	}

	@Test void a15_dictionary() throws Exception {
		var o = client().beanDictionary(A15a.class,A15b.class).addRootType().addBeanTypes().build().post("/echoBody",A15a.get()).run().cacheContent().assertContent().isContains("{_type:'foo',foo:'1'}").getContent().as(Object.class);
		assertTrue(o instanceof A15a);

		var m = JsonMap.of("x",A15a.get(),"y",A15b.get());
		m = client().beanDictionary(A15a.class,A15b.class).addRootType().addBeanTypes().build().post("/echoBody",m).run().cacheContent().assertContent("{x:{_type:'foo',foo:'1'},y:{_type:'bar',foo:'2'}}").getContent().as(JsonMap.class);
		assertTrue(m.get("x") instanceof A15a);
		assertTrue(m.get("y") instanceof A15b);

		var x = client().dictionaryOn(A15c.class,A15a.class,A15b.class).addRootType().addBeanTypes().build().post("/echoBody",A15c.get()).run().cacheContent().assertContent("{foo:{_type:'foo',foo:'1'}}").getContent().as(A15c.class);
		assertTrue(x.foo instanceof A15a);
	}

	public static class A16 {
		private String foo;
		public String getFoo() { return foo; }

		static A16 get() {
			var x = new A16();
			x.foo = "foo";
			return x;
		}
	}

	@Test void a16_disableIgnorePropertiesWithoutSetters() throws Exception {
		var x = client().build().post("/echoBody",A16.get()).run().cacheContent().assertContent().isContains("{foo:'foo'}").getContent().as(A16.class);
		assertNull(x.foo);
		assertThrowsWithMessage(Exception.class, "Setter or public field not defined", ()->client().disableIgnoreMissingSetters().build().post("/echoBody",A16.get()).run().cacheContent().assertContent().isContains("{foo:'foo'}").getContent().as(A16.class));
	}

	public static class A17 {
		public String foo;
		public transient String bar;
		static A17 get() {
			var x = new A17();
			x.foo = "1";
			x.bar = "2";
			return x;
		}
	}

	@Test void a17_disableIgnoreTransientFields() throws Exception {
		var x = client().build().post("/echoBody",A17.get()).run().cacheContent().assertContent().isContains("{foo:'1'}").getContent().as(A17.class);
		assertNull(x.bar);
		x = client().disableIgnoreTransientFields().build().post("/echoBody",A17.get()).run().cacheContent().assertContent().isContains("{bar:'2',foo:'1'}").getContent().as(A17.class);
		assertEquals("2",x.bar);
	}

	public static class A18 {
		public String foo;
	}

	@Test void a18_disableIgnoreUnknownNullBeanProperties() throws Exception {
		client().build().post("/echoBody",reader("{foo:'1',bar:null}")).run().cacheContent().assertContent().isContains("{foo:'1',bar:null}").getContent().as(A18.class);
		assertThrowsWithMessage(Exception.class, "Unknown property 'bar'", ()->client().disableIgnoreUnknownNullBeanProperties().build().post("/echoBody",reader("{foo:'1',bar:null}")).run().cacheContent().assertContent().isContains("{foo:'1',bar:null}").getContent().as(A18.class));
	}

	public interface A19 {
		String getFoo();
		void setFoo(String foo);
	}

	@Test void a19_disableInterfaceProxies() throws Exception {
		var x = client().build().post("/echoBody",reader("{foo:'1'}")).run().cacheContent().assertContent().isContains("{foo:'1'}").getContent().as(A19.class);
		assertEquals("1",x.getFoo());
		assertThrowsWithMessage(Exception.class, "could not be instantiated", ()->client().disableInterfaceProxies().build().post("/echoBody",reader("{foo:'1'}")).run().cacheContent().assertContent().isContains("{foo:'1'}").getContent().as(A19.class));
	}

	public static class A20 {
		private String foo;
		public String getFoo() { return foo; }

		public A20 foo(String foo) {
			this.foo = foo;
			return this;
		}
	}

	@Test void a20_fluentSetters() throws Exception {
		var x = client().findFluentSetters().build().post("/echoBody",reader("{foo:'1'}")).run().cacheContent().assertContent().isContains("{foo:'1'}").getContent().as(A20.class);
		assertEquals("1",x.getFoo());
		x = client().findFluentSetters(A20.class).build().post("/echoBody",reader("{foo:'1'}")).run().cacheContent().assertContent().isContains("{foo:'1'}").getContent().as(A20.class);
		assertEquals("1",x.getFoo());
	}

	public static class A21 {
		private String foo;
		public String getFoo() { return foo; }
		public void setFoo(String v) { foo = v; }

		@SuppressWarnings("unused") private String bar;
		public String getBar() { throw new RuntimeException("xxx"); }

		static A21 get() {
			var x = new A21();
			x.foo = "1";
			x.bar = "2";
			return x;
		}
	}

	@Test void a21_ignoreInvocationExceptionsOnGetters() throws Exception {
		assertThrowsWithMessage(Exception.class, "xxx", ()->client().build().post("/echoBody",A21.get()).run());
		var x = client().ignoreInvocationExceptionsOnGetters().build().post("/echoBody",A21.get()).run().cacheContent().assertContent().isContains("{foo:'1'}").getContent().as(A21.class);
		assertEquals("1",x.getFoo());
	}

	public static class A22 {
		private String foo;
		public String getFoo() { return foo; }
		public void setFoo(String v) { foo = v; }

		private String bar;
		public String getBar() { return bar; }
		public void setBar(String v) { throw new RuntimeException("xxx"); }

		static A22 get() {
			var x = new A22();
			x.foo = "1";
			x.bar = "2";
			return x;
		}
	}

	@Test void a22_ignoreInvocationExceptionsOnSetters() throws Exception {
		assertThrowsWithMessage(Exception.class, "Error occurred trying to set property 'bar'", ()->client().build().post("/echoBody",A22.get()).run().getContent().as(A22.class));
		var x = client().ignoreInvocationExceptionsOnSetters().build().post("/echoBody",A22.get()).run().cacheContent().getContent().as(A22.class);
		assertEquals("1",x.getFoo());
	}

	public static class A23 {
		public String foo;
	}

	@Test void a23_ignoreUnknownBeanProperties() throws Exception {
		assertThrowsWithMessage(Exception.class, "Unknown property 'bar' encountered", ()->client().build().post("/echoBody",reader("{foo:'1',bar:'2'}")).run().getContent().as(A23.class));
		var x = client().ignoreUnknownBeanProperties().build().post("/echoBody",reader("{foo:'1',bar:'2'}")).run().cacheContent().getContent().as(A23.class);
		assertEquals("1",x.foo);
	}

	public interface A24a {
		void setFoo(int foo);
		int getFoo();
	}

	public static class A24b implements A24a {
		private int foo;
		@Override public int getFoo() { return foo; }
		@Override public void setFoo(int v) { foo = v; }
	}

	@Test void a24_implClass() throws Exception {
		var x = client().implClass(A24a.class,A24b.class).build().post("/echoBody",reader("{foo:1}")).run().getContent().as(A24a.class);
		assertEquals(1,x.getFoo());
		assertTrue(x instanceof A24b);

		x = client().implClasses(map(A24a.class,A24b.class)).build().post("/echoBody",reader("{foo:1}")).run().getContent().as(A24a.class);
		assertEquals(1,x.getFoo());
		assertTrue(x instanceof A24b);
	}

	public interface A25a {
		void setFoo(int foo);
		int getFoo();
	}

	public static class A25b implements A25a {
		private int foo;
		@Override public int getFoo() { return foo; }
		@Override public void setFoo(int v) { foo = v; }

		private int bar;
		public int getBar() { return bar; }  // Not executed
		public void setBar(int v) { bar = v; }  // Not executed

		static A25b get() {
			var x = new A25b();
			x.foo = 1;
			x.bar = 2;
			return x;
		}
	}

	@Test void a25_interfaceClass() throws Exception {
		var x = client().interfaceClass(A25b.class,A25a.class).build().post("/echoBody",A25b.get()).run().cacheContent().assertContent("{foo:1}").getContent().as(A25b.class);
		assertEquals(1,x.getFoo());
		x = client().interfaces(A25a.class).build().post("/echoBody",A25b.get()).run().assertContent("{foo:1}").getContent().as(A25b.class);
		assertEquals(1,x.getFoo());
	}

	public static class A26 {
		public int foo;
		static A26 get() {
			var x = new A26();
			x.foo = 1;
			return x;
		}
	}

	@Test void a26_locale() throws Exception {
		var x = client().locale(Locale.UK).build().post("/echoBody",A26.get()).run().cacheContent().assertContent("{foo:1}").getContent().as(A26.class);
		assertEquals(1,x.foo);
	}

	@Test void a27_mediaType() throws Exception {
		var x = client().mediaType(MediaType.JSON).build().post("/echoBody",A26.get()).run().cacheContent().assertContent("{foo:1}").getContent().as(A26.class);
		assertEquals(1,x.foo);
	}

	public static class A28 {
		public int foo;
		static A28 get() {
			var x = new A28();
			x.foo = 1;
			return x;
		}
		@Override
		public String toString() {
			return String.valueOf(foo);
		}
		public static A28 fromString(String foo) throws ParseException {
			var x = new A28();
			x.foo = JsonParser.DEFAULT.parse(foo,int.class);
			return x;
		}
	}

	@Test void a28_notBeanClasses() throws Exception {
		var x = client().notBeanClasses(A28.class).build().post("/echoBody",A28.get()).run().cacheContent().assertContent("'1'").getContent().as(A28.class);
		assertEquals(1,x.foo);
	}

	@Test void a29_notBeanPackages() throws Exception {
		var x = client().notBeanPackages(A28.class.getPackage().getName()).build().post("/echoBody",A28.get()).run().cacheContent().assertContent("'1'").getContent().as(A28.class);
		assertEquals(1,x.foo);
	}

	public static class A30a {
		private String foo;
		public String getFoo() { return foo; }
		public void setFoo(String v) { foo = v; }

		static A30a get() {
			var x = new A30a();
			x.foo = "foo";
			return x;
		}
	}

	public static class A30b extends BeanInterceptor<A30a> {
		static boolean getterCalled,setterCalled;
		@Override
		public Object readProperty(A30a bean,String name,Object value) {
			getterCalled = true;
			return "x" + value;
		}
		@Override
		public Object writeProperty(A30a bean,String name,Object value) {
			setterCalled = true;
			return value.toString().substring(1);
		}
	}

	@Test void a30_beanInterceptor() throws Exception {
		var x = client().beanInterceptor(A30a.class,A30b.class).build().post("/echoBody",A30a.get()).run().cacheContent().assertContent("{foo:'xfoo'}").getContent().as(A30a.class);
		assertEquals("foo",x.foo);
		assertTrue(A30b.getterCalled);
		assertTrue(A30b.setterCalled);
	}

	public static class A31 {
		private String fooBar;
		public String getFooBar() { return fooBar; }
		public void setFooBar(String v) { fooBar = v; }

		static A31 get() {
			var x = new A31();
			x.fooBar = "fooBar";
			return x;
		}
	}

	@Test void a31_propertyNamer() throws Exception {
		var x = client().propertyNamer(PropertyNamerDLC.class).build().post("/echoBody",A31.get()).run().cacheContent().assertContent("{'foo-bar':'fooBar'}").getContent().as(A31.class);
		assertEquals("fooBar",x.fooBar);
		x = client().propertyNamer(A31.class,PropertyNamerDLC.class).build().post("/echoBody",A31.get()).run().cacheContent().assertContent("{'foo-bar':'fooBar'}").getContent().as(A31.class);
		assertEquals("fooBar",x.fooBar);
	}

	public static class A32 {
		public int foo, bar, baz;
		static A32 get() {
			var x = new A32();
			x.foo = 1;
			x.bar = 2;
			x.baz = 3;
			return x;
		}
	}

	@Test void a32_sortProperties() throws Exception {
		var x = client().sortProperties().build().post("/echoBody",A32.get()).run().cacheContent().assertContent("{bar:2,baz:3,foo:1}").getContent().as(A32.class);
		assertEquals(1,x.foo);
		x = client().sortProperties(A32.class).build().post("/echoBody",A32.get()).run().cacheContent().assertContent("{bar:2,baz:3,foo:1}").getContent().as(A32.class);
		assertEquals(1,x.foo);
	}

	public static class A33a {
		public int foo;
	}

	public static class A33b extends A33a {
		public int bar;
		static A33b get() {
			var x = new A33b();
			x.foo = 1;
			x.bar = 2;
			return x;
		}
	}

	@Test void a33_stopClass() throws Exception {
		var x = client().stopClass(A33b.class,A33a.class).build().post("/echoBody",A33b.get()).run().cacheContent().assertContent("{bar:2}").getContent().as(A33b.class);
		assertEquals(0,x.foo);
		assertEquals(2,x.bar);
	}

	public static class A34a {
		public int foo;
		static A34a get() {
			var x = new A34a();
			x.foo = 1;
			return x;
		}
	}

	public static class A34b extends ObjectSwap<A34a,Integer> {
		@Override public Integer swap(BeanSession session,A34a o) { return o.foo; }
		@Override public A34a unswap(BeanSession session,Integer f,ClassMeta<?> hint) {return A34a.get(); }
	}

	@Test void a34_swaps() throws Exception {
		var x = client().swaps(A34b.class).build().post("/echoBody",A34a.get()).run().cacheContent().assertContent("1").getContent().as(A34a.class);
		assertEquals(1,x.foo);
	}

	public static class A35 {
		public int foo;
		static A35 get() {
			var x = new A35();
			x.foo = 1;
			return x;
		}
	}

	@Test void a35_timeZone() throws Exception {
		var x = client().timeZone(TimeZone.getTimeZone("Z")).build().post("/echoBody",A35.get()).run().cacheContent().assertContent("{foo:1}").getContent().as(A35.class);
		assertEquals(1,x.foo);
	}

	public static class A36 {
		public int foo;
		static A36 get() {
			var x = new A36();
			x.foo = 1;
			return x;
		}
	}

	@Test void a36_typeName() throws Exception {
		var x = client().typeName(A36.class,"foo").addRootType().build().post("/echoBody",A36.get()).run().cacheContent().assertContent("{_type:'foo',foo:1}").getContent().as(A36.class);
		assertEquals(1,x.foo);
	}

	@Test void a37_typePropertyName() throws Exception {
		var x = client().typeName(A36.class,"foo").typePropertyName("X").addRootType().build().post("/echoBody",A36.get()).run().cacheContent().assertContent("{X:'foo',foo:1}").getContent().as(A36.class);
		assertEquals(1,x.foo);
		x = client().typeName(A36.class,"foo").typePropertyName(A36.class,"X").addRootType().build().post("/echoBody",A36.get()).run().cacheContent().assertContent("{X:'foo',foo:1}").getContent().as(A36.class);
		assertEquals(1,x.foo);
	}

	public enum A38a {
		ONE(1),TWO(2);
		private int value;
		A38a(int value) {
			this.value = value;
		}
		@Override
		public String toString() {
			return String.valueOf(value);  // Not executed
		}
	}

	public static class A38b {
		public A38a foo;
		static A38b get() {
			var x = new A38b();
			x.foo = A38a.ONE;
			return x;
		}
	}

	@Test void a38_useEnumNames() throws Exception {
		var x = client().useEnumNames().build().post("/echoBody",A38b.get()).run().cacheContent().assertContent("{foo:'ONE'}").getContent().as(A38b.class);
		assertEquals(A38a.ONE,x.foo);
	}

	public static class A39 {
		private int foo;
		public int getFoo() { return foo; }
		public void setFoo(int v) { foo = v; }

		public int bar;

		static A39 get() {
			var x = new A39();
			x.foo = 1;
			x.bar = 2;
			return x;
		}
	}

	@Test void a39_useJavaIntrospector() throws Exception {
		var x = client().useJavaBeanIntrospector().build().post("/echoBody",A39.get()).run().cacheContent().assertContent("{foo:1}").getContent().as(A39.class);
		assertEquals(1,x.foo);
	}

	//------------------------------------------------------------------------------------------------------------------
	// Helper methods.
	//------------------------------------------------------------------------------------------------------------------

	private static RestClient.Builder client() {
		return MockRestClient.create(A.class).json5();
	}

	private static RestClient.Builder client(Class<?> c) {
		return MockRestClient.create(c).json5();
	}
}