/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.InetSocketAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketTimeoutException;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import org.apache.nifi.NiFiEntryPoint;
import org.apache.nifi.controller.DecommissionTask;
import org.apache.nifi.controller.status.history.StatusHistoryDump;
import org.apache.nifi.diagnostics.DiagnosticsDump;
import org.apache.nifi.util.LimitingInputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BootstrapListener {
    private static final Logger logger = LoggerFactory.getLogger(BootstrapListener.class);
    private final NiFiEntryPoint nifi;
    private final int bootstrapPort;
    private final String secretKey;
    private volatile Listener listener;
    private volatile ServerSocket serverSocket;
    private volatile boolean nifiLoaded = false;

    public BootstrapListener(NiFiEntryPoint nifi, int bootstrapPort) {
        this.nifi = nifi;
        this.bootstrapPort = bootstrapPort;
        this.secretKey = UUID.randomUUID().toString();
    }

    public void start(int listenPort) throws IOException {
        logger.debug("Starting Bootstrap Listener to communicate with Bootstrap Port {}", (Object)this.bootstrapPort);
        this.serverSocket = new ServerSocket();
        this.serverSocket.bind(new InetSocketAddress("localhost", listenPort));
        this.serverSocket.setSoTimeout(2000);
        int localPort = this.serverSocket.getLocalPort();
        logger.info("Started Bootstrap Listener, Listening for incoming requests on port {}", (Object)localPort);
        this.listener = new Listener(this.serverSocket);
        Thread listenThread = new Thread(this.listener);
        listenThread.setDaemon(true);
        listenThread.setName("Listen to Bootstrap");
        listenThread.start();
        logger.debug("Notifying Bootstrap that local port is {}", (Object)localPort);
        this.sendCommand("PORT", new String[]{String.valueOf(localPort), this.secretKey});
    }

    public void reload() throws IOException {
        if (this.listener != null) {
            this.listener.stop();
        }
        this.sendCommand("RELOAD", new String[0]);
    }

    public void stop() {
        if (this.listener != null) {
            this.listener.stop();
        }
    }

    public void setNiFiLoaded(boolean nifiLoaded) {
        this.nifiLoaded = nifiLoaded;
    }

    public void sendStartedStatus(boolean status) throws IOException {
        logger.debug("Notifying Bootstrap that the status of starting NiFi is {}", (Object)status);
        this.sendCommand("STARTED", new String[]{String.valueOf(status)});
    }

    private void sendCommand(String command, String[] args) throws IOException {
        try (Socket socket = new Socket();){
            socket.setSoTimeout(60000);
            socket.connect(new InetSocketAddress("localhost", this.bootstrapPort));
            socket.setSoTimeout(60000);
            StringBuilder commandBuilder = new StringBuilder(command);
            for (String arg : args) {
                commandBuilder.append(" ").append(arg);
            }
            commandBuilder.append("\n");
            String commandWithArgs = commandBuilder.toString();
            logger.debug("Sending command to Bootstrap: " + commandWithArgs);
            OutputStream out = socket.getOutputStream();
            out.write(commandWithArgs.getBytes(StandardCharsets.UTF_8));
            out.flush();
            logger.debug("Awaiting response from Bootstrap...");
            BufferedReader reader = new BufferedReader(new InputStreamReader(socket.getInputStream()));
            String response = reader.readLine();
            if ("OK".equals(response)) {
                logger.info("Successfully initiated communication with Bootstrap");
            } else {
                logger.error("Failed to communicate with Bootstrap. Bootstrap may be unable to issue or receive commands from NiFi");
            }
        }
    }

    private void writeDump(OutputStream out) throws IOException {
        DiagnosticsDump diagnosticsDump = this.nifi.getServer().getThreadDumpFactory().create(true);
        diagnosticsDump.writeTo(out);
    }

    private void decommission() throws InterruptedException {
        DecommissionTask decommissionTask = this.nifi.getServer().getDecommissionTask();
        if (decommissionTask == null) {
            throw new IllegalArgumentException("This NiFi instance does not support decommissioning");
        }
        decommissionTask.decommission();
    }

    private void writeDiagnostics(OutputStream out, boolean verbose) throws IOException {
        DiagnosticsDump diagnosticsDump = this.nifi.getServer().getDiagnosticsFactory().create(verbose);
        diagnosticsDump.writeTo(out);
    }

    private void writeNodeStatusHistory(OutputStream out, int days) throws IOException {
        StatusHistoryDump statusHistoryDump = this.nifi.getServer().getStatusHistoryDumpFactory().create(days);
        statusHistoryDump.writeTo(out);
    }

    private void sendAnswer(OutputStream out, String answer) throws IOException {
        out.write((answer + "\n").getBytes(StandardCharsets.UTF_8));
        out.flush();
    }

    private BootstrapRequest readRequest(InputStream in) throws IOException {
        LimitingInputStream limitingIn = new LimitingInputStream(in, 4096L);
        BufferedReader reader = new BufferedReader(new InputStreamReader(limitingIn));
        String line = reader.readLine();
        String[] splits = line.split(" ");
        if (splits.length < 1) {
            throw new IOException("Received invalid request from Bootstrap: " + line);
        }
        String requestType = splits[0];
        if (splits.length == 1) {
            throw new IOException("Received invalid request from Bootstrap; request did not have a secret key; request type = " + requestType);
        }
        String[] args = splits.length == 2 ? new String[]{} : Arrays.copyOfRange(splits, 2, splits.length);
        String requestKey = splits[1];
        if (!this.secretKey.equals(requestKey)) {
            throw new IOException("Received invalid Secret Key for request type " + requestType);
        }
        try {
            return new BootstrapRequest(requestType, args);
        }
        catch (Exception e) {
            throw new IOException("Received invalid request from Bootstrap; request type = " + requestType);
        }
    }

    private static class BootstrapRequest {
        private final RequestType requestType;
        private final String[] args;

        public BootstrapRequest(String request, String[] args) {
            this.requestType = RequestType.valueOf(request);
            this.args = args;
        }

        public RequestType getRequestType() {
            return this.requestType;
        }

        public String[] getArgs() {
            return this.args;
        }

        public static enum RequestType {
            RELOAD,
            SHUTDOWN,
            DUMP,
            DIAGNOSTICS,
            DECOMMISSION,
            PING,
            IS_LOADED,
            STATUS_HISTORY;

        }
    }

    private class Listener
    implements Runnable {
        private final ServerSocket serverSocket;
        private final ExecutorService executor;
        private volatile boolean stopped = false;

        public Listener(ServerSocket serverSocket) {
            this.serverSocket = serverSocket;
            this.executor = Executors.newFixedThreadPool(2);
        }

        public void stop() {
            this.stopped = true;
            this.executor.shutdownNow();
            try {
                this.serverSocket.close();
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }

        @Override
        public void run() {
            while (!this.stopped) {
                try {
                    Socket socket;
                    try {
                        logger.debug("Listening for Bootstrap Requests");
                        socket = this.serverSocket.accept();
                    }
                    catch (SocketTimeoutException ste) {
                        if (!this.stopped) continue;
                        return;
                    }
                    catch (IOException ioe) {
                        if (this.stopped) {
                            return;
                        }
                        throw ioe;
                    }
                    logger.debug("Received connection from Bootstrap");
                    socket.setSoTimeout(5000);
                    this.executor.submit(new Runnable(){

                        /*
                         * WARNING - Removed try catching itself - possible behaviour change.
                         * Enabled aggressive block sorting
                         * Enabled unnecessary exception pruning
                         * Enabled aggressive exception aggregation
                         */
                        @Override
                        public void run() {
                            try {
                                BootstrapRequest request = BootstrapListener.this.readRequest(socket.getInputStream());
                                BootstrapRequest.RequestType requestType = request.getRequestType();
                                switch (requestType) {
                                    case PING: {
                                        logger.debug("Received PING request from Bootstrap; responding");
                                        BootstrapListener.this.sendAnswer(socket.getOutputStream(), "PING");
                                        logger.debug("Responded to PING request from Bootstrap");
                                        return;
                                    }
                                    case RELOAD: {
                                        logger.info("Received RELOAD request from Bootstrap");
                                        BootstrapListener.this.sendAnswer(socket.getOutputStream(), "RELOAD");
                                        BootstrapListener.this.nifi.shutdownHook(true);
                                        return;
                                    }
                                    case SHUTDOWN: {
                                        logger.info("Received SHUTDOWN request from Bootstrap");
                                        BootstrapListener.this.sendAnswer(socket.getOutputStream(), "SHUTDOWN");
                                        socket.close();
                                        BootstrapListener.this.nifi.shutdownHook(false);
                                        return;
                                    }
                                    case DUMP: {
                                        logger.info("Received DUMP request from Bootstrap");
                                        BootstrapListener.this.writeDump(socket.getOutputStream());
                                        return;
                                    }
                                    case DECOMMISSION: {
                                        logger.info("Received DECOMMISSION request from Bootstrap");
                                        try {
                                            BootstrapListener.this.decommission();
                                            BootstrapListener.this.sendAnswer(socket.getOutputStream(), "DECOMMISSION");
                                            BootstrapListener.this.nifi.shutdownHook(false);
                                            return;
                                        }
                                        catch (Exception e) {
                                            OutputStream out = socket.getOutputStream();
                                            out.write(("Failed to decommission node: " + e + "; see app-log for additional details").getBytes(StandardCharsets.UTF_8));
                                            out.flush();
                                            return;
                                        }
                                        finally {
                                            socket.close();
                                        }
                                    }
                                    case DIAGNOSTICS: {
                                        logger.info("Received DIAGNOSTICS request from Bootstrap");
                                        String[] args = request.getArgs();
                                        boolean verbose = false;
                                        if (args == null) {
                                            verbose = false;
                                        } else {
                                            for (String arg : args) {
                                                if (!"--verbose=true".equalsIgnoreCase(arg)) continue;
                                                verbose = true;
                                                break;
                                            }
                                        }
                                        BootstrapListener.this.writeDiagnostics(socket.getOutputStream(), verbose);
                                        return;
                                    }
                                    case STATUS_HISTORY: {
                                        logger.info("Received STATUS_HISTORY request from Bootstrap");
                                        String[] statusHistoryArgs = request.getArgs();
                                        int days = Integer.parseInt(statusHistoryArgs[0]);
                                        BootstrapListener.this.writeNodeStatusHistory(socket.getOutputStream(), days);
                                        return;
                                    }
                                    case IS_LOADED: {
                                        logger.debug("Received IS_LOADED request from Bootstrap");
                                        String answer = String.valueOf(BootstrapListener.this.nifiLoaded);
                                        BootstrapListener.this.sendAnswer(socket.getOutputStream(), answer);
                                        logger.debug("Responded to IS_LOADED request from Bootstrap with value: " + answer);
                                        return;
                                    }
                                }
                                return;
                            }
                            catch (Throwable t) {
                                logger.error("Failed to process request from Bootstrap due to " + t.toString(), t);
                                return;
                            }
                            finally {
                                try {
                                    socket.close();
                                }
                                catch (IOException ioe) {
                                    logger.warn("Failed to close socket to Bootstrap due to {}", (Object)ioe.toString());
                                }
                            }
                        }
                    });
                }
                catch (Throwable t) {
                    logger.error("Failed to process request from Bootstrap due to " + t.toString(), t);
                }
            }
        }
    }
}

