package diagnostics;

=head1 NAME

diagnostics - Perl compiler pragma to force verbose warning diagnostics

splain - standalone program to do the same thing

=head1 SYNOPSIS

As a pragma:

    use diagnostics;
    use diagnostics -verbose;

    enable  diagnostics;
    disable diagnostics;

Aa a program:

    perl program 2>diag.out
    splain [-v] [-p] diag.out


=head1 DESCRIPTION

=head2 The C<diagnostics> Pragma

This module extends the terse diagnostics normally emitted by both the
perl compiler and the perl interpeter, augmenting them with the more
explicative and endearing descriptions found in L<perldiag>.  Like the
other pragmata, it affects the compilation phase of your program rather
than merely the execution phase.

To use in your program as a pragma, merely invoke

    use diagnostics;

at the start (or near the start) of your program.  (Note 
that this I<does> enable perl's B<-w> flag.)  Your whole
compilation will then be subject(ed :-) to the enhanced diagnostics.
These still go out B<STDERR>.

Due to the interaction between runtime and compiletime issues,
and because it's probably not a very good idea anyway,
you may not use C<no diagnostics> to turn them off at compiletime.
However, you may control there behaviour at runtime using the 
disable() and enable() methods to turn them off and on respectively.

The B<-verbose> flag first prints out the L<perldiag> introduction before
any other diagnostics.  The $diagnostics::PRETTY variable can generate nicer
escape sequences for pagers.

=head2 The I<splain> Program

While apparently a whole nuther program, I<splain> is actually nothing
more than a link to the (executable) F<diagnostics.pm> module, as well as
a link to the F<diagnostics.pod> documentation.  The B<-v> flag is like
the C<use diagnostics -verbose> directive.
The B<-p> flag is like the
$diagnostics::PRETTY variable.  Since you're post-processing with 
I<splain>, there's no sense in being able to enable() or disable() processing.

Output from I<splain> is directed to B<STDOUT>, unlike the pragma.

=head1 EXAMPLES

The following file is certain to trigger a few errors at both
runtime and compiletime:

    use diagnostics;
    print NOWHERE "nothing\n";
    print STDERR "\n\tThis message should be unadorned.\n";
    warn "\tThis is a user warning";
    print "\nDIAGNOSTIC TESTER: Please enter a <CR> here: ";
    my $a, $b = scalar <STDIN>;
    print "\n";
    print $x/$y;

If you prefer to run your program first and look at its problem
afterwards, do this:

    perl -w test.pl 2>test.out
    ./splain < test.out

Note that this is not in general possible in shells of more dubious heritage, 
as the theoretical 

    (perl -w test.pl >/dev/tty) >& test.out
    ./splain < test.out

Because you just moved the existing B<stdout> to somewhere else.

If you don't want to modify your source code, but still have on-the-fly
warnings, do this:

    exec 3>&1; perl -w test.pl 2>&1 1>&3 3>&- | splain 1>&2 3>&- 

Nifty, eh?

If you want to control warnings on the fly, do something like this.
Make sure you do the C<use> first, or you won't be able to get
at the enable() or disable() methods.

    use diagnostics; # checks entire compilation phase 
	print "\ntime for 1st bogus diags: SQUAWKINGS\n";
	print BOGUS1 'nada';
	print "done with 1st bogus\n";

    disable diagnostics; # only turns off runtime warnings
	print "\ntime for 2nd bogus: (squelched)\n";
	print BOGUS2 'nada';
	print "done with 2nd bogus\n";

    enable diagnostics; # turns back on runtime warnings
	print "\ntime for 3rd bogus: SQUAWKINGS\n";
	print BOGUS3 'nada';
	print "done with 3rd bogus\n";

    disable diagnostics;
	print "\ntime for 4th bogus: (squelched)\n";
	print BOGUS4 'nada';
	print "done with 4th bogus\n";

=head1 INTERNALS

Diagnostic messages derive from the F<perldiag.pod> file when available at
runtime.  Otherwise, they may be embedded in the file itself when the
splain package is built.   See the F<Makefile> for details.

If an extant $SIG{__WARN__} handler is discovered, it will continue
to be honored, but only after the diagnostics::splainthis() function 
(the module's $SIG{__WARN__} interceptor) has had its way with your
warnings.

There is a $diagnostics::DEBUG variable you may set if you're desperately
curious what sorts of things are being intercepted.

    BEGIN { $diagnostics::DEBUG = 1 } 


=head1 BUGS

Not being able to say "no diagnostics" is annoying, but may not be
insurmountable.

The C<-pretty> directive is called too late to affect matters.
You have to do this instead, and I<before> you load the module.

    BEGIN { $diagnostics::PRETTY = 1 } 

I could start up faster by delaying compilation until it should be
needed, but this gets a "panic: top_level" when using the pragma form
in Perl 5.001e.

While it's true that this documentation is somewhat subserious, if you use
a program named I<splain>, you should expect a bit of whimsy.

=head1 AUTHOR

Tom Christiansen <F<tchrist@mox.perl.com>>, 25 June 1995.

=cut

require 5.001;
use Carp;

use Config;
($privlib, $archlib) = @Config{qw(privlibexp archlibexp)};
if ($^O eq 'VMS') {
    require VMS::Filespec;
    $privlib = VMS::Filespec::unixify($privlib);
    $archlib = VMS::Filespec::unixify($archlib);
}
@trypod = ("$archlib/pod/perldiag.pod",
	   "$privlib/pod/perldiag-$].pod",
	   "$privlib/pod/perldiag.pod");
# handy for development testing of new warnings etc
unshift @trypod, "./pod/perldiag.pod" if -e "pod/perldiag.pod";
($PODFILE) = ((grep { -e } @trypod), $trypod[$#trypod])[0];

$DEBUG ||= 0;
my $WHOAMI = ref bless [];  # nobody's business, prolly not even mine

$| = 1;

local $_;

CONFIG: {
    $opt_p = $opt_d = $opt_v = $opt_f = '';
    %HTML_2_Troff = %HTML_2_Latin_1 = %HTML_2_ASCII_7 = ();  
    %exact_duplicate = ();

    unless (caller) { 
	$standalone++;
	require Getopt::Std;
	Getopt::Std::getopts('pdvf:')
	    or die "Usage: $0 [-v] [-p] [-f splainpod]";
	$PODFILE = $opt_f if $opt_f;
	$DEBUG = 2 if $opt_d;
	$VERBOSE = $opt_v;
	$PRETTY = $opt_p;
    } 

    if (open(POD_DIAG, $PODFILE)) {
	warn "Happy happy podfile from real $PODFILE\n" if $DEBUG;
	last CONFIG;
    } 

    if (caller) {
	INCPATH: {
	    for $file ( (map { "$_/$WHOAMI.pm" } @INC), $0) {
		warn "Checking $file\n" if $DEBUG;
		if (open(POD_DIAG, $file)) {
		    while (<POD_DIAG>) {
			next unless /^__END__\s*# wish diag dbase were more accessible/;
			print STDERR "podfile is $file\n" if $DEBUG;
			last INCPATH;
		    }
		}
	    }
	}
	*POD_DIAG = *DATA;
    } else { 
	print STDERR "podfile is <DATA>\n" if $DEBUG;
	*POD_DIAG = *main::DATA;
    }
}
if (eof(POD_DIAG)) { 
    die "couldn't find diagnostic data in $PODFILE @INC $0";
}


%HTML_2_Troff = (
    'amp'	=>	'&',	#   ampersand
    'lt'	=>	'<',	#   left chevron, less-than
    'gt'	=>	'>',	#   right chevron, greater-than
    'quot'	=>	'"',	#   double quote

    "Aacute"	=>	"A\\*'",	#   capital A, acute accent
    # etc

);

%HTML_2_Latin_1 = (
    'amp'	=>	'&',	#   ampersand
    'lt'	=>	'<',	#   left chevron, less-than
    'gt'	=>	'>',	#   right chevron, greater-than
    'quot'	=>	'"',	#   double quote

    "Aacute"	=>	"\xC1"	#   capital A, acute accent

    # etc
);

%HTML_2_ASCII_7 = (
    'amp'	=>	'&',	#   ampersand
    'lt'	=>	'<',	#   left chevron, less-than
    'gt'	=>	'>',	#   right chevron, greater-than
    'quot'	=>	'"',	#   double quote

    "Aacute"	=>	"A"	#   capital A, acute accent
    # etc
);

*HTML_Escapes = do {
    if ($standalone) {
	$PRETTY ? \%HTML_2_Latin_1 : \%HTML_2_ASCII_7; 
    } else {
	\%HTML_2_Latin_1; 
    }
}; 

*THITHER = $standalone ? *STDOUT : *STDERR;

$transmo = <<EOFUNC;
sub transmo {
    local \$^W = 0;  # recursive warnings we do NOT need!
    study;
EOFUNC

### sub finish_compilation {  # 5.001e panic: top_level for embedded version
    print STDERR "FINISHING COMPILATION for $_\n" if $DEBUG;
    ### local 
    $RS = '';
    local $_;
    while (<POD_DIAG>) {
	#s/(.*)\n//;
	#$header = $1;

	unescape();
	if ($PRETTY) {
	    sub noop   { return $_[0] }  # spensive for a noop
	    sub bold   { my $str =$_[0];  $str =~ s/(.)/$1\b$1/g; return $str; } 
	    sub italic { my $str = $_[0]; $str =~ s/(.)/_\b$1/g;  return $str; } 
	    s/[BC]<(.*?)>/bold($1)/ges;
	    s/[LIF]<(.*?)>/italic($1)/ges;
	} else {
	    s/[BC]<(.*?)>/$1/gs;
	    s/[LIF]<(.*?)>/$1/gs;
	} 
	unless (/^=/) {
	    if (defined $header) { 
		if ( $header eq 'DESCRIPTION' && 
		    (   /Optional warnings are enabled/ 
		     || /Some of these messages are generic./
		    ) )
		{
		    next;
		} 
		s/^/    /gm;
		$msg{$header} .= $_;
	    }
	    next;
	} 
	unless ( s/=item (.*)\s*\Z//) {

	    if ( s/=head1\sDESCRIPTION//) {
		$msg{$header = 'DESCRIPTION'} = '';
	    }
	    next;
	}

	# strip formatting directives in =item line
	($header = $1) =~ s/[A-Z]<(.*?)>/$1/g;

	if ($header =~ /%[sd]/) {
	    $rhs = $lhs = $header;
	    #if ($lhs =~ s/(.*?)%d(?!%d)(.*)/\Q$1\E\\d+\Q$2\E\$/g)  {
	    if ($lhs =~ s/(.*?)%d(?!%d)(.*)/\Q$1\E\\d+\Q$2\E/g)  {
		$lhs =~ s/\\%s/.*?/g;
	    } else {
		# if i had lookbehind negations, i wouldn't have to do this \377 noise
		$lhs =~ s/(.*?)%s/\Q$1\E.*?\377/g;
		#$lhs =~ s/\377([^\377]*)$/\Q$1\E\$/;
		$lhs =~ s/\377([^\377]*)$/\Q$1\E/;
		$lhs =~ s/\377//g;
		$lhs =~ s/\.\*\?$/.*/; # Allow %s at the end to eat it all
	    } 
	    $transmo .= "    s{^$lhs}\n     {\Q$rhs\E}s\n\t&& return 1;\n";
	} else {
	    $transmo .= "    m{^\Q$header\E} && return 1;\n";
	} 

	print STDERR "$WHOAMI: Duplicate entry: \"$header\"\n"
	    if $msg{$header};

	$msg{$header} = '';
    } 


    close POD_DIAG unless *main::DATA eq *POD_DIAG;

    die "No diagnostics?" unless %msg;

    $transmo .= "    return 0;\n}\n";
    print STDERR $transmo if $DEBUG;
    eval $transmo;
    die $@ if $@;
    $RS = "\n";
### }

if ($standalone) {
    if (!@ARGV and -t STDIN) { print STDERR "$0: Reading from STDIN\n" } 
    while (defined ($error = <>)) {
	splainthis($error) || print THITHER $error;
    } 
    exit;
} else { 
    $old_w = 0; $oldwarn = ''; $olddie = '';
}

sub import {
    shift;
    $old_w = $^W;
    $^W = 1; # yup, clobbered the global variable; tough, if you
	     # want diags, you want diags.
    return if $SIG{__WARN__} eq \&warn_trap;

    for (@_) {

	/^-d(ebug)?$/ 	   	&& do {
				    $DEBUG++;
				    next;
				   };

	/^-v(erbose)?$/ 	&& do {
				    $VERBOSE++;
				    next;
				   };

	/^-p(retty)?$/ 		&& do {
				    print STDERR "$0: I'm afraid it's too late for prettiness.\n";
				    $PRETTY++;
				    next;
			       };

	warn "Unknown flag: $_";
    } 

    $oldwarn = $SIG{__WARN__};
    $olddie = $SIG{__DIE__};
    $SIG{__WARN__} = \&warn_trap;
    $SIG{__DIE__} = \&death_trap;
} 

sub enable { &import }

sub disable {
    shift;
    $^W = $old_w;
    return unless $SIG{__WARN__} eq \&warn_trap;
    $SIG{__WARN__} = $oldwarn;
    $SIG{__DIE__} = $olddie;
} 

sub warn_trap {
    my $warning = $_[0];
    if (caller eq $WHOAMI or !splainthis($warning)) {
	print STDERR $warning;
    } 
    &$oldwarn if defined $oldwarn and $oldwarn and $oldwarn ne \&warn_trap;
};

sub death_trap {
    my $exception = $_[0];

    # See if we are coming from anywhere within an eval. If so we don't
    # want to explain the exception because it's going to get caught.
    my $in_eval = 0;
    my $i = 0;
    while (1) {
      my $caller = (caller($i++))[3] or last;
      if ($caller eq '(eval)') {
	$in_eval = 1;
	last;
      }
    }

    splainthis($exception) unless $in_eval;
    if (caller eq $WHOAMI) { print STDERR "INTERNAL EXCEPTION: $exception"; } 
    &$olddie if defined $olddie and $olddie and $olddie ne \&death_trap;

    # We don't want to unset these if we're coming from an eval because
    # then we've turned off diagnostics. (Actually what does this next
    # line do?  -PSeibel)
    $SIG{__DIE__} = $SIG{__WARN__} = '' unless $in_eval;
    local($Carp::CarpLevel) = 1;
    confess "Uncaught exception from user code:\n\t$exception";
	# up we go; where we stop, nobody knows, but i think we die now
	# but i'm deeply afraid of the &$olddie guy reraising and us getting
	# into an indirect recursion loop
};

sub splainthis {
    local $_ = shift;
    local $\;
    ### &finish_compilation unless %msg;
    s/\.?\n+$//;
    my $orig = $_;
    # return unless defined;
    if ($exact_duplicate{$_}++) {
	return 1;
    } 
    s/, <.*?> (?:line|chunk).*$//;
    $real = s/(.*?) at .*? (?:line|chunk) \d+.*/$1/;
    s/^\((.*)\)$/$1/;
    return 0 unless &transmo;
    $orig = shorten($orig);
    if ($old_diag{$_}) {
	autodescribe();
	print THITHER "$orig (#$old_diag{$_})\n";
	$wantspace = 1;
    } else {
	autodescribe();
	$old_diag{$_} = ++$count;
	print THITHER "\n" if $wantspace;
	$wantspace = 0;
	print THITHER "$orig (#$old_diag{$_})\n";
	if ($msg{$_}) {
	    print THITHER $msg{$_};
	} else {
	    if (0 and $standalone) { 
		print THITHER "    **** Error #$old_diag{$_} ",
			($real ? "is" : "appears to be"),
			" an unknown diagnostic message.\n\n";
	    }
	    return 0;
	} 
    }
    return 1;
} 

sub autodescribe {
    if ($VERBOSE and not $count) {
	print THITHER &{$PRETTY ? \&bold : \&noop}("DESCRIPTION OF DIAGNOSTICS"),
		"\n$msg{DESCRIPTION}\n";
    } 
} 

sub unescape { 
    s {
            E<  
            ( [A-Za-z]+ )       
            >   
    } { 
         do {   
             exists $HTML_Escapes{$1}
                ? do { $HTML_Escapes{$1} }
                : do {
                    warn "Unknown escape: E<$1> in $_";
                    "E<$1>";
                } 
         } 
    }egx;
}

sub shorten {
    my $line = $_[0];
    if (length($line) > 79 and index($line, "\n") == -1) {
	my $space_place = rindex($line, ' ', 79);
	if ($space_place != -1) {
	    substr($line, $space_place, 1) = "\n\t";
	} 
    } 
    return $line;
} 


# have to do this: RS isn't set until run time, but we're executing at compile time
$RS = "\n";

1 unless $standalone;  # or it'll complain about itself
__DATA__ # wish diag dbase were more accessible
=head1 NAME

perldiag - various Perl diagnostics

=head1 DESCRIPTION

These messages are classified as follows (listed in increasing order of
desperation):

    (W) A warning (optional).
    (D) A deprecation (optional).
    (S) A severe warning (mandatory).
    (F) A fatal error (trappable).
    (P) An internal error you should never see (trappable).
    (X) A very fatal error (nontrappable).
    (A) An alien error message (not generated by Perl).

Optional warnings are enabled by using the B<-w> switch.  Warnings may
be captured by setting C<$SIG{__WARN__}> to a reference to a routine that
will be called on each warning instead of printing it.  See L<perlvar>.
Trappable errors may be trapped using the eval operator.  See
L<perlfunc/eval>.

Some of these messages are generic.  Spots that vary are denoted with a %s,
just as in a printf format.  Note that some messages start with a %s!
The symbols C<"%(-?@> sort before the letters, while C<[> and C<\> sort after.

=over 4

=item "my" variable %s can't be in a package

(F) Lexically scoped variables aren't in a package, so it doesn't make sense
to try to declare one with a package qualifier on the front.  Use local()
if you want to localize a package variable.

=item "my" variable %s masks earlier declaration in same scope

(W) A lexical variable has been redeclared in the same scope, effectively
eliminating all access to the previous instance.  This is almost always
a typographical error.  Note that the earlier variable will still exist
until the end of the scope or until all closure referents to it are
destroyed.

=item "no" not allowed in expression

(F) The "no" keyword is recognized and executed at compile time, and returns
no useful value.  See L<perlmod>.

=item "use" not allowed in expression

(F) The "use" keyword is recognized and executed at compile time, and returns
no useful value.  See L<perlmod>.

=item % may only be used in unpack

(F) You can't pack a string by supplying a checksum, because the
checksumming process loses information, and you can't go the other
way.  See L<perlfunc/unpack>.

=item %s (...) interpreted as function

(W) You've run afoul of the rule that says that any list operator followed
by parentheses turns into a function, with all the list operators arguments
found inside the parentheses.  See L<perlop/Terms and List Operators (Leftward)>.

=item %s argument is not a HASH element

(F) The argument to exists() must be a hash element, such as

    $foo{$bar}
    $ref->[12]->{"susie"}

=item %s argument is not a HASH element or slice

(F) The argument to delete() must be either a hash element, such as

    $foo{$bar}
    $ref->[12]->{"susie"}

or a hash slice, such as

    @foo{$bar, $baz, $xyzzy}
    @{$ref->[12]}{"susie", "queue"}

=item %s did not return a true value

(F) A required (or used) file must return a true value to indicate that
it compiled correctly and ran its initialization code correctly.  It's
traditional to end such a file with a "1;", though any true value would
do.  See L<perlfunc/require>.

=item %s found where operator expected

(S) The Perl lexer knows whether to expect a term or an operator.  If it
sees what it knows to be a term when it was expecting to see an operator,
it gives you this warning.  Usually it indicates that an operator or
delimiter was omitted, such as a semicolon.

=item %s had compilation errors

(F) The final summary message when a C<perl -c> fails.

=item %s has too many errors

(F) The parser has given up trying to parse the program after 10 errors.
Further error messages would likely be uninformative.

=item %s matches null string many times

(W) The pattern you've specified would be an infinite loop if the
regular expression engine didn't specifically check for that.  See L<perlre>.

=item %s never introduced

(S) The symbol in question was declared but somehow went out of scope
before it could possibly have been used.

=item %s syntax OK

(F) The final summary message when a C<perl -c> succeeds.

=item %s: Command not found

(A) You've accidentally run your script through B<csh> instead
of Perl.  Check the #! line, or manually feed your script into
Perl yourself.

=item %s: Expression syntax

(A) You've accidentally run your script through B<csh> instead
of Perl.  Check the #! line, or manually feed your script into
Perl yourself.

=item %s: Undefined variable

(A) You've accidentally run your script through B<csh> instead
of Perl.  Check the #! line, or manually feed your script into
Perl yourself.

=item %s: not found

(A) You've accidentally run your script through the Bourne shell
instead of Perl.  Check the #! line, or manually feed your script
into Perl yourself.

=item         (Missing semicolon on previous line?)

(S) This is an educated guess made in conjunction with the message "%s
found where operator expected".  Don't automatically put a semicolon on
the previous line just because you saw this message.

=item B<-P> not allowed for setuid/setgid script

(F) The script would have to be opened by the C preprocessor by name,
which provides a race condition that breaks security.

=item C<-T> and C<-B> not implemented on filehandles

(F) Perl can't peek at the stdio buffer of filehandles when it doesn't
know about your kind of stdio.  You'll have to use a filename instead.

=item C<-p> destination: %s

(F) An error occurred during the implicit output invoked by the C<-p>
command-line switch.  (This output goes to STDOUT unless you've
redirected it with select().)

=item 500 Server error

See Server error.

=item ?+* follows nothing in regexp

(F) You started a regular expression with a quantifier.  Backslash it
if you meant it literally.   See L<perlre>.

=item @ outside of string

(F) You had a pack template that specified an absolute position outside
the string being unpacked.  See L<perlfunc/pack>.

=item accept() on closed fd

(W) You tried to do an accept on a closed socket.  Did you forget to check
the return value of your socket() call?  See L<perlfunc/accept>.

=item Allocation too large: %lx

(X) You can't allocate more than 64K on an MS-DOS machine.

=item Applying %s to %s will act on scalar(%s)

(W) The pattern match (//), substitution (s///), and transliteration (tr///)
operators work on scalar values.  If you apply one of them to an array
or a hash, it will convert the array or hash to a scalar value -- the
length of an array, or the population info of a hash -- and then work on
that scalar value.  This is probably not what you meant to do.  See
L<perlfunc/grep> and L<perlfunc/map> for alternatives.

=item Arg too short for msgsnd

(F) msgsnd() requires a string at least as long as sizeof(long).

=item Ambiguous use of %s resolved as %s

(W)(S) You said something that may not be interpreted the way
you thought.  Normally it's pretty easy to disambiguate it by supplying
a missing quote, operator, parenthesis pair or declaration.

=item Ambiguous call resolved as CORE::%s(), qualify as such or use &

(W) A subroutine you have declared has the same name as a Perl keyword,
and you have used the name without qualification for calling one or the
other.  Perl decided to call the builtin because the subroutine is
not imported.

To force interpretation as a subroutine call, either put an ampersand
before the subroutine name, or qualify the name with its package.
Alternatively, you can import the subroutine (or pretend that it's
imported with the C<use subs> pragma).

To silently interpret it as the Perl operator, use the C<CORE::> prefix
on the operator (e.g. C<CORE::log($x)>) or by declaring the subroutine
to be an object method (see L<attrs>).

=item Args must match #! line

(F) The setuid emulator requires that the arguments Perl was invoked
with match the arguments specified on the #! line.  Since some systems
impose a one-argument limit on the #! line, try combining switches;
for example, turn C<-w -U> into C<-wU>.

=item Argument "%s" isn't numeric%s

(W) The indicated string was fed as an argument to an operator that
expected a numeric value instead.  If you're fortunate the message
will identify which operator was so unfortunate.

=item Array @%s missing the @ in argument %d of %s()

(D) Really old Perl let you omit the @ on array names in some spots.  This
is now heavily deprecated.

=item assertion botched: %s

(P) The malloc package that comes with Perl had an internal failure.

=item Assertion failed: file "%s"

(P) A general assertion failed.  The file in question must be examined.

=item Assignment to both a list and a scalar

(F) If you assign to a conditional operator, the 2nd and 3rd arguments
must either both be scalars or both be lists.  Otherwise Perl won't
know which context to supply to the right side.

=item Attempt to free non-arena SV: 0x%lx

(P) All SV objects are supposed to be allocated from arenas that will
be garbage collected on exit.  An SV was discovered to be outside any
of those arenas.

=item Attempt to free nonexistent shared string

(P) Perl maintains a reference counted internal table of strings to
optimize the storage and access of hash keys and other strings.  This
indicates someone tried to decrement the reference count of a string
that can no longer be found in the table.

=item Attempt to free temp prematurely

(W) Mortalized values are supposed to be freed by the free_tmps()
routine.  This indicates that something else is freeing the SV before
the free_tmps() routine gets a chance, which means that the free_tmps()
routine will be freeing an unreferenced scalar when it does try to free
it.

=item Attempt to free unreferenced glob pointers

(P) The reference counts got screwed up on symbol aliases.

=item Attempt to free unreferenced scalar

(W) Perl went to decrement the reference count of a scalar to see if it
would go to 0, and discovered that it had already gone to 0 earlier,
and should have been freed, and in fact, probably was freed.  This
could indicate that SvREFCNT_dec() was called too many times, or that
SvREFCNT_inc() was called too few times, or that the SV was mortalized
when it shouldn't have been, or that memory has been corrupted.

=item Attempt to pack pointer to temporary value

(W) You tried to pass a temporary value (like the result of a
function, or a computed expression) to the "p" pack() template.  This
means the result contains a pointer to a location that could become
invalid anytime, even before the end of the current statement.  Use
literals or global values as arguments to the "p" pack() template to
avoid this warning.

=item Attempt to use reference as lvalue in substr

(W) You supplied a reference as the first argument to substr() used
as an lvalue, which is pretty strange.  Perhaps you forgot to
dereference it first.  See L<perlfunc/substr>.

=item Bad arg length for %s, is %d, should be %d

(F) You passed a buffer of the wrong size to one of msgctl(), semctl() or
shmctl().  In C parlance, the correct sizes are, respectively,
S<sizeof(struct msqid_ds *)>, S<sizeof(struct semid_ds *)>, and
S<sizeof(struct shmid_ds *)>.

=item Bad filehandle: %s

(F) A symbol was passed to something wanting a filehandle, but the symbol
has no filehandle associated with it.  Perhaps you didn't do an open(), or
did it in another package.

=item Bad free() ignored

(S) An internal routine called free() on something that had never been
malloc()ed in the first place. Mandatory, but can be disabled by
setting environment variable C<PERL_BADFREE> to 1.

This message can be quite often seen with DB_File on systems with
"hard" dynamic linking, like C<AIX> and C<OS/2>. It is a bug of
C<Berkeley DB> which is left unnoticed if C<DB> uses I<forgiving>
system malloc().

=item Bad hash

(P) One of the internal hash routines was passed a null HV pointer.

=item Bad index while coercing array into hash

(F) The index looked up in the hash found as the 0'th element of a
pseudo-hash is not legal.  Index values must be at 1 or greater.
See L<perlref>.

=item Bad name after %s::

(F) You started to name a symbol by using a package prefix, and then didn't
finish the symbol.  In particular, you can't interpolate outside of quotes,
so

    $var = 'myvar';
    $sym = mypack::$var;

is not the same as

    $var = 'myvar';
    $sym = "mypack::$var";

=item Bad symbol for array

(P) An internal request asked to add an array entry to something that
wasn't a symbol table entry.

=item Bad symbol for filehandle

(P) An internal request asked to add a filehandle entry to something that
wasn't a symbol table entry.

=item Bad symbol for hash

(P) An internal request asked to add a hash entry to something that
wasn't a symbol table entry.

=item Badly placed ()'s

(A) You've accidentally run your script through B<csh> instead
of Perl.  Check the #! line, or manually feed your script into
Perl yourself.

=item Bareword "%s" not allowed while "strict subs" in use

(F) With "strict subs" in use, a bareword is only allowed as a
subroutine identifier, in curly braces or to the left of the "=>" symbol.
Perhaps you need to predeclare a subroutine?

=item Bareword "%s" refers to nonexistent package

(W) You used a qualified bareword of the form C<Foo::>, but
the compiler saw no other uses of that namespace before that point.
Perhaps you need to predeclare a package?

=item BEGIN failed--compilation aborted

(F) An untrapped exception was raised while executing a BEGIN subroutine.
Compilation stops immediately and the interpreter is exited.

=item BEGIN not safe after errors--compilation aborted

(F) Perl found a C<BEGIN {}> subroutine (or a C<use> directive, which
implies a C<BEGIN {}>) after one or more compilation errors had
already occurred.  Since the intended environment for the C<BEGIN {}>
could not be guaranteed (due to the errors), and since subsequent code
likely depends on its correct operation, Perl just gave up.

=item bind() on closed fd

(W) You tried to do a bind on a closed socket.  Did you forget to check
the return value of your socket() call?  See L<perlfunc/bind>.

=item Bizarre copy of %s in %s

(P) Perl detected an attempt to copy an internal value that is not copiable.

=item Callback called exit

(F) A subroutine invoked from an external package via perl_call_sv()
exited by calling exit.

=item Can't "goto" outside a block

(F) A "goto" statement was executed to jump out of what might look
like a block, except that it isn't a proper block.  This usually
occurs if you tried to jump out of a sort() block or subroutine, which
is a no-no.  See L<perlfunc/goto>.

=item Can't "goto" into the middle of a foreach loop

(F) A "goto" statement was executed to jump into the middle of a
foreach loop.  You can't get there from here.  See L<perlfunc/goto>.

=item Can't "last" outside a block

(F) A "last" statement was executed to break out of the current block,
except that there's this itty bitty problem called there isn't a
current block.  Note that an "if" or "else" block doesn't count as a
"loopish" block, as doesn't a block given to sort().  You can usually double
the curlies to get the same effect though, because the inner curlies
will be considered a block that loops once.  See L<perlfunc/last>.

=item Can't "next" outside a block

(F) A "next" statement was executed to reiterate the current block, but
there isn't a current block.  Note that an "if" or "else" block doesn't
count as a "loopish" block, as doesn't a block given to sort().  You can
usually double the curlies to get the same effect though, because the inner
curlies will be considered a block that loops once.  See L<perlfunc/next>.

=item Can't "redo" outside a block

(F) A "redo" statement was executed to restart the current block, but
there isn't a current block.  Note that an "if" or "else" block doesn't
count as a "loopish" block, as doesn't a block given to sort().  You can
usually double the curlies to get the same effect though, because the inner
curlies will be considered a block that loops once.  See L<perlfunc/redo>.

=item Can't bless non-reference value

(F) Only hard references may be blessed.  This is how Perl "enforces"
encapsulation of objects.  See L<perlobj>.

=item Can't break at that line

(S) A warning intended to only be printed while running within the debugger, indicating
the line number specified wasn't the location of a statement that could
be stopped at.

=item Can't call method "%s" in empty package "%s"

(F) You called a method correctly, and it correctly indicated a package
functioning as a class, but that package doesn't have ANYTHING defined
in it, let alone methods.  See L<perlobj>.

=item Can't call method "%s" on unblessed reference

(F) A method call must know in what package it's supposed to run.  It
ordinarily finds this out from the object reference you supply, but
you didn't supply an object reference in this case.  A reference isn't
an object reference until it has been blessed.  See L<perlobj>.

=item Can't call method "%s" without a package or object reference

(F) You used the syntax of a method call, but the slot filled by the
object reference or package name contains an expression that returns
a defined value which is neither an object reference nor a package name.
Something like this will reproduce the error:

    $BADREF = 42;
    process $BADREF 1,2,3;
    $BADREF->process(1,2,3);

=item Can't call method "%s" on an undefined value

(F) You used the syntax of a method call, but the slot filled by the
object reference or package name contains an undefined value.
Something like this will reproduce the error:

    $BADREF = undef;
    process $BADREF 1,2,3;
    $BADREF->process(1,2,3);

=item Can't chdir to %s

(F) You called C<perl -x/foo/bar>, but C</foo/bar> is not a directory
that you can chdir to, possibly because it doesn't exist.

=item Can't coerce %s to integer in %s

(F) Certain types of SVs, in particular real symbol table entries
(typeglobs), can't be forced to stop being what they are.  So you can't
say things like:

    *foo += 1;

You CAN say

    $foo = *foo;
    $foo += 1;

but then $foo no longer contains a glob.

=item Can't coerce %s to number in %s

(F) Certain types of SVs, in particular real symbol table entries
(typeglobs), can't be forced to stop being what they are.

=item Can't coerce %s to string in %s

(F) Certain types of SVs, in particular real symbol table entries
(typeglobs), can't be forced to stop being what they are.

=item Can't coerce array into hash

(F) You used an array where a hash was expected, but the array has no
information on how to map from keys to array indices.  You can do that
only with arrays that have a hash reference at index 0.

=item Can't create pipe mailbox

(P) An error peculiar to VMS.  The process is suffering from exhausted quotas
or other plumbing problems.

=item Can't declare %s in my

(F) Only scalar, array, and hash variables may be declared as lexical variables.
They must have ordinary identifiers as names.

=item Can't do inplace edit on %s: %s

(S) The creation of the new file failed for the indicated reason.

=item Can't do inplace edit without backup

(F) You're on a system such as MS-DOS that gets confused if you try reading
from a deleted (but still opened) file.  You have to say C<-i.bak>, or some
such.

=item Can't do inplace edit: %s E<gt> 14 characters

(S) There isn't enough room in the filename to make a backup name for the file.

=item Can't do inplace edit: %s is not a regular file

(S) You tried to use the B<-i> switch on a special file, such as a file in
/dev, or a FIFO.  The file was ignored.

=item Can't do setegid!

(P) The setegid() call failed for some reason in the setuid emulator
of suidperl.

=item Can't do seteuid!

(P) The setuid emulator of suidperl failed for some reason.

=item Can't do setuid

(F) This typically means that ordinary perl tried to exec suidperl to
do setuid emulation, but couldn't exec it.  It looks for a name of the
form sperl5.000 in the same directory that the perl executable resides
under the name perl5.000, typically /usr/local/bin on Unix machines.
If the file is there, check the execute permissions.  If it isn't, ask
your sysadmin why he and/or she removed it.

=item Can't do waitpid with flags

(F) This machine doesn't have either waitpid() or wait4(), so only waitpid()
without flags is emulated.

=item Can't do {n,m} with n E<gt> m

(F) Minima must be less than or equal to maxima.  If you really want
your regexp to match something 0 times, just put {0}.  See L<perlre>.

=item Can't emulate -%s on #! line

(F) The #! line specifies a switch that doesn't make sense at this point.
For example, it'd be kind of silly to put a B<-x> on the #! line.

=item Can't exec "%s": %s

(W) An system(), exec(), or piped open call could not execute the named
program for the indicated reason.  Typical reasons include: the permissions
were wrong on the file, the file wasn't found in C<$ENV{PATH}>, the
executable in question was compiled for another architecture, or the
#! line in a script points to an interpreter that can't be run for
similar reasons.  (Or maybe your system doesn't support #! at all.)

=item Can't exec %s

(F) Perl was trying to execute the indicated program for you because that's
what the #! line said.  If that's not what you wanted, you may need to
mention "perl" on the #! line somewhere.

=item Can't execute %s

(F) You used the B<-S> switch, but the copies of the script to execute found
in the PATH did not have correct permissions.

=item Can't find %s on PATH, '.' not in PATH

(F) You used the B<-S> switch, but the script to execute could not be found
in the PATH, or at least not with the correct permissions.  The script
exists in the current directory, but PATH prohibits running it.

=item Can't find %s on PATH

(F) You used the B<-S> switch, but the script to execute could not be found
in the PATH.

=item Can't find label %s

(F) You said to goto a label that isn't mentioned anywhere that it's possible
for us to go to.  See L<perlfunc/goto>.

=item Can't find string terminator %s anywhere before EOF

(F) Perl strings can stretch over multiple lines.  This message means that
the closing delimiter was omitted.  Because bracketed quotes count nesting
levels, the following is missing its final parenthesis:

    print q(The character '(' starts a side comment.);

If you're getting this error from a here-document, you may have 
included unseen whitespace before or after your closing tag. A good 
programmer's editor will have a way to help you find these characters.

=item Can't fork

(F) A fatal error occurred while trying to fork while opening a pipeline.

=item Can't get filespec - stale stat buffer?

(S) A warning peculiar to VMS.  This arises because of the difference between
access checks under VMS and under the Unix model Perl assumes.  Under VMS,
access checks are done by filename, rather than by bits in the stat buffer, so
that ACLs and other protections can be taken into account.  Unfortunately, Perl
assumes that the stat buffer contains all the necessary information, and passes
it, instead of the filespec, to the access checking routine.  It will try to
retrieve the filespec using the device name and FID present in the stat buffer,
but this works only if you haven't made a subsequent call to the CRTL stat()
routine, because the device name is overwritten with each call.  If this warning
appears, the name lookup failed, and the access checking routine gave up and
returned FALSE, just to be conservative.  (Note: The access checking routine
knows about the Perl C<stat> operator and file tests, so you shouldn't ever
see this warning in response to a Perl command; it arises only if some internal
code takes stat buffers lightly.)

=item Can't get pipe mailbox device name

(P) An error peculiar to VMS.  After creating a mailbox to act as a pipe, Perl
can't retrieve its name for later use.

=item Can't get SYSGEN parameter value for MAXBUF

(P) An error peculiar to VMS.  Perl asked $GETSYI how big you want your
mailbox buffers to be, and didn't get an answer.

=item Can't goto subroutine outside a subroutine

(F) The deeply magical "goto subroutine" call can only replace one subroutine
call for another.  It can't manufacture one out of whole cloth.  In general
you should be calling it out of only an AUTOLOAD routine anyway.  See
L<perlfunc/goto>.

=item Can't goto subroutine from an eval-string

(F) The "goto subroutine" call can't be used to jump out of an eval "string".
(You can use it to jump out of an eval {BLOCK}, but you probably don't want to.)

=item Can't localize through a reference

(F) You said something like C<local $$ref>, which Perl can't currently
handle, because when it goes to restore the old value of whatever $ref
pointed to after the scope of the local() is finished, it can't be
sure that $ref will still be a reference.  

=item Can't localize lexical variable %s

(F) You used local on a variable name that was previously declared as a
lexical variable using "my".  This is not allowed.  If you want to
localize a package variable of the same name, qualify it with the
package name.

=item Can't localize pseudo-hash element

(F) You said something like C<local $ar-E<gt>{'key'}>, where $ar is
a reference to a pseudo-hash.  That hasn't been implemented yet, but
you can get a similar effect by localizing the corresponding array
element directly -- C<local $ar-E<gt>[$ar-E<gt>[0]{'key'}]>.

=item Can't locate auto/%s.al in @INC

(F) A function (or method) was called in a package which allows autoload,
but there is no function to autoload.  Most probable causes are a misprint
in a function/method name or a failure to C<AutoSplit> the file, say, by
doing C<make install>.

=item Can't locate %s in @INC

(F) You said to do (or require, or use) a file that couldn't be found
in any of the libraries mentioned in @INC.  Perhaps you need to set the
PERL5LIB or PERL5OPT environment variable to say where the extra library
is, or maybe the script needs to add the library name to @INC.  Or maybe
you just misspelled the name of the file.  See L<perlfunc/require>.

=item Can't locate object method "%s" via package "%s"

(F) You called a method correctly, and it correctly indicated a package
functioning as a class, but that package doesn't define that particular
method, nor does any of its base classes.  See L<perlobj>.

=item Can't locate package %s for @%s::ISA

(W) The @ISA array contained the name of another package that doesn't seem
to exist.

=item Can't make list assignment to \%ENV on this system

(F) List assignment to %ENV is not supported on some systems, notably VMS.

=item Can't modify %s in %s

(F) You aren't allowed to assign to the item indicated, or otherwise try to
change it, such as with an auto-increment.

=item Can't modify nonexistent substring

(P) The internal routine that does assignment to a substr() was handed
a NULL.

=item Can't msgrcv to read-only var

(F) The target of a msgrcv must be modifiable to be used as a receive
buffer.

=item Can't open %s: %s

(S) The implicit opening of a file through use of the C<E<lt>E<gt>>
filehandle, either implicitly under the C<-n> or C<-p> command-line
switches, or explicitly, failed for the indicated reason.  Usually this
is because you don't have read permission for a file which you named
on the command line.

=item Can't open bidirectional pipe

(W) You tried to say C<open(CMD, "|cmd|")>, which is not supported.  You can
try any of several modules in the Perl library to do this, such as
IPC::Open2.  Alternately, direct the pipe's output to a file using "E<gt>",
and then read it in under a different file handle.

=item Can't open error file %s as stderr

(F) An error peculiar to VMS.  Perl does its own command line redirection, and
couldn't open the file specified after '2E<gt>' or '2E<gt>E<gt>' on the
command line for writing.

=item Can't open input file %s as stdin

(F) An error peculiar to VMS.  Perl does its own command line redirection, and
couldn't open the file specified after 'E<lt>' on the command line for reading.

=item Can't open output file %s as stdout

(F) An error peculiar to VMS.  Perl does its own command line redirection, and
couldn't open the file specified after 'E<gt>' or 'E<gt>E<gt>' on the command
line for writing.

=item Can't open output pipe (name: %s)

(P) An error peculiar to VMS.  Perl does its own command line redirection, and
couldn't open the pipe into which to send data destined for stdout.

=item Can't open perl script "%s": %s

(F) The script you specified can't be opened for the indicated reason.

=item Can't redefine active sort subroutine %s

(F) Perl optimizes the internal handling of sort subroutines and keeps
pointers into them.  You tried to redefine one such sort subroutine when it
was currently active, which is not allowed.  If you really want to do
this, you should write C<sort { &func } @x> instead of C<sort func @x>.

=item Can't rename %s to %s: %s, skipping file

(S) The rename done by the B<-i> switch failed for some reason, probably because
you don't have write permission to the directory.

=item Can't reopen input pipe (name: %s) in binary mode

(P) An error peculiar to VMS.  Perl thought stdin was a pipe, and tried to
reopen it to accept binary data.  Alas, it failed.

=item Can't reswap uid and euid

(P) The setreuid() call failed for some reason in the setuid emulator
of suidperl.

=item Can't return outside a subroutine

(F) The return statement was executed in mainline code, that is, where
there was no subroutine call to return out of.  See L<perlsub>.

=item Can't stat script "%s"

(P) For some reason you can't fstat() the script even though you have
it open already.  Bizarre.

=item Can't swap uid and euid

(P) The setreuid() call failed for some reason in the setuid emulator
of suidperl.

=item Can't take log of %g

(F) For ordinary real numbers, you can't take the logarithm of a
negative number or zero. There's a Math::Complex package that comes
standard with Perl, though, if you really want to do that for
the negative numbers.

=item Can't take sqrt of %g

(F) For ordinary real numbers, you can't take the square root of a
negative number.  There's a Math::Complex package that comes standard
with Perl, though, if you really want to do that.

=item Can't undef active subroutine

(F) You can't undefine a routine that's currently running.  You can,
however, redefine it while it's running, and you can even undef the
redefined subroutine while the old routine is running.  Go figure.

=item Can't unshift

(F) You tried to unshift an "unreal" array that can't be unshifted, such
as the main Perl stack.

=item Can't upgrade that kind of scalar

(P) The internal sv_upgrade routine adds "members" to an SV, making
it into a more specialized kind of SV.  The top several SV types are
so specialized, however, that they cannot be interconverted.  This
message indicates that such a conversion was attempted.

=item Can't upgrade to undef

(P) The undefined SV is the bottom of the totem pole, in the scheme
of upgradability.  Upgrading to undef indicates an error in the
code calling sv_upgrade.

=item Can't use %%! because Errno.pm is not available

(F) The first time the %! hash is used, perl automatically loads the
Errno.pm module. The Errno module is expected to tie the %! hash to
provide symbolic names for C<$!> errno values.

=item Can't use "my %s" in sort comparison

(F) The global variables $a and $b are reserved for sort comparisons.
You mentioned $a or $b in the same line as the E<lt>=E<gt> or cmp operator,
and the variable had earlier been declared as a lexical variable.
Either qualify the sort variable with the package name, or rename the
lexical variable.

=item Can't use %s for loop variable

(F) Only a simple scalar variable may be used as a loop variable on a foreach.

=item Can't use %s ref as %s ref

(F) You've mixed up your reference types.  You have to dereference a
reference of the type needed.  You can use the ref() function to
test the type of the reference, if need be.

=item Can't use \1 to mean $1 in expression

(W) In an ordinary expression, backslash is a unary operator that creates
a reference to its argument.  The use of backslash to indicate a backreference
to a matched substring is valid only as part of a regular expression pattern.
Trying to do this in ordinary Perl code produces a value that prints
out looking like SCALAR(0xdecaf).  Use the $1 form instead.

=item Can't use bareword ("%s") as %s ref while \"strict refs\" in use

(F) Only hard references are allowed by "strict refs".  Symbolic references
are disallowed.  See L<perlref>.

=item Can't use string ("%s") as %s ref while "strict refs" in use

(F) Only hard references are allowed by "strict refs".  Symbolic references
are disallowed.  See L<perlref>.

=item Can't use an undefined value as %s reference

(F) A value used as either a hard reference or a symbolic reference must
be a defined value.  This helps to delurk some insidious errors.

=item Can't use global %s in "my"

(F) You tried to declare a magical variable as a lexical variable.  This is
not allowed, because the magic can be tied to only one location (namely
the global variable) and it would be incredibly confusing to have
variables in your program that looked like magical variables but
weren't.

=item Can't use subscript on %s

(F) The compiler tried to interpret a bracketed expression as a
subscript.  But to the left of the brackets was an expression that
didn't look like an array reference, or anything else subscriptable.

=item Can't x= to read-only value

(F) You tried to repeat a constant value (often the undefined value) with
an assignment operator, which implies modifying the value itself.
Perhaps you need to copy the value to a temporary, and repeat that.

=item Cannot find an opnumber for "%s"

(F) A string of a form C<CORE::word> was given to prototype(), but
there is no builtin with the name C<word>.

=item Cannot resolve method `%s' overloading `%s' in package `%s'

(F|P) Error resolving overloading specified by a method name (as
opposed to a subroutine reference): no such method callable via the
package. If method name is C<???>, this is an internal error.

=item Character class syntax [. .] is reserved for future extensions

(W) Within regular expression character classes ([]) the syntax beginning
with "[." and ending with ".]" is reserved for future extensions.
If you need to represent those character sequences inside a regular
expression character class, just quote the square brackets with the
backslash: "\[." and ".\]".

=item Character class syntax [: :] is reserved for future extensions

(W) Within regular expression character classes ([]) the syntax beginning
with "[:" and ending with ":]" is reserved for future extensions.
If you need to represent those character sequences inside a regular
expression character class, just quote the square brackets with the
backslash: "\[:" and ":\]".

=item Character class syntax [= =] is reserved for future extensions

(W) Within regular expression character classes ([]) the syntax
beginning with "[=" and ending with "=]" is reserved for future extensions.
If you need to represent those character sequences inside a regular
expression character class, just quote the square brackets with the
backslash: "\[=" and "=\]".

=item chmod: mode argument is missing initial 0

(W) A novice will sometimes say

    chmod 777, $filename

not realizing that 777 will be interpreted as a decimal number, equivalent
to 01411.  Octal constants are introduced with a leading 0 in Perl, as in C.

=item Close on unopened file E<lt>%sE<gt>

(W) You tried to close a filehandle that was never opened.

=item Compilation failed in require

(F) Perl could not compile a file specified in a C<require> statement.
Perl uses this generic message when none of the errors that it encountered
were severe enough to halt compilation immediately.

=item Complex regular subexpression recursion limit (%d) exceeded

(W) The regular expression engine uses recursion in complex situations
where back-tracking is required.  Recursion depth is limited to 32766,
or perhaps less in architectures where the stack cannot grow
arbitrarily.  ("Simple" and "medium" situations are handled without
recursion and are not subject to a limit.)  Try shortening the string
under examination; looping in Perl code (e.g. with C<while>) rather
than in the regular expression engine; or rewriting the regular
expression so that it is simpler or backtracks less.  (See L<perlbook>
for information on I<Mastering Regular Expressions>.)

=item connect() on closed fd

(W) You tried to do a connect on a closed socket.  Did you forget to check
the return value of your socket() call?  See L<perlfunc/connect>.

=item Constant subroutine %s redefined

(S) You redefined a subroutine which had previously been eligible for
inlining.  See L<perlsub/"Constant Functions"> for commentary and
workarounds.

=item Constant subroutine %s undefined

(S) You undefined a subroutine which had previously been eligible for
inlining.  See L<perlsub/"Constant Functions"> for commentary and
workarounds.

=item Copy method did not return a reference

(F) The method which overloads "=" is buggy. See L<overload/Copy Constructor>.

=item Corrupt malloc ptr 0x%lx at 0x%lx

(P) The malloc package that comes with Perl had an internal failure.

=item corrupted regexp pointers

(P) The regular expression engine got confused by what the regular
expression compiler gave it.

=item corrupted regexp program

(P) The regular expression engine got passed a regexp program without
a valid magic number.

=item Deep recursion on subroutine "%s"

(W) This subroutine has called itself (directly or indirectly) 100
times more than it has returned.  This probably indicates an infinite
recursion, unless you're writing strange benchmark programs, in which
case it indicates something else.

=item Delimiter for here document is too long

(F) In a here document construct like C<E<lt>E<lt>FOO>, the label
C<FOO> is too long for Perl to handle.  You have to be seriously
twisted to write code that triggers this error.

=item Did you mean &%s instead?

(W) You probably referred to an imported subroutine &FOO as $FOO or some such.

=item Did you mean $ or @ instead of %?

(W) You probably said %hash{$key} when you meant $hash{$key} or @hash{@keys}.
On the other hand, maybe you just meant %hash and got carried away.

=item Died

(F) You passed die() an empty string (the equivalent of C<die "">) or
you called it with no args and both C<$@> and C<$_> were empty.

=item Do you need to predeclare %s?

(S) This is an educated guess made in conjunction with the message "%s
found where operator expected".  It often means a subroutine or module
name is being referenced that hasn't been declared yet.  This may be
because of ordering problems in your file, or because of a missing
"sub", "package", "require", or "use" statement.  If you're
referencing something that isn't defined yet, you don't actually have
to define the subroutine or package before the current location.  You
can use an empty "sub foo;" or "package FOO;" to enter a "forward"
declaration.

=item Don't know how to handle magic of type '%s'

(P) The internal handling of magical variables has been cursed.

=item do_study: out of memory

(P) This should have been caught by safemalloc() instead.

=item Duplicate free() ignored

(S) An internal routine called free() on something that had already
been freed.

=item elseif should be elsif

(S) There is no keyword "elseif" in Perl because Larry thinks it's
ugly.  Your code will be interpreted as an attempt to call a method
named "elseif" for the class returned by the following block.  This is
unlikely to be what you want.

=item END failed--cleanup aborted

(F) An untrapped exception was raised while executing an END subroutine.
The interpreter is immediately exited.

=item Error converting file specification %s

(F) An error peculiar to VMS.  Because Perl may have to deal with file
specifications in either VMS or Unix syntax, it converts them to a
single form when it must operate on them directly.  Either you've
passed an invalid file specification to Perl, or you've found a
case the conversion routines don't handle.  Drat.

=item %s: Eval-group in insecure regular expression

(F) Perl detected tainted data when trying to compile a regular expression
that contains the C<(?{ ... })> zero-width assertion, which is unsafe.
See L<perlre/(?{ code })>, and L<perlsec>.

=item %s: Eval-group not allowed, use re 'eval'

(F) A regular expression contained the C<(?{ ... })> zero-width assertion,
but that construct is only allowed when the C<use re 'eval'> pragma is
in effect.  See L<perlre/(?{ code })>.

=item %s: Eval-group not allowed at run time

(F) Perl tried to compile a regular expression containing the C<(?{ ... })>
zero-width assertion at run time, as it would when the pattern contains
interpolated values.  Since that is a security risk, it is not allowed.
If you insist, you may still do this by explicitly building the pattern
from an interpolated string at run time and using that in an eval().
See L<perlre/(?{ code })>.

=item Excessively long <> operator

(F) The contents of a <> operator may not exceed the maximum size of a
Perl identifier.  If you're just trying to glob a long list of
filenames, try using the glob() operator, or put the filenames into a
variable and glob that.

=item Execution of %s aborted due to compilation errors

(F) The final summary message when a Perl compilation fails.

=item Exiting eval via %s

(W) You are exiting an eval by unconventional means, such as
a goto, or a loop control statement.

=item Exiting pseudo-block via %s

(W) You are exiting a rather special block construct (like a sort block or
subroutine) by unconventional means, such as a goto, or a loop control
statement.  See L<perlfunc/sort>.

=item Exiting subroutine via %s

(W) You are exiting a subroutine by unconventional means, such as
a goto, or a loop control statement.

=item Exiting substitution via %s

(W) You are exiting a substitution by unconventional means, such as
a return, a goto, or a loop control statement.

=item Explicit blessing to '' (assuming package main)

(W) You are blessing a reference to a zero length string.  This has
the effect of blessing the reference into the package main.  This is
usually not what you want.  Consider providing a default target
package, e.g. bless($ref, $p or 'MyPackage');

=item Fatal VMS error at %s, line %d

(P) An error peculiar to VMS.  Something untoward happened in a VMS system
service or RTL routine; Perl's exit status should provide more details.  The
filename in "at %s" and the line number in "line %d" tell you which section of
the Perl source code is distressed.

=item fcntl is not implemented

(F) Your machine apparently doesn't implement fcntl().  What is this, a
PDP-11 or something?

=item Filehandle %s never opened

(W) An I/O operation was attempted on a filehandle that was never initialized.
You need to do an open() or a socket() call, or call a constructor from
the FileHandle package.

=item Filehandle %s opened for only input

(W) You tried to write on a read-only filehandle.  If you
intended it to be a read-write filehandle, you needed to open it with
"+E<lt>" or "+E<gt>" or "+E<gt>E<gt>" instead of with "E<lt>" or nothing.  If
you intended only to write the file, use "E<gt>" or "E<gt>E<gt>".  See
L<perlfunc/open>.

=item Filehandle opened for only input

(W) You tried to write on a read-only filehandle.  If you
intended it to be a read-write filehandle, you needed to open it with
"+E<lt>" or "+E<gt>" or "+E<gt>E<gt>" instead of with "E<lt>" or nothing.  If
you intended only to write the file, use "E<gt>" or "E<gt>E<gt>".  See
L<perlfunc/open>.

=item Final $ should be \$ or $name

(F) You must now decide whether the final $ in a string was meant to be
a literal dollar sign, or was meant to introduce a variable name
that happens to be missing.  So you have to put either the backslash or
the name.

=item Final @ should be \@ or @name

(F) You must now decide whether the final @ in a string was meant to be
a literal "at" sign, or was meant to introduce a variable name
that happens to be missing.  So you have to put either the backslash or
the name.

=item Format %s redefined

(W) You redefined a format.  To suppress this warning, say

    {
	local $^W = 0;
	eval "format NAME =...";
    }

=item Format not terminated

(F) A format must be terminated by a line with a solitary dot.  Perl got
to the end of your file without finding such a line.

=item Found = in conditional, should be ==

(W) You said

    if ($foo = 123)

when you meant

    if ($foo == 123)

(or something like that).

=item gdbm store returned %d, errno %d, key "%s"

(S) A warning from the GDBM_File extension that a store failed.

=item gethostent not implemented

(F) Your C library apparently doesn't implement gethostent(), probably
because if it did, it'd feel morally obligated to return every hostname
on the Internet.

=item get{sock,peer}name() on closed fd

(W) You tried to get a socket or peer socket name on a closed socket.
Did you forget to check the return value of your socket() call?

=item getpwnam returned invalid UIC %#o for user "%s"

(S) A warning peculiar to VMS.  The call to C<sys$getuai> underlying the
C<getpwnam> operator returned an invalid UIC.


=item Glob not terminated

(F) The lexer saw a left angle bracket in a place where it was expecting
a term, so it's looking for the corresponding right angle bracket, and not
finding it.  Chances are you left some needed parentheses out earlier in
the line, and you really meant a "less than".

=item Global symbol "%s" requires explicit package name

(F) You've said "use strict vars", which indicates that all variables
must either be lexically scoped (using "my"), or explicitly qualified to
say which package the global variable is in (using "::").

=item goto must have label

(F) Unlike with "next" or "last", you're not allowed to goto an
unspecified destination.  See L<perlfunc/goto>.

=item Had to create %s unexpectedly

(S) A routine asked for a symbol from a symbol table that ought to have
existed already, but for some reason it didn't, and had to be created on
an emergency basis to prevent a core dump.

=item Hash %%s missing the % in argument %d of %s()

(D) Really old Perl let you omit the % on hash names in some spots.  This
is now heavily deprecated.

=item Identifier too long

(F) Perl limits identifiers (names for variables, functions, etc.) to
about 250 characters for simple names, and somewhat more for compound
names (like C<$A::B>).  You've exceeded Perl's limits.  Future
versions of Perl are likely to eliminate these arbitrary limitations.

=item Ill-formed logical name |%s| in prime_env_iter

(W) A warning peculiar to VMS.  A logical name was encountered when preparing
to iterate over %ENV which violates the syntactic rules governing logical
names.  Because it cannot be translated normally, it is skipped, and will not
appear in %ENV.  This may be a benign occurrence, as some software packages
might directly modify logical name tables and introduce nonstandard names,
or it may indicate that a logical name table has been corrupted.

=item Illegal character %s (carriage return)

(F) A carriage return character was found in the input.  This is an
error, and not a warning, because carriage return characters can break
multi-line strings, including here documents (e.g., C<print E<lt>E<lt>EOF;>).

Under Unix, this error is usually caused by executing Perl code --
either the main program, a module, or an eval'd string -- that was
transferred over a network connection from a non-Unix system without
properly converting the text file format.

Under systems that use something other than '\n' to delimit lines of
text, this error can also be caused by reading Perl code from a file
handle that is in binary mode (as set by the C<binmode> operator).

In either case, the Perl code in question will probably need to be
converted with something like C<s/\x0D\x0A?/\n/g> before it can be
executed.

=item Illegal division by zero

(F) You tried to divide a number by 0.  Either something was wrong in your
logic, or you need to put a conditional in to guard against meaningless input.

=item Illegal modulus zero

(F) You tried to divide a number by 0 to get the remainder.  Most numbers
don't take to this kindly.

=item Illegal octal digit

(F) You used an 8 or 9 in a octal number.

=item Illegal octal digit ignored

(W) You may have tried to use an 8 or 9 in a octal number.  Interpretation
of the octal number stopped before the 8 or 9.

=item Illegal hex digit ignored

(W) You may have tried to use a character other than 0 - 9 or A - F in a
hexadecimal number.  Interpretation of the hexadecimal number stopped
before the illegal character.

=item Illegal switch in PERL5OPT: %s

(X) The PERL5OPT environment variable may only be used to set the
following switches: B<-[DIMUdmw]>.

=item In string, @%s now must be written as \@%s

(F) It used to be that Perl would try to guess whether you wanted an
array interpolated or a literal @.  It did this when the string was first
used at runtime.  Now strings are parsed at compile time, and ambiguous
instances of @ must be disambiguated, either by prepending a backslash to
indicate a literal, or by declaring (or using) the array within the
program before the string (lexically).  (Someday it will simply assume
that an unbackslashed @ interpolates an array.)

=item Insecure dependency in %s

(F) You tried to do something that the tainting mechanism didn't like.
The tainting mechanism is turned on when you're running setuid or setgid,
or when you specify B<-T> to turn it on explicitly.  The tainting mechanism
labels all data that's derived directly or indirectly from the user,
who is considered to be unworthy of your trust.  If any such data is
used in a "dangerous" operation, you get this error.  See L<perlsec>
for more information.

=item Insecure directory in %s

(F) You can't use system(), exec(), or a piped open in a setuid or setgid
script if C<$ENV{PATH}> contains a directory that is writable by the world.
See L<perlsec>.

=item Insecure $ENV{%s} while running %s

(F) You can't use system(), exec(), or a piped open in a setuid or
setgid script if any of C<$ENV{PATH}>, C<$ENV{IFS}>, C<$ENV{CDPATH}>,
C<$ENV{ENV}> or C<$ENV{BASH_ENV}> are derived from data supplied (or
potentially supplied) by the user.  The script must set the path to a
known value, using trustworthy data.  See L<perlsec>.

=item Integer overflow in hex number

(S) The literal hex number you have specified is too big for your
architecture. On a 32-bit architecture the largest hex literal is
0xFFFFFFFF.

=item Integer overflow in octal number

(S) The literal octal number you have specified is too big for your
architecture. On a 32-bit architecture the largest octal literal is
037777777777.

=item Internal inconsistency in tracking vforks

(S) A warning peculiar to VMS.  Perl keeps track of the number
of times you've called C<fork> and C<exec>, to determine
whether the current call to C<exec> should affect the current
script or a subprocess (see L<perlvms/exec>).  Somehow, this count
has become scrambled, so Perl is making a guess and treating
this C<exec> as a request to terminate the Perl script
and execute the specified command.

=item internal disaster in regexp

(P) Something went badly wrong in the regular expression parser.

=item internal error: glob failed

(P) Something went wrong with the external program(s) used for C<glob>
and C<E<lt>*.cE<gt>>.  This may mean that your csh (C shell) is
broken.  If so, you should change all of the csh-related variables in
config.sh:  If you have tcsh, make the variables refer to it as if it
were csh (e.g. C<full_csh='/usr/bin/tcsh'>); otherwise, make them all
empty (except that C<d_csh> should be C<'undef'>) so that Perl will
think csh is missing.  In either case, after editing config.sh, run
C<./Configure -S> and rebuild Perl.

=item internal urp in regexp at /%s/

(P) Something went badly awry in the regular expression parser.

=item invalid [] range in regexp

(F) The range specified in a character class had a minimum character
greater than the maximum character.  See L<perlre>.

=item Invalid conversion in %s: "%s"

(W) Perl does not understand the given format conversion.
See L<perlfunc/sprintf>.

=item Invalid type in pack: '%s'

(F) The given character is not a valid pack type.  See L<perlfunc/pack>.
(W) The given character is not a valid pack type but used to be silently
ignored.

=item Invalid type in unpack: '%s'

(F) The given character is not a valid unpack type.  See L<perlfunc/unpack>.
(W) The given character is not a valid unpack type but used to be silently
ignored.

=item ioctl is not implemented

(F) Your machine apparently doesn't implement ioctl(), which is pretty
strange for a machine that supports C.

=item junk on end of regexp

(P) The regular expression parser is confused.

=item Label not found for "last %s"

(F) You named a loop to break out of, but you're not currently in a
loop of that name, not even if you count where you were called from.
See L<perlfunc/last>.

=item Label not found for "next %s"

(F) You named a loop to continue, but you're not currently in a loop of
that name, not even if you count where you were called from.  See
L<perlfunc/last>.

=item Label not found for "redo %s"

(F) You named a loop to restart, but you're not currently in a loop of
that name, not even if you count where you were called from.  See
L<perlfunc/last>.

=item listen() on closed fd

(W) You tried to do a listen on a closed socket.  Did you forget to check
the return value of your socket() call?  See L<perlfunc/listen>.

=item Method for operation %s not found in package %s during blessing

(F) An attempt was made to specify an entry in an overloading table that
doesn't resolve to a valid subroutine.  See L<overload>.

=item Might be a runaway multi-line %s string starting on line %d

(S) An advisory indicating that the previous error may have been caused
by a missing delimiter on a string or pattern, because it eventually
ended earlier on the current line.

=item Misplaced _ in number

(W) An underline in a decimal constant wasn't on a 3-digit boundary.

=item Missing $ on loop variable

(F) Apparently you've been programming in B<csh> too much.  Variables are always
mentioned with the $ in Perl, unlike in the shells, where it can vary from
one line to the next.

=item Missing comma after first argument to %s function

(F) While certain functions allow you to specify a filehandle or an
"indirect object" before the argument list, this ain't one of them.

=item Missing operator before %s?

(S) This is an educated guess made in conjunction with the message "%s
found where operator expected".  Often the missing operator is a comma.

=item Missing right bracket

(F) The lexer counted more opening curly brackets (braces) than closing ones.
As a general rule, you'll find it's missing near the place you were last
editing.

=item Modification of a read-only value attempted

(F) You tried, directly or indirectly, to change the value of a
constant.  You didn't, of course, try "2 = 1", because the compiler
catches that.  But an easy way to do the same thing is:

    sub mod { $_[0] = 1 }
    mod(2);

Another way is to assign to a substr() that's off the end of the string.

=item Modification of non-creatable array value attempted, subscript %d

(F) You tried to make an array value spring into existence, and the
subscript was probably negative, even counting from end of the array
backwards.

=item Modification of non-creatable hash value attempted, subscript "%s"

(P) You tried to make a hash value spring into existence, and it couldn't
be created for some peculiar reason.

=item Module name must be constant

(F) Only a bare module name is allowed as the first argument to a "use".

=item msg%s not implemented

(F) You don't have System V message IPC on your system.

=item Multidimensional syntax %s not supported

(W) Multidimensional arrays aren't written like C<$foo[1,2,3]>.  They're written
like C<$foo[1][2][3]>, as in C.

=item Name "%s::%s" used only once: possible typo

(W) Typographical errors often show up as unique variable names.
If you had a good reason for having a unique name, then just mention
it again somehow to suppress the message.  The C<use vars> pragma is
provided for just this purpose.

=item Negative length

(F) You tried to do a read/write/send/recv operation with a buffer length
that is less than 0.  This is difficult to imagine.

=item nested *?+ in regexp

(F) You can't quantify a quantifier without intervening parentheses.  So
things like ** or +* or ?* are illegal.

Note, however, that the minimal matching quantifiers, C<*?>, C<+?>, and C<??> appear
to be nested quantifiers, but aren't.  See L<perlre>.

=item No #! line

(F) The setuid emulator requires that scripts have a well-formed #! line
even on machines that don't support the #! construct.

=item No %s allowed while running setuid

(F) Certain operations are deemed to be too insecure for a setuid or setgid
script to even be allowed to attempt.  Generally speaking there will be
another way to do what you want that is, if not secure, at least securable.
See L<perlsec>.

=item No B<-e> allowed in setuid scripts

(F) A setuid script can't be specified by the user.

=item No comma allowed after %s

(F) A list operator that has a filehandle or "indirect object" is not
allowed to have a comma between that and the following arguments.
Otherwise it'd be just another one of the arguments.

One possible cause for this is that you expected to have imported a
constant to your name space with B<use> or B<import> while no such
importing took place, it may for example be that your operating system
does not support that particular constant. Hopefully you did use an
explicit import list for the constants you expect to see, please see
L<perlfunc/use> and L<perlfunc/import>. While an explicit import list
would probably have caught this error earlier it naturally does not
remedy the fact that your operating system still does not support that
constant. Maybe you have a typo in the constants of the symbol import
list of B<use> or B<import> or in the constant name at the line where
this error was triggered?

=item No command into which to pipe on command line

(F) An error peculiar to VMS.  Perl handles its own command line redirection,
and found a '|' at the end of the command line, so it doesn't know where you
want to pipe the output from this command.

=item No DB::DB routine defined

(F) The currently executing code was compiled with the B<-d> switch,
but for some reason the perl5db.pl file (or some facsimile thereof)
didn't define a routine to be called at the beginning of each
statement.  Which is odd, because the file should have been required
automatically, and should have blown up the require if it didn't parse
right.

=item No dbm on this machine

(P) This is counted as an internal error, because every machine should
supply dbm nowadays, because Perl comes with SDBM.  See L<SDBM_File>.

=item No DBsub routine

(F) The currently executing code was compiled with the B<-d> switch,
but for some reason the perl5db.pl file (or some facsimile thereof)
didn't define a DB::sub routine to be called at the beginning of each
ordinary subroutine call.

=item No error file after 2E<gt> or 2E<gt>E<gt> on command line

(F) An error peculiar to VMS.  Perl handles its own command line redirection,
and found a '2E<gt>' or a '2E<gt>E<gt>' on the command line, but can't find
the name of the file to which to write data destined for stderr.

=item No input file after E<lt> on command line

(F) An error peculiar to VMS.  Perl handles its own command line redirection,
and found a 'E<lt>' on the command line, but can't find the name of the file
from which to read data for stdin.

=item No output file after E<gt> on command line

(F) An error peculiar to VMS.  Perl handles its own command line redirection,
and found a lone 'E<gt>' at the end of the command line, so it doesn't know
where you wanted to redirect stdout.

=item No output file after E<gt> or E<gt>E<gt> on command line

(F) An error peculiar to VMS.  Perl handles its own command line redirection,
and found a 'E<gt>' or a 'E<gt>E<gt>' on the command line, but can't find the
name of the file to which to write data destined for stdout.

=item No Perl script found in input

(F) You called C<perl -x>, but no line was found in the file beginning
with #! and containing the word "perl".

=item No setregid available

(F) Configure didn't find anything resembling the setregid() call for
your system.

=item No setreuid available

(F) Configure didn't find anything resembling the setreuid() call for
your system.

=item No space allowed after B<-I>

(F) The argument to B<-I> must follow the B<-I> immediately with no
intervening space.

=item No such array field

(F) You tried to access an array as a hash, but the field name used is
not defined.  The hash at index 0 should map all valid field names to
array indices for that to work.

=item No such field "%s" in variable %s of type %s

(F) You tried to access a field of a typed variable where the type
does not know about the field name.  The field names are looked up in
the %FIELDS hash in the type package at compile time.  The %FIELDS hash
is usually set up with the 'fields' pragma.

=item No such pipe open

(P) An error peculiar to VMS.  The internal routine my_pclose() tried to
close a pipe which hadn't been opened.  This should have been caught earlier as
an attempt to close an unopened filehandle.

=item No such signal: SIG%s

(W) You specified a signal name as a subscript to %SIG that was not recognized.
Say C<kill -l> in your shell to see the valid signal names on your system.

=item Not a CODE reference

(F) Perl was trying to evaluate a reference to a code value (that is, a
subroutine), but found a reference to something else instead.  You can
use the ref() function to find out what kind of ref it really was.
See also L<perlref>.

=item Not a format reference

(F) I'm not sure how you managed to generate a reference to an anonymous
format, but this indicates you did, and that it didn't exist.

=item Not a GLOB reference

(F) Perl was trying to evaluate a reference to a "typeglob" (that is,
a symbol table entry that looks like C<*foo>), but found a reference to
something else instead.  You can use the ref() function to find out
what kind of ref it really was.  See L<perlref>.

=item Not a HASH reference

(F) Perl was trying to evaluate a reference to a hash value, but
found a reference to something else instead.  You can use the ref()
function to find out what kind of ref it really was.  See L<perlref>.

=item Not a perl script

(F) The setuid emulator requires that scripts have a well-formed #! line
even on machines that don't support the #! construct.  The line must
mention perl.

=item Not a SCALAR reference

(F) Perl was trying to evaluate a reference to a scalar value, but
found a reference to something else instead.  You can use the ref()
function to find out what kind of ref it really was.  See L<perlref>.

=item Not a subroutine reference

(F) Perl was trying to evaluate a reference to a code value (that is, a
subroutine), but found a reference to something else instead.  You can
use the ref() function to find out what kind of ref it really was.
See also L<perlref>.

=item Not a subroutine reference in overload table

(F) An attempt was made to specify an entry in an overloading table that
doesn't somehow point to a valid subroutine.  See L<overload>.

=item Not an ARRAY reference

(F) Perl was trying to evaluate a reference to an array value, but
found a reference to something else instead.  You can use the ref()
function to find out what kind of ref it really was.  See L<perlref>.

=item Not enough arguments for %s

(F) The function requires more arguments than you specified.

=item Not enough format arguments

(W) A format specified more picture fields than the next line supplied.
See L<perlform>.

=item Null filename used

(F) You can't require the null filename, especially because on many machines
that means the current directory!  See L<perlfunc/require>.

=item Null picture in formline

(F) The first argument to formline must be a valid format picture
specification.  It was found to be empty, which probably means you
supplied it an uninitialized value.  See L<perlform>.

=item NULL OP IN RUN

(P) Some internal routine called run() with a null opcode pointer.

=item Null realloc

(P) An attempt was made to realloc NULL.

=item NULL regexp argument

(P) The internal pattern matching routines blew it big time.

=item NULL regexp parameter

(P) The internal pattern matching routines are out of their gourd.

=item Number too long

(F) Perl limits the representation of decimal numbers in programs to about
about 250 characters.  You've exceeded that length.  Future versions of
Perl are likely to eliminate this arbitrary limitation.  In the meantime,
try using scientific notation (e.g. "1e6" instead of "1_000_000").

=item Odd number of elements in hash assignment

(S) You specified an odd number of elements to initialize a hash, which
is odd, because hashes come in key/value pairs.

=item Offset outside string

(F) You tried to do a read/write/send/recv operation with an offset
pointing outside the buffer.  This is difficult to imagine.
The sole exception to this is that C<sysread()>ing past the buffer
will extend the buffer and zero pad the new area.

=item oops: oopsAV

(S) An internal warning that the grammar is screwed up.

=item oops: oopsHV

(S) An internal warning that the grammar is screwed up.

=item Operation `%s': no method found, %s

(F) An attempt was made to perform an overloaded operation for which
no handler was defined.  While some handlers can be autogenerated in
terms of other handlers, there is no default handler for any
operation, unless C<fallback> overloading key is specified to be
true.  See L<overload>.

=item Operator or semicolon missing before %s

(S) You used a variable or subroutine call where the parser was
expecting an operator.  The parser has assumed you really meant
to use an operator, but this is highly likely to be incorrect.
For example, if you say "*foo *foo" it will be interpreted as
if you said "*foo * 'foo'".

=item Out of memory for yacc stack

(F) The yacc parser wanted to grow its stack so it could continue parsing,
but realloc() wouldn't give it more memory, virtual or otherwise.

=item Out of memory during request for %s

(X|F) The malloc() function returned 0, indicating there was insufficient
remaining memory (or virtual memory) to satisfy the request.

The request was judged to be small, so the possibility to trap it
depends on the way perl was compiled.  By default it is not trappable.
However, if compiled for this, Perl may use the contents of C<$^M> as
an emergency pool after die()ing with this message.  In this case the
error is trappable I<once>.

=item Out of memory during "large" request for %s

(F) The malloc() function returned 0, indicating there was insufficient
remaining memory (or virtual memory) to satisfy the request. However,
the request was judged large enough (compile-time default is 64K), so
a possibility to shut down by trapping this error is granted.

=item Out of memory during ridiculously large request

(F) You can't allocate more than 2^31+"small amount" bytes.  This error
is most likely to be caused by a typo in the Perl program. e.g., C<$arr[time]>
instead of C<$arr[$time]>.

=item page overflow

(W) A single call to write() produced more lines than can fit on a page.
See L<perlform>.

=item panic: ck_grep

(P) Failed an internal consistency check trying to compile a grep.

=item panic: ck_split

(P) Failed an internal consistency check trying to compile a split.

=item panic: corrupt saved stack index

(P) The savestack was requested to restore more localized values than there
are in the savestack.

=item panic: die %s

(P) We popped the context stack to an eval context, and then discovered
it wasn't an eval context.

=item panic: do_match

(P) The internal pp_match() routine was called with invalid operational data.

=item panic: do_split

(P) Something terrible went wrong in setting up for the split.

=item panic: do_subst

(P) The internal pp_subst() routine was called with invalid operational data.

=item panic: do_trans

(P) The internal do_trans() routine was called with invalid operational data.

=item panic: frexp

(P) The library function frexp() failed, making printf("%f") impossible.

=item panic: goto

(P) We popped the context stack to a context with the specified label,
and then discovered it wasn't a context we know how to do a goto in.

=item panic: INTERPCASEMOD

(P) The lexer got into a bad state at a case modifier.

=item panic: INTERPCONCAT

(P) The lexer got into a bad state parsing a string with brackets.

=item panic: last

(P) We popped the context stack to a block context, and then discovered
it wasn't a block context.

=item panic: leave_scope clearsv

(P) A writable lexical variable became read-only somehow within the scope.

=item panic: leave_scope inconsistency

(P) The savestack probably got out of sync.  At least, there was an
invalid enum on the top of it.

=item panic: malloc

(P) Something requested a negative number of bytes of malloc.

=item panic: mapstart

(P) The compiler is screwed up with respect to the map() function.

=item panic: null array

(P) One of the internal array routines was passed a null AV pointer.

=item panic: pad_alloc

(P) The compiler got confused about which scratch pad it was allocating
and freeing temporaries and lexicals from.

=item panic: pad_free curpad

(P) The compiler got confused about which scratch pad it was allocating
and freeing temporaries and lexicals from.

=item panic: pad_free po

(P) An invalid scratch pad offset was detected internally.

=item panic: pad_reset curpad

(P) The compiler got confused about which scratch pad it was allocating
and freeing temporaries and lexicals from.

=item panic: pad_sv po

(P) An invalid scratch pad offset was detected internally.

=item panic: pad_swipe curpad

(P) The compiler got confused about which scratch pad it was allocating
and freeing temporaries and lexicals from.

=item panic: pad_swipe po

(P) An invalid scratch pad offset was detected internally.

=item panic: pp_iter

(P) The foreach iterator got called in a non-loop context frame.

=item panic: realloc

(P) Something requested a negative number of bytes of realloc.

=item panic: restartop

(P) Some internal routine requested a goto (or something like it), and
didn't supply the destination.

=item panic: return

(P) We popped the context stack to a subroutine or eval context, and
then discovered it wasn't a subroutine or eval context.

=item panic: scan_num

(P) scan_num() got called on something that wasn't a number.

=item panic: sv_insert

(P) The sv_insert() routine was told to remove more string than there
was string.

=item panic: top_env

(P) The compiler attempted to do a goto, or something weird like that.

=item panic: yylex

(P) The lexer got into a bad state while processing a case modifier.

=item Parentheses missing around "%s" list

(W) You said something like

    my $foo, $bar = @_;

when you meant

    my ($foo, $bar) = @_;

Remember that "my" and "local" bind closer than comma.

=item Perl %3.3f required--this is only version %s, stopped

(F) The module in question uses features of a version of Perl more recent
than the currently running version.  How long has it been since you upgraded,
anyway?  See L<perlfunc/require>.

=item Permission denied

(F) The setuid emulator in suidperl decided you were up to no good.

=item pid %d not a child

(W) A warning peculiar to VMS.  Waitpid() was asked to wait for a process which
isn't a subprocess of the current process.  While this is fine from VMS'
perspective, it's probably not what you intended.

=item POSIX getpgrp can't take an argument

(F) Your C compiler uses POSIX getpgrp(), which takes no argument, unlike
the BSD version, which takes a pid.

=item Possible attempt to put comments in qw() list

(W) qw() lists contain items separated by whitespace; as with literal
strings, comment characters are not ignored, but are instead treated
as literal data.  (You may have used different delimiters than the
parentheses shown here; braces are also frequently used.)

You probably wrote something like this:

    @list = qw(
	a # a comment
        b # another comment
    );

when you should have written this:

    @list = qw(
	a
        b
    );

If you really want comments, build your list the
old-fashioned way, with quotes and commas:

    @list = (
        'a',    # a comment
        'b',    # another comment
    );

=item Possible attempt to separate words with commas

(W) qw() lists contain items separated by whitespace; therefore commas
aren't needed to separate the items.  (You may have used different
delimiters than the parentheses shown here; braces are also frequently
used.)

You probably wrote something like this:

    qw! a, b, c !;

which puts literal commas into some of the list items.  Write it without
commas if you don't want them to appear in your data:

    qw! a b c !;

=item Possible memory corruption: %s overflowed 3rd argument

(F) An ioctl() or fcntl() returned more than Perl was bargaining for.
Perl guesses a reasonable buffer size, but puts a sentinel byte at the
end of the buffer just in case.  This sentinel byte got clobbered, and
Perl assumes that memory is now corrupted.  See L<perlfunc/ioctl>.

=item Precedence problem: open %s should be open(%s)

(S) The old irregular construct

    open FOO || die;

is now misinterpreted as

    open(FOO || die);

because of the strict regularization of Perl 5's grammar into unary
and list operators.  (The old open was a little of both.)  You must
put parentheses around the filehandle, or use the new "or" operator
instead of "||".

=item print on closed filehandle %s

(W) The filehandle you're printing on got itself closed sometime before now.
Check your logic flow.

=item printf on closed filehandle %s

(W) The filehandle you're writing to got itself closed sometime before now.
Check your logic flow.

=item Probable precedence problem on %s

(W) The compiler found a bareword where it expected a conditional,
which often indicates that an || or && was parsed as part of the
last argument of the previous construct, for example:

    open FOO || die;

=item Prototype mismatch: %s vs %s

(S) The subroutine being declared or defined had previously been declared
or defined with a different function prototype.

=item Range iterator outside integer range

(F) One (or both) of the numeric arguments to the range operator ".."
are outside the range which can be represented by integers internally.
One possible workaround is to force Perl to use magical string
increment by prepending "0" to your numbers.

=item Read on closed filehandle E<lt>%sE<gt>

(W) The filehandle you're reading from got itself closed sometime before now.
Check your logic flow.

=item Reallocation too large: %lx

(F) You can't allocate more than 64K on an MS-DOS machine.

=item Recompile perl with B<-D>DEBUGGING to use B<-D> switch

(F) You can't use the B<-D> option unless the code to produce the
desired output is compiled into Perl, which entails some overhead,
which is why it's currently left out of your copy.

=item Recursive inheritance detected in package '%s'

(F) More than 100 levels of inheritance were used.  Probably indicates
an unintended loop in your inheritance hierarchy.

=item Recursive inheritance detected while looking for method '%s' in package '%s'

(F) More than 100 levels of inheritance were encountered while invoking a
method.  Probably indicates an unintended loop in your inheritance hierarchy.

=item Reference found where even-sized list expected

(W) You gave a single reference where Perl was expecting a list with
an even number of elements (for assignment to a hash). This
usually means that you used the anon hash constructor when you meant 
to use parens. In any case, a hash requires key/value B<pairs>.

    %hash = { one => 1, two => 2, };	# WRONG
    %hash = [ qw/ an anon array / ];	# WRONG
    %hash = ( one => 1, two => 2, );	# right
    %hash = qw( one 1 two 2 );			# also fine

=item Reference miscount in sv_replace()

(W) The internal sv_replace() function was handed a new SV with a
reference count of other than 1.

=item regexp *+ operand could be empty

(F) The part of the regexp subject to either the * or + quantifier
could match an empty string.

=item regexp memory corruption

(P) The regular expression engine got confused by what the regular
expression compiler gave it.

=item regexp out of space

(P) A "can't happen" error, because safemalloc() should have caught it earlier.

=item regexp too big

(F) The current implementation of regular expressions uses shorts as
address offsets within a string.  Unfortunately this means that if
the regular expression compiles to longer than 32767, it'll blow up.
Usually when you want a regular expression this big, there is a better
way to do it with multiple statements.  See L<perlre>.

=item Reversed %s= operator

(W) You wrote your assignment operator backwards.  The = must always
comes last, to avoid ambiguity with subsequent unary operators.

=item Runaway format

(F) Your format contained the ~~ repeat-until-blank sequence, but it
produced 200 lines at once, and the 200th line looked exactly like the
199th line.  Apparently you didn't arrange for the arguments to exhaust
themselves, either by using ^ instead of @ (for scalar variables), or by
shifting or popping (for array variables).  See L<perlform>.

=item Scalar value @%s[%s] better written as $%s[%s]

(W) You've used an array slice (indicated by @) to select a single element of
an array.  Generally it's better to ask for a scalar value (indicated by $).
The difference is that C<$foo[&bar]> always behaves like a scalar, both when
assigning to it and when evaluating its argument, while C<@foo[&bar]> behaves
like a list when you assign to it, and provides a list context to its
subscript, which can do weird things if you're expecting only one subscript.

On the other hand, if you were actually hoping to treat the array
element as a list, you need to look into how references work, because
Perl will not magically convert between scalars and lists for you.  See
L<perlref>.

=item Scalar value @%s{%s} better written as $%s{%s}

(W) You've used a hash slice (indicated by @) to select a single element of
a hash.  Generally it's better to ask for a scalar value (indicated by $).
The difference is that C<$foo{&bar}> always behaves like a scalar, both when
assigning to it and when evaluating its argument, while C<@foo{&bar}> behaves
like a list when you assign to it, and provides a list context to its
subscript, which can do weird things if you're expecting only one subscript.

On the other hand, if you were actually hoping to treat the hash
element as a list, you need to look into how references work, because
Perl will not magically convert between scalars and lists for you.  See
L<perlref>.

=item Script is not setuid/setgid in suidperl

(F) Oddly, the suidperl program was invoked on a script without a setuid
or setgid bit set.  This doesn't make much sense.

=item Search pattern not terminated

(F) The lexer couldn't find the final delimiter of a // or m{}
construct.  Remember that bracketing delimiters count nesting level.
Missing the leading C<$> from a variable C<$m> may cause this error.

=item %sseek() on unopened file

(W) You tried to use the seek() or sysseek() function on a filehandle that
was either never opened or has since been closed.

=item select not implemented

(F) This machine doesn't implement the select() system call.

=item sem%s not implemented

(F) You don't have System V semaphore IPC on your system.

=item semi-panic: attempt to dup freed string

(S) The internal newSVsv() routine was called to duplicate a scalar
that had previously been marked as free.

=item Semicolon seems to be missing

(W) A nearby syntax error was probably caused by a missing semicolon,
or possibly some other missing operator, such as a comma.

=item Send on closed socket

(W) The filehandle you're sending to got itself closed sometime before now.
Check your logic flow.

=item Sequence (? incomplete

(F) A regular expression ended with an incomplete extension (?.
See L<perlre>.

=item Sequence (?#... not terminated

(F) A regular expression comment must be terminated by a closing
parenthesis.  Embedded parentheses aren't allowed.  See L<perlre>.

=item Sequence (?%s...) not implemented

(F) A proposed regular expression extension has the character reserved
but has not yet been written.  See L<perlre>.

=item Sequence (?%s...) not recognized

(F) You used a regular expression extension that doesn't make sense.
See L<perlre>.

=item Server error

Also known as "500 Server error".

B<This is a CGI error, not a Perl error>.

You need to make sure your script is executable, is accessible by the user
CGI is running the script under (which is probably not the user account you
tested it under), does not rely on any environment variables (like PATH)
from the user it isn't running under, and isn't in a location where the CGI
server can't find it, basically, more or less.  Please see the following
for more information:

	http://www.perl.com/perl/faq/idiots-guide.html
	http://www.perl.com/perl/faq/perl-cgi-faq.html
	ftp://rtfm.mit.edu/pub/usenet/news.answers/www/cgi-faq
	http://hoohoo.ncsa.uiuc.edu/cgi/interface.html
	http://www-genome.wi.mit.edu/WWW/faqs/www-security-faq.html

=item setegid() not implemented

(F) You tried to assign to C<$)>, and your operating system doesn't support
the setegid() system call (or equivalent), or at least Configure didn't
think so.

=item seteuid() not implemented

(F) You tried to assign to C<$E<gt>>, and your operating system doesn't support
the seteuid() system call (or equivalent), or at least Configure didn't
think so.

=item setrgid() not implemented

(F) You tried to assign to C<$(>, and your operating system doesn't support
the setrgid() system call (or equivalent), or at least Configure didn't
think so.

=item setruid() not implemented

(F) You tried to assign to C<$E<lt>>, and your operating system doesn't support
the setruid() system call (or equivalent), or at least Configure didn't
think so.

=item Setuid/gid script is writable by world

(F) The setuid emulator won't run a script that is writable by the world,
because the world might have written on it already.

=item shm%s not implemented

(F) You don't have System V shared memory IPC on your system.

=item shutdown() on closed fd

(W) You tried to do a shutdown on a closed socket.  Seems a bit superfluous.

=item SIG%s handler "%s" not defined

(W) The signal handler named in %SIG doesn't, in fact, exist.  Perhaps you
put it into the wrong package?

=item sort is now a reserved word

(F) An ancient error message that almost nobody ever runs into anymore.
But before sort was a keyword, people sometimes used it as a filehandle.

=item Sort subroutine didn't return a numeric value

(F) A sort comparison routine must return a number.  You probably blew
it by not using C<E<lt>=E<gt>> or C<cmp>, or by not using them correctly.
See L<perlfunc/sort>.

=item Sort subroutine didn't return single value

(F) A sort comparison subroutine may not return a list value with more
or less than one element.  See L<perlfunc/sort>.

=item Split loop

(P) The split was looping infinitely.  (Obviously, a split shouldn't iterate
more times than there are characters of input, which is what happened.)
See L<perlfunc/split>.

=item Stat on unopened file E<lt>%sE<gt>

(W) You tried to use the stat() function (or an equivalent file test)
on a filehandle that was either never opened or has since been closed.

=item Statement unlikely to be reached

(W) You did an exec() with some statement after it other than a die().
This is almost always an error, because exec() never returns unless
there was a failure.  You probably wanted to use system() instead,
which does return.  To suppress this warning, put the exec() in a block
by itself.

=item Stub found while resolving method `%s' overloading `%s' in package `%s'

(P) Overloading resolution over @ISA tree may be broken by importation stubs.
Stubs should never be implicitely created, but explicit calls to C<can>
may break this.

=item Subroutine %s redefined

(W) You redefined a subroutine.  To suppress this warning, say

    {
	local $^W = 0;
	eval "sub name { ... }";
    }

=item Substitution loop

(P) The substitution was looping infinitely.  (Obviously, a
substitution shouldn't iterate more times than there are characters of
input, which is what happened.)  See the discussion of substitution in
L<perlop/"Quote and Quote-like Operators">.

=item Substitution pattern not terminated

(F) The lexer couldn't find the interior delimiter of a s/// or s{}{}
construct.  Remember that bracketing delimiters count nesting level.
Missing the leading C<$> from variable C<$s> may cause this error.

=item Substitution replacement not terminated

(F) The lexer couldn't find the final delimiter of a s/// or s{}{}
construct.  Remember that bracketing delimiters count nesting level.
Missing the leading C<$> from variable C<$s> may cause this error.

=item substr outside of string

(S),(W) You tried to reference a substr() that pointed outside of a
string.  That is, the absolute value of the offset was larger than the
length of the string.  See L<perlfunc/substr>.  This warning is
mandatory if substr is used in an lvalue context (as the left hand side
of an assignment or as a subroutine argument for example).

=item suidperl is no longer needed since %s

(F) Your Perl was compiled with B<-D>SETUID_SCRIPTS_ARE_SECURE_NOW, but a
version of the setuid emulator somehow got run anyway.

=item syntax error

(F) Probably means you had a syntax error.  Common reasons include:

    A keyword is misspelled.
    A semicolon is missing.
    A comma is missing.
    An opening or closing parenthesis is missing.
    An opening or closing brace is missing.
    A closing quote is missing.

Often there will be another error message associated with the syntax
error giving more information.  (Sometimes it helps to turn on B<-w>.)
The error message itself often tells you where it was in the line when
it decided to give up.  Sometimes the actual error is several tokens
before this, because Perl is good at understanding random input.
Occasionally the line number may be misleading, and once in a blue moon
the only way to figure out what's triggering the error is to call
C<perl -c> repeatedly, chopping away half the program each time to see
if the error went away.  Sort of the cybernetic version of S<20 questions>.

=item syntax error at line %d: `%s' unexpected

(A) You've accidentally run your script through the Bourne shell
instead of Perl.  Check the #! line, or manually feed your script
into Perl yourself.

=item System V %s is not implemented on this machine

(F) You tried to do something with a function beginning with "sem",
"shm", or "msg" but that System V IPC is not implemented in your
machine.  In some machines the functionality can exist but be
unconfigured.  Consult your system support.

=item Syswrite on closed filehandle

(W) The filehandle you're writing to got itself closed sometime before now.
Check your logic flow.

=item Target of goto is too deeply nested

(F) You tried to use C<goto> to reach a label that was too deeply
nested for Perl to reach.  Perl is doing you a favor by refusing.

=item tell() on unopened file

(W) You tried to use the tell() function on a filehandle that was either
never opened or has since been closed.

=item Test on unopened file E<lt>%sE<gt>

(W) You tried to invoke a file test operator on a filehandle that isn't
open.  Check your logic.  See also L<perlfunc/-X>.

=item That use of $[ is unsupported

(F) Assignment to C<$[> is now strictly circumscribed, and interpreted as
a compiler directive.  You may say only one of

    $[ = 0;
    $[ = 1;
    ...
    local $[ = 0;
    local $[ = 1;
    ...

This is to prevent the problem of one module changing the array base
out from under another module inadvertently.  See L<perlvar/$[>.

=item The %s function is unimplemented

The function indicated isn't implemented on this architecture, according
to the probings of Configure.

=item The crypt() function is unimplemented due to excessive paranoia

(F) Configure couldn't find the crypt() function on your machine,
probably because your vendor didn't supply it, probably because they
think the U.S. Government thinks it's a secret, or at least that they
will continue to pretend that it is.  And if you quote me on that, I
will deny it.

=item The stat preceding C<-l _> wasn't an lstat

(F) It makes no sense to test the current stat buffer for symbolic linkhood
if the last stat that wrote to the stat buffer already went past
the symlink to get to the real file.  Use an actual filename instead.

=item times not implemented

(F) Your version of the C library apparently doesn't do times().  I suspect
you're not running on Unix.

=item Too few args to syscall

(F) There has to be at least one argument to syscall() to specify the
system call to call, silly dilly.

=item Too late for "B<-T>" option

(X) The #! line (or local equivalent) in a Perl script contains the
B<-T> option, but Perl was not invoked with B<-T> in its command line.
This is an error because, by the time Perl discovers a B<-T> in a
script, it's too late to properly taint everything from the environment.
So Perl gives up.

If the Perl script is being executed as a command using the #!
mechanism (or its local equivalent), this error can usually be fixed
by editing the #! line so that the B<-T> option is a part of Perl's
first argument: e.g. change C<perl -n -T> to C<perl -T -n>.

If the Perl script is being executed as C<perl scriptname>, then the
B<-T> option must appear on the command line: C<perl -T scriptname>.

=item Too late for "-%s" option

(X) The #! line (or local equivalent) in a Perl script contains the
B<-M> or B<-m> option.  This is an error because B<-M> and B<-m> options
are not intended for use inside scripts.  Use the C<use> pragma instead.

=item Too many ('s

=item Too many )'s

(A) You've accidentally run your script through B<csh> instead
of Perl.  Check the #! line, or manually feed your script into
Perl yourself.

=item Too many args to syscall

(F) Perl supports a maximum of only 14 args to syscall().

=item Too many arguments for %s

(F) The function requires fewer arguments than you specified.

=item trailing \ in regexp

(F) The regular expression ends with an unbackslashed backslash.  Backslash
it.   See L<perlre>.

=item Transliteration pattern not terminated

(F) The lexer couldn't find the interior delimiter of a tr/// or tr[][]
or y/// or y[][] construct.  Missing the leading C<$> from variables
C<$tr> or C<$y> may cause this error.

=item Transliteration replacement not terminated

(F) The lexer couldn't find the final delimiter of a tr/// or tr[][]
construct.

=item truncate not implemented

(F) Your machine doesn't implement a file truncation mechanism that
Configure knows about.

=item Type of arg %d to %s must be %s (not %s)

(F) This function requires the argument in that position to be of a
certain type.  Arrays must be @NAME or C<@{EXPR}>.  Hashes must be
%NAME or C<%{EXPR}>.  No implicit dereferencing is allowed--use the
{EXPR} forms as an explicit dereference.  See L<perlref>.

=item umask: argument is missing initial 0

(W) A umask of 222 is incorrect.  It should be 0222, because octal
literals always start with 0 in Perl, as in C.

=item umask not implemented

(F) Your machine doesn't implement the umask function and you tried
to use it to restrict permissions for yourself (EXPR & 0700).

=item Unable to create sub named "%s"

(F) You attempted to create or access a subroutine with an illegal name.

=item Unbalanced context: %d more PUSHes than POPs

(W) The exit code detected an internal inconsistency in how many execution
contexts were entered and left.

=item Unbalanced saves: %d more saves than restores

(W) The exit code detected an internal inconsistency in how many
values were temporarily localized.

=item Unbalanced scopes: %d more ENTERs than LEAVEs

(W) The exit code detected an internal inconsistency in how many blocks
were entered and left.

=item Unbalanced tmps: %d more allocs than frees

(W) The exit code detected an internal inconsistency in how many mortal
scalars were allocated and freed.

=item Undefined format "%s" called

(F) The format indicated doesn't seem to exist.  Perhaps it's really in
another package?  See L<perlform>.

=item Undefined sort subroutine "%s" called

(F) The sort comparison routine specified doesn't seem to exist.  Perhaps
it's in a different package?  See L<perlfunc/sort>.

=item Undefined subroutine &%s called

(F) The subroutine indicated hasn't been defined, or if it was, it
has since been undefined.

=item Undefined subroutine called

(F) The anonymous subroutine you're trying to call hasn't been defined,
or if it was, it has since been undefined.

=item Undefined subroutine in sort

(F) The sort comparison routine specified is declared but doesn't seem to
have been defined yet.  See L<perlfunc/sort>.

=item Undefined top format "%s" called

(F) The format indicated doesn't seem to exist.  Perhaps it's really in
another package?  See L<perlform>.

=item Undefined value assigned to typeglob

(W) An undefined value was assigned to a typeglob, a la C<*foo = undef>.
This does nothing.  It's possible that you really mean C<undef *foo>.

=item unexec of %s into %s failed!

(F) The unexec() routine failed for some reason.  See your local FSF
representative, who probably put it there in the first place.

=item Unknown BYTEORDER

(F) There are no byte-swapping functions for a machine with this byte order.

=item unmatched () in regexp

(F) Unbackslashed parentheses must always be balanced in regular
expressions.  If you're a vi user, the % key is valuable for finding
the matching parenthesis.  See L<perlre>.

=item Unmatched right bracket

(F) The lexer counted more closing curly brackets (braces) than opening
ones, so you're probably missing an opening bracket.  As a general
rule, you'll find the missing one (so to speak) near the place you were
last editing.

=item unmatched [] in regexp

(F) The brackets around a character class must match.  If you wish to
include a closing bracket in a character class, backslash it or put it first.
See L<perlre>.

=item Unquoted string "%s" may clash with future reserved word

(W) You used a bareword that might someday be claimed as a reserved word.
It's best to put such a word in quotes, or capitalize it somehow, or insert
an underbar into it.  You might also declare it as a subroutine.

=item Unrecognized character %s

(F) The Perl parser has no idea what to do with the specified character
in your Perl script (or eval).  Perhaps you tried to run a compressed
script, a binary program, or a directory as a Perl program.

=item Unrecognized signal name "%s"

(F) You specified a signal name to the kill() function that was not recognized.
Say C<kill -l> in your shell to see the valid signal names on your system.

=item Unrecognized switch: -%s  (-h will show valid options)

(F) You specified an illegal option to Perl.  Don't do that.
(If you think you didn't do that, check the #! line to see if it's
supplying the bad switch on your behalf.)

=item Unsuccessful %s on filename containing newline

(W) A file operation was attempted on a filename, and that operation
failed, PROBABLY because the filename contained a newline, PROBABLY
because you forgot to chop() or chomp() it off.  See L<perlfunc/chomp>.

=item Unsupported directory function "%s" called

(F) Your machine doesn't support opendir() and readdir().

=item Unsupported function fork

(F) Your version of executable does not support forking.

Note that under some systems, like OS/2, there may be different flavors of
Perl executables, some of which may support fork, some not. Try changing
the name you call Perl by to C<perl_>, C<perl__>, and so on.

=item Unsupported function %s

(F) This machine doesn't implement the indicated function, apparently.
At least, Configure doesn't think so.

=item Unsupported socket function "%s" called

(F) Your machine doesn't support the Berkeley socket mechanism, or at
least that's what Configure thought.

=item Unterminated E<lt>E<gt> operator

(F) The lexer saw a left angle bracket in a place where it was expecting
a term, so it's looking for the corresponding right angle bracket, and not
finding it.  Chances are you left some needed parentheses out earlier in
the line, and you really meant a "less than".

=item Use of "$$<digit>" to mean "${$}<digit>" is deprecated

(D) Perl versions before 5.004 misinterpreted any type marker followed
by "$" and a digit.  For example, "$$0" was incorrectly taken to mean
"${$}0" instead of "${$0}".  This bug is (mostly) fixed in Perl 5.004.

However, the developers of Perl 5.004 could not fix this bug completely,
because at least two widely-used modules depend on the old meaning of
"$$0" in a string.  So Perl 5.004 still interprets "$$<digit>" in the
old (broken) way inside strings; but it generates this message as a
warning.  And in Perl 5.005, this special treatment will cease.

=item Use of $# is deprecated

(D) This was an ill-advised attempt to emulate a poorly defined B<awk> feature.
Use an explicit printf() or sprintf() instead.

=item Use of $* is deprecated

(D) This variable magically turned on multi-line pattern matching, both for
you and for any luckless subroutine that you happen to call.  You should
use the new C<//m> and C<//s> modifiers now to do that without the dangerous
action-at-a-distance effects of C<$*>.

=item Use of %s in printf format not supported

(F) You attempted to use a feature of printf that is accessible from
only C.  This usually means there's a better way to do it in Perl.

=item Use of bare E<lt>E<lt> to mean E<lt>E<lt>"" is deprecated

(D) You are now encouraged to use the explicitly quoted form if you
wish to use an empty line as the terminator of the here-document.

=item Use of implicit split to @_ is deprecated

(D) It makes a lot of work for the compiler when you clobber a
subroutine's argument list, so it's better if you assign the results of
a split() explicitly to an array (or list).

=item Use of inherited AUTOLOAD for non-method %s() is deprecated

(D) As an (ahem) accidental feature, C<AUTOLOAD> subroutines are looked
up as methods (using the C<@ISA> hierarchy) even when the subroutines to
be autoloaded were called as plain functions (e.g. C<Foo::bar()>), not
as methods (e.g. C<Foo-E<gt>bar()> or C<$obj-E<gt>bar()>).

This bug will be rectified in Perl 5.005, which will use method lookup
only for methods' C<AUTOLOAD>s.  However, there is a significant base
of existing code that may be using the old behavior.  So, as an
interim step, Perl 5.004 issues an optional warning when non-methods
use inherited C<AUTOLOAD>s.

The simple rule is:  Inheritance will not work when autoloading
non-methods.  The simple fix for old code is:  In any module that used to
depend on inheriting C<AUTOLOAD> for non-methods from a base class named
C<BaseClass>, execute C<*AUTOLOAD = \&BaseClass::AUTOLOAD> during startup.

In code that currently says C<use AutoLoader; @ISA = qw(AutoLoader);> you
should remove AutoLoader from @ISA and change C<use AutoLoader;> to
C<use AutoLoader 'AUTOLOAD';>.

=item Use of reserved word "%s" is deprecated

(D) The indicated bareword is a reserved word.  Future versions of perl
may use it as a keyword, so you're better off either explicitly quoting
the word in a manner appropriate for its context of use, or using a
different name altogether.  The warning can be suppressed for subroutine
names by either adding a C<&> prefix, or using a package qualifier,
e.g. C<&our()>, or C<Foo::our()>.

=item Use of %s is deprecated

(D) The construct indicated is no longer recommended for use, generally
because there's a better way to do it, and also because the old way has
bad side effects.

=item Use of uninitialized value

(W) An undefined value was used as if it were already defined.  It was
interpreted as a "" or a 0, but maybe it was a mistake.  To suppress this
warning assign an initial value to your variables.

=item Useless use of "re" pragma

(W) You did C<use re;> without any arguments.   That isn't very useful.

=item Useless use of %s in void context

(W) You did something without a side effect in a context that does nothing
with the return value, such as a statement that doesn't return a value
from a block, or the left side of a scalar comma operator.  Very often
this points not to stupidity on your part, but a failure of Perl to parse
your program the way you thought it would.  For example, you'd get this
if you mixed up your C precedence with Python precedence and said

    $one, $two = 1, 2;

when you meant to say

    ($one, $two) = (1, 2);

Another common error is to use ordinary parentheses to construct a list
reference when you should be using square or curly brackets, for
example, if you say

    $array = (1,2);

when you should have said

    $array = [1,2];

The square brackets explicitly turn a list value into a scalar value,
while parentheses do not.  So when a parenthesized list is evaluated in
a scalar context, the comma is treated like C's comma operator, which
throws away the left argument, which is not what you want.  See
L<perlref> for more on this.

=item untie attempted while %d inner references still exist

(W) A copy of the object returned from C<tie> (or C<tied>) was still
valid when C<untie> was called.

=item Value of %s can be "0"; test with defined()

(W) In a conditional expression, you used <HANDLE>, <*> (glob), C<each()>,
or C<readdir()> as a boolean value.  Each of these constructs can return a
value of "0"; that would make the conditional expression false, which is
probably not what you intended.  When using these constructs in conditional
expressions, test their values with the C<defined> operator.

=item Variable "%s" is not imported%s

(F) While "use strict" in effect, you referred to a global variable
that you apparently thought was imported from another module, because
something else of the same name (usually a subroutine) is exported
by that module.  It usually means you put the wrong funny character
on the front of your variable.

=item Variable "%s" may be unavailable

(W) An inner (nested) I<anonymous> subroutine is inside a I<named>
subroutine, and outside that is another subroutine; and the anonymous
(innermost) subroutine is referencing a lexical variable defined in
the outermost subroutine.  For example:

   sub outermost { my $a; sub middle { sub { $a } } }

If the anonymous subroutine is called or referenced (directly or
indirectly) from the outermost subroutine, it will share the variable
as you would expect.  But if the anonymous subroutine is called or
referenced when the outermost subroutine is not active, it will see
the value of the shared variable as it was before and during the
*first* call to the outermost subroutine, which is probably not what
you want.

In these circumstances, it is usually best to make the middle
subroutine anonymous, using the C<sub {}> syntax.  Perl has specific
support for shared variables in nested anonymous subroutines; a named
subroutine in between interferes with this feature.

=item Variable "%s" will not stay shared

(W) An inner (nested) I<named> subroutine is referencing a lexical
variable defined in an outer subroutine.

When the inner subroutine is called, it will probably see the value of
the outer subroutine's variable as it was before and during the
*first* call to the outer subroutine; in this case, after the first
call to the outer subroutine is complete, the inner and outer
subroutines will no longer share a common value for the variable.  In
other words, the variable will no longer be shared.

Furthermore, if the outer subroutine is anonymous and references a
lexical variable outside itself, then the outer and inner subroutines
will I<never> share the given variable.

This problem can usually be solved by making the inner subroutine
anonymous, using the C<sub {}> syntax.  When inner anonymous subs that
reference variables in outer subroutines are called or referenced,
they are automatically rebound to the current values of such
variables.

=item Variable syntax

(A) You've accidentally run your script through B<csh> instead
of Perl.  Check the #! line, or manually feed your script into
Perl yourself.

=item perl: warning: Setting locale failed.

(S) The whole warning message will look something like:

	perl: warning: Setting locale failed.
	perl: warning: Please check that your locale settings:
	        LC_ALL = "En_US",
	        LANG = (unset)
	    are supported and installed on your system.
	perl: warning: Falling back to the standard locale ("C").

Exactly what were the failed locale settings varies.  In the above the
settings were that the LC_ALL was "En_US" and the LANG had no value.
This error means that Perl detected that you and/or your system
administrator have set up the so-called variable system but Perl could
not use those settings.  This was not dead serious, fortunately: there
is a "default locale" called "C" that Perl can and will use, the
script will be run.  Before you really fix the problem, however, you
will get the same error message each time you run Perl.  How to really
fix the problem can be found in L<perllocale> section B<LOCALE PROBLEMS>.

=item Warning: something's wrong

(W) You passed warn() an empty string (the equivalent of C<warn "">) or
you called it with no args and C<$_> was empty.

=item Warning: unable to close filehandle %s properly

(S) The implicit close() done by an open() got an error indication on the
close().  This usually indicates your file system ran out of disk space.

=item Warning: Use of "%s" without parentheses is ambiguous

(S) You wrote a unary operator followed by something that looks like a
binary operator that could also have been interpreted as a term or
unary operator.  For instance, if you know that the rand function
has a default argument of 1.0, and you write

    rand + 5;

you may THINK you wrote the same thing as

    rand() + 5;

but in actual fact, you got

    rand(+5);

So put in parentheses to say what you really mean.

=item Write on closed filehandle

(W) The filehandle you're writing to got itself closed sometime before now.
Check your logic flow.

=item X outside of string

(F) You had a pack template that specified a relative position before
the beginning of the string being unpacked.  See L<perlfunc/pack>.

=item x outside of string

(F) You had a pack template that specified a relative position after
the end of the string being unpacked.  See L<perlfunc/pack>.

=item Xsub "%s" called in sort

(F) The use of an external subroutine as a sort comparison is not yet supported.

=item Xsub called in sort

(F) The use of an external subroutine as a sort comparison is not yet supported.

=item You can't use C<-l> on a filehandle

(F) A filehandle represents an opened file, and when you opened the file it
already went past any symlink you are presumably trying to look for.
Use a filename instead.

=item YOU HAVEN'T DISABLED SET-ID SCRIPTS IN THE KERNEL YET!

(F) And you probably never will, because you probably don't have the
sources to your kernel, and your vendor probably doesn't give a rip
about what you want.  Your best bet is to use the wrapsuid script in
the eg directory to put a setuid C wrapper around your script.

=item You need to quote "%s"

(W) You assigned a bareword as a signal handler name.  Unfortunately, you
already have a subroutine of that name declared, which means that Perl 5
will try to call the subroutine when the assignment is executed, which is
probably not what you want.  (If it IS what you want, put an & in front.)

=item [gs]etsockopt() on closed fd

(W) You tried to get or set a socket option on a closed socket.
Did you forget to check the return value of your socket() call?
See L<perlfunc/getsockopt>.

=item \1 better written as $1

(W) Outside of patterns, backreferences live on as variables.  The use
of backslashes is grandfathered on the right-hand side of a
substitution, but stylistically it's better to use the variable form
because other Perl programmers will expect it, and it works better
if there are more than 9 backreferences.

=item '|' and 'E<lt>' may not both be specified on command line

(F) An error peculiar to VMS.  Perl does its own command line redirection, and
found that STDIN was a pipe, and that you also tried to redirect STDIN using
'E<lt>'.  Only one STDIN stream to a customer, please.

=item '|' and 'E<gt>' may not both be specified on command line

(F) An error peculiar to VMS.  Perl does its own command line redirection, and
thinks you tried to redirect stdout both to a file and into a pipe to another
command.  You need to choose one or the other, though nothing's stopping you
from piping into a program or Perl script which 'splits' output into two
streams, such as

    open(OUT,">$ARGV[0]") or die "Can't write to $ARGV[0]: $!";
    while (<STDIN>) {
        print;
        print OUT;
    }
    close OUT;

=item Got an error from DosAllocMem

(P) An error peculiar to OS/2.  Most probably you're using an obsolete
version of Perl, and this should not happen anyway.

=item Malformed PERLLIB_PREFIX

(F) An error peculiar to OS/2.  PERLLIB_PREFIX should be of the form

    prefix1;prefix2

or

    prefix1 prefix2

with nonempty prefix1 and prefix2.  If C<prefix1> is indeed a prefix
of a builtin library search path, prefix2 is substituted.  The error
may appear if components are not found, or are too long.  See
"PERLLIB_PREFIX" in F<README.os2>.

=item PERL_SH_DIR too long

(F) An error peculiar to OS/2. PERL_SH_DIR is the directory to find the
C<sh>-shell in.  See "PERL_SH_DIR" in F<README.os2>.

=item Process terminated by SIG%s

(W) This is a standard message issued by OS/2 applications, while *nix
applications die in silence.  It is considered a feature of the OS/2
port.  One can easily disable this by appropriate sighandlers, see
L<perlipc/"Signals">.  See also "Process terminated by SIGTERM/SIGINT"
in F<README.os2>.

=back

