import os
import shutil
import datetime
from PIL import Image
from mock import Mock, patch
from test.xbmctestcase import XBMCTestCase

TEST_FOLDER = os.path.dirname(__file__)
DATA_FOLDER = os.path.join(TEST_FOLDER, 'data')
RESULTS_FOLDER = os.path.join(TEST_FOLDER, 'results')
OBSERVATIONHOURLY = os.path.join(DATA_FOLDER, 'observationhourly.json')
OBSERVATIONHOURLY2 = os.path.join(DATA_FOLDER, 'observationhourly2.json')
FORECASTDAILY = os.path.join(DATA_FOLDER, 'forecastdaily.json')
FORECAST3HOURLY = os.path.join(DATA_FOLDER, 'forecast3hourly.json')
FORECASTTEXT = os.path.join(DATA_FOLDER, 'forecasttext.json')
FORECASTLAYERCAPABILITIES = os.path.join(DATA_FOLDER, 'forecastlayercapabilities.json')
OBSERVATIONLAYERCAPABILITIES = os.path.join(DATA_FOLDER, 'observationlayercapabilities.json')
CORRUPTFORECASTLAYERCAPABILITIES = os.path.join(DATA_FOLDER, 'corruptforecastlayercapabilities.json')
CORRUPTOBSERVATIONLAYERCAPABILITIES = os.path.join(DATA_FOLDER, 'corruptobservationlayercapabilities.json')
FORECASTSITELIST = os.path.join(DATA_FOLDER, 'forecastsitelist.json')
TEXTSITELIST = os.path.join(DATA_FOLDER, 'textsitelist.json')
GEOIP = os.path.join(DATA_FOLDER, 'ip-api.json')
EMPTY_FILE = os.path.join(DATA_FOLDER, 'empty.json')
GOOGLE_SURFACE_IMAGE = os.path.join(DATA_FOLDER, 'google_surface.png')
GOOGLE_MARKER_IMAGE = os.path.join(DATA_FOLDER, 'google_marker.png')
PRECIPITATION_LAYER_IMAGE = os.path.join(RESULTS_FOLDER, 'precipitation_layer.png')

PRECIPITATION_LAYER_HOUR0_URL = 'http://datapoint.metoffice.gov.uk/public/data/layer/wxfcs/Precipitation_Rate/png?RUN=2014-03-19T09:00:00Z&FORECAST=0&key=12345'
PRECIPITATION_LAYER_HOUR36_URL = 'http://datapoint.metoffice.gov.uk/public/data/layer/wxfcs/Precipitation_Rate/png?RUN=2014-03-19T09:00:00Z&FORECAST=36&key=12345'
OBSERVATION_LAYER0_URL = 'http://datapoint.metoffice.gov.uk/public/data/layer/wxobs/RADAR_UK_Composite_Highres/png?TIME=2014-04-01T16:30:00Z&key=12345'
OBSERVATION_LAYER1_URL = 'http://datapoint.metoffice.gov.uk/public/data/layer/wxobs/RADAR_UK_Composite_Highres/png?TIME=2014-04-01T13:30:00Z&key=12345'
class TestProperties(XBMCTestCase):
    def setUp(self):
        super(TestProperties, self).setUp()
        self.xbmc.translatePath.side_effect = lambda x: x

        self.settings = {'ApiKey' : '12345',
                         'GeoLocation' : 'true',
                         'GeoIPProvider' : '0',
                         'ForecastLocation' : 'CAMBRIDGE NIAB',
                         'ForecastLocationID' : '99123',
                         'ForecastLocationLatitude' : '52.245',
                         'ForecastLocationLongitude' : '0.103',
                         'ObservationLocation' : 'BEDFORD',
                         'ObservationLocationID' : '3560',
                         'RegionalLocation' : 'Wales',
                         'RegionalLocationID' : '516'
                         }

        self.window_properties = {'ForecastMap.LayerSelection' : 'Rainfall',
                                  'ObservationMap.LayerSelection' : 'Rainfall',
                                  'ForecastMap.Slider' : '0',
                                  'ObservationMap.Slider' : '0',
                                  'Weather.CurrentView' : 'Doesnt matter'}

        addon = self.xbmcaddon.Addon.return_value
        addon.getSetting.side_effect = self.mock_getSetting
        addon.setSetting.side_effect = self.mock_setSetting

        window = self.xbmcgui.Window.return_value
        window.getProperty.side_effect = self.mock_getProperty
        window.setProperty.side_effect = self.mock_setProperty

        from metoffice import constants
        self.constants = constants

        #create a disposable area for testing
        try:
            os.mkdir(RESULTS_FOLDER)
        except OSError:
            pass

        shutil.copy(os.path.join(DATA_FOLDER, 'precipitation_layer.png'), os.path.join(RESULTS_FOLDER, 'precipitation_layer.png'))

    def mock_getSetting(self, key):
        return self.settings[key]

    def mock_setSetting(self, key, value):
        self.settings[key] = value
        
    def mock_getProperty(self, key):
        return self.window_properties[key]
    
    def mock_setProperty(self, key, value):
        self.window_properties[key] = value

    def mock_get(self, url, expiry_callback, resource_callback=None):
        return {
                self.constants.FORECAST_SITELIST_URL: FORECASTSITELIST,
                self.constants.DAILY_LOCATION_FORECAST_URL: FORECASTDAILY,
                self.constants.THREEHOURLY_LOCATION_FORECAST_URL: FORECAST3HOURLY,
                self.constants.FORECAST_LAYER_CAPABILITIES_URL: FORECASTLAYERCAPABILITIES,
                self.constants.OBSERVATION_LAYER_CAPABILITIES_URL: OBSERVATIONLAYERCAPABILITIES,
                self.constants.TEXT_FORECAST_URL: FORECASTTEXT,
                self.constants.HOURLY_LOCATION_OBSERVATION_URL: OBSERVATIONHOURLY,
                self.constants.GEOIP_PROVIDER['url']:GEOIP,
                self.constants.GOOGLE_SURFACE: GOOGLE_SURFACE_IMAGE,
                self.constants.GOOGLE_MARKER: GOOGLE_MARKER_IMAGE,
                PRECIPITATION_LAYER_HOUR0_URL: PRECIPITATION_LAYER_IMAGE,
                PRECIPITATION_LAYER_HOUR36_URL: PRECIPITATION_LAYER_IMAGE,
                OBSERVATION_LAYER0_URL: PRECIPITATION_LAYER_IMAGE,
                OBSERVATION_LAYER1_URL: PRECIPITATION_LAYER_IMAGE,
                }[url]

    def mock_panelbusy(self, pane):
        def decorate(f):
            def wrapper(*args, **kwargs):
                return f(*args, **kwargs)
            return wrapper
        return decorate

    @patch('metoffice.utilities.panelbusy')
    @patch('metoffice.urlcache.URLCache')
    def test_observation(self, mock_cache, mock_panelbusy):
        mock_panelbusy.side_effect = self.mock_panelbusy
        mock_cache.return_value.__enter__.return_value.get = Mock(side_effect=self.mock_get)
        from metoffice import properties        
        properties.observation()
        self.assertIn('HourlyObservation.IssuedAt', self.window_properties)
        self.assertEqual(self.window_properties['HourlyObservation.IssuedAt'], '17:00 Thu 06 Mar 2014')
        self.assertIn('Current.Condition', self.window_properties)
        self.assertEqual(self.window_properties['Current.Condition'], 'Cloudy')
        self.assertIn('Current.Visibility', self.window_properties)
        self.assertEqual(self.window_properties['Current.Visibility'], '45000')
        self.assertIn('Current.Pressure', self.window_properties)
        self.assertEqual(self.window_properties['Current.Pressure'], '1021')
        self.assertIn('Current.Temperature', self.window_properties)
        self.assertEqual(self.window_properties['Current.Temperature'], '10')
        self.assertIn('Current.Wind', self.window_properties)
        self.assertEqual(self.window_properties['Current.Wind'], '23.0')
        self.assertIn('Current.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['Current.WindDirection'], 'SW')
        self.assertIn('Current.WindGust', self.window_properties)
        self.assertEqual(self.window_properties['Current.WindGust'], 'n/a')
        self.assertIn('Current.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['Current.OutlookIcon'], '26.png')
        self.assertIn('Current.FanartCode', self.window_properties)
        self.assertEqual(self.window_properties['Current.FanartCode'], '26.png')
        self.assertIn('Current.DewPoint', self.window_properties)
        self.assertEqual(self.window_properties['Current.DewPoint'], '6')
        self.assertIn('Current.Humidity', self.window_properties)
        self.assertEqual(self.window_properties['Current.Humidity'], '79')

        #Test exceptions when reports don't contain list items.
        mock_cache.return_value.__enter__.return_value.get = Mock(return_value=OBSERVATIONHOURLY2)
        properties.observation()
        self.assertIn('HourlyObservation.IssuedAt', self.window_properties)
        self.assertEqual(self.window_properties['HourlyObservation.IssuedAt'], '17:00 Thu 06 Mar 2014')
        self.assertIn('Current.Condition', self.window_properties)
        self.assertEqual(self.window_properties['Current.Condition'], 'Sunny')
        self.assertIn('Current.Visibility', self.window_properties)
        self.assertEqual(self.window_properties['Current.Visibility'], '45000')
        self.assertIn('Current.Pressure', self.window_properties)
        self.assertEqual(self.window_properties['Current.Pressure'], '1021')
        self.assertIn('Current.Temperature', self.window_properties)
        self.assertEqual(self.window_properties['Current.Temperature'], '10')
        self.assertIn('Current.Wind', self.window_properties)
        self.assertEqual(self.window_properties['Current.Wind'], '16.0')
        self.assertIn('Current.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['Current.WindDirection'], 'WSW')
        self.assertIn('Current.WindGust', self.window_properties)
        self.assertEqual(self.window_properties['Current.WindGust'], 'n/a')
        self.assertIn('Current.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['Current.OutlookIcon'], '32.png')
        self.assertIn('Current.FanartCode', self.window_properties)
        self.assertEqual(self.window_properties['Current.FanartCode'], '32.png')
        self.assertIn('Current.DewPoint', self.window_properties)
        self.assertEqual(self.window_properties['Current.DewPoint'], '4')
        self.assertIn('Current.Humidity', self.window_properties)
        self.assertEqual(self.window_properties['Current.Humidity'], '66')

        #Test exception handling when given json without proper keys
        mock_cache.return_value.__enter__.return_value.get = Mock(return_value=EMPTY_FILE)
        with self.assertRaises(KeyError) as cm:
            properties.observation()
        self.assertEqual(('Key Error in JSON File',
                          "Key 'SiteRep' not found while processing file from url:",
                          self.constants.HOURLY_LOCATION_OBSERVATION_URL), cm.exception.args)

    @patch('metoffice.utilities.panelbusy')
    @patch('metoffice.urlcache.URLCache')
    def test_daily(self, mock_cache, mock_panelbusy):
        mock_panelbusy.side_effect = self.mock_panelbusy
        mock_cache.return_value.__enter__.return_value.get = Mock(side_effect=self.mock_get)
        from metoffice import properties
        properties.daily()
        self.assertIn('DailyForecast.IssuedAt', self.window_properties)
        self.assertEqual(self.window_properties['DailyForecast.IssuedAt'], '14:00 Mon 24 Feb 2014')
        self.assertIn('Day0.Title', self.window_properties)
        self.assertEqual(self.window_properties['Day0.Title'], 'Mon')
        self.assertIn('Day0.HighTemp', self.window_properties)
        self.assertEqual(self.window_properties['Day0.HighTemp'], '13')
        self.assertIn('Day0.LowTemp', self.window_properties)
        self.assertEqual(self.window_properties['Day0.LowTemp'], '7')
        self.assertIn('Day0.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['Day0.WindSpeed'], '13')
        self.assertIn('Day0.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['Day0.WindDirection'], 'ssw')
        self.assertIn('Day0.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['Day0.Outlook'], 'Light Rain')
        self.assertIn('Day0.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['Day0.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/11.png')

        self.assertIn('Day1.Title', self.window_properties)
        self.assertEqual(self.window_properties['Day1.Title'], 'Tue')
        self.assertIn('Day1.HighTemp', self.window_properties)
        self.assertEqual(self.window_properties['Day1.HighTemp'], '12')
        self.assertIn('Day1.LowTemp', self.window_properties)
        self.assertEqual(self.window_properties['Day1.LowTemp'], '3')
        self.assertIn('Day1.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['Day1.WindSpeed'], '18')
        self.assertIn('Day1.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['Day1.WindDirection'], 'ssw')
        self.assertIn('Day1.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['Day1.Outlook'], 'Sunny')
        self.assertIn('Day1.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['Day1.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/32.png')

        self.assertIn('Day2.Title', self.window_properties)
        self.assertEqual(self.window_properties['Day2.Title'], 'Wed')
        self.assertIn('Day2.HighTemp', self.window_properties)
        self.assertEqual(self.window_properties['Day2.HighTemp'], '11')
        self.assertIn('Day2.LowTemp', self.window_properties)
        self.assertEqual(self.window_properties['Day2.LowTemp'], '4')
        self.assertIn('Day2.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['Day2.WindSpeed'], '13')
        self.assertIn('Day2.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['Day2.WindDirection'], 'wsw')
        self.assertIn('Day2.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['Day2.Outlook'], 'Sunny')
        self.assertIn('Day2.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['Day2.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/32.png')

        self.assertIn('Day3.Title', self.window_properties)
        self.assertEqual(self.window_properties['Day3.Title'], 'Thu')
        self.assertIn('Day3.HighTemp', self.window_properties)
        self.assertEqual(self.window_properties['Day3.HighTemp'], '11')
        self.assertIn('Day3.LowTemp', self.window_properties)
        self.assertEqual(self.window_properties['Day3.LowTemp'], '3')
        self.assertIn('Day3.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['Day3.WindSpeed'], '16')
        self.assertIn('Day3.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['Day3.WindDirection'], 'wsw')
        self.assertIn('Day3.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['Day3.Outlook'], 'Heavy Rain')
        self.assertIn('Day3.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['Day3.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/40.png')

        self.assertIn('Day4.Title', self.window_properties)
        self.assertEqual(self.window_properties['Day4.Title'], 'Fri')
        self.assertIn('Day4.HighTemp', self.window_properties)
        self.assertEqual(self.window_properties['Day4.HighTemp'], '9')
        self.assertIn('Day4.LowTemp', self.window_properties)
        self.assertEqual(self.window_properties['Day4.LowTemp'], '2')
        self.assertIn('Day4.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['Day4.WindSpeed'], '11')
        self.assertIn('Day4.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['Day4.WindDirection'], 'sw')
        self.assertIn('Day4.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['Day4.Outlook'], 'Light Rain')
        self.assertIn('Day4.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['Day4.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/11.png')

        #Test exception handling when given json without proper keys
        mock_cache.return_value.__enter__.return_value.get = Mock(return_value=EMPTY_FILE)
        with self.assertRaises(KeyError) as cm:
            properties.daily()
        self.assertEqual(('Key Error in JSON File',
                          "Key 'SiteRep' not found while processing file from url:",
                          self.constants.DAILY_LOCATION_FORECAST_URL), cm.exception.args)

    @patch('metoffice.utilities.panelbusy')
    @patch('metoffice.urlcache.URLCache')
    def test_threehourly(self, mock_cache, mock_panelbusy):
        mock_panelbusy.side_effect = self.mock_panelbusy
        mock_cache.return_value.__enter__.return_value.get = Mock(side_effect=self.mock_get)
        from metoffice import properties
        properties.threehourly()
        self.assertIn('3HourlyForecast.IssuedAt', self.window_properties)
        self.assertEqual(self.window_properties['3HourlyForecast.IssuedAt'], '16:00 Sat 01 Mar 2014')

        self.assertIn('3Hourly0.Day', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly0.Day'], 'Sat')
        self.assertIn('3Hourly0.Time', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly0.Time'], '12:00')
        self.assertIn('3Hourly0.ActualTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly0.ActualTemp'], '6')
        self.assertIn('3Hourly0.FeelsLikeTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly0.FeelsLikeTemp'], '4')
        self.assertIn('3Hourly0.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly0.WindSpeed'], '4')
        self.assertIn('3Hourly0.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly0.WindDirection'], 'nnw')
        self.assertIn('3Hourly0.GustSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly0.GustSpeed'], '7')
        self.assertIn('3Hourly0.UVIndex', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly0.UVIndex'], '1')
        self.assertIn('3Hourly0.Precipitation', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly0.Precipitation'], '6')
        self.assertIn('3Hourly0.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly0.Outlook'], 'Cloudy')
        self.assertIn('3Hourly0.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly0.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/26.png')

        self.assertIn('3Hourly1.Day', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly1.Day'], 'Sat')
        self.assertIn('3Hourly1.Time', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly1.Time'], '15:00')
        self.assertIn('3Hourly1.ActualTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly1.ActualTemp'], '8')
        self.assertIn('3Hourly1.FeelsLikeTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly1.FeelsLikeTemp'], '6')
        self.assertIn('3Hourly1.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly1.WindSpeed'], '7')
        self.assertIn('3Hourly1.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly1.WindDirection'], 'wsw')
        self.assertIn('3Hourly1.GustSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly1.GustSpeed'], '11')
        self.assertIn('3Hourly1.UVIndex', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly1.UVIndex'], '1')
        self.assertIn('3Hourly1.Precipitation', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly1.Precipitation'], '6')
        self.assertIn('3Hourly1.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly1.Outlook'], 'Cloudy')
        self.assertIn('3Hourly1.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly1.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/26.png')

        self.assertIn('3Hourly2.Day', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly2.Day'], 'Sat')
        self.assertIn('3Hourly2.Time', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly2.Time'], '18:00')
        self.assertIn('3Hourly2.ActualTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly2.ActualTemp'], '6')
        self.assertIn('3Hourly2.FeelsLikeTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly2.FeelsLikeTemp'], '5')
        self.assertIn('3Hourly2.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly2.WindSpeed'], '2')
        self.assertIn('3Hourly2.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly2.WindDirection'], 'wsw')
        self.assertIn('3Hourly2.GustSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly2.GustSpeed'], '7')
        self.assertIn('3Hourly2.UVIndex', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly2.UVIndex'], '0')
        self.assertIn('3Hourly2.Precipitation', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly2.Precipitation'], '5')
        self.assertIn('3Hourly2.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly2.Outlook'], 'Cloudy')
        self.assertIn('3Hourly2.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly2.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/26.png')

        self.assertIn('3Hourly3.Day', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly3.Day'], 'Sat')
        self.assertIn('3Hourly3.Time', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly3.Time'], '21:00')
        self.assertIn('3Hourly3.ActualTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly3.ActualTemp'], '5')
        self.assertIn('3Hourly3.FeelsLikeTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly3.FeelsLikeTemp'], '3')
        self.assertIn('3Hourly3.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly3.WindSpeed'], '4')
        self.assertIn('3Hourly3.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly3.WindDirection'], 'ssw')
        self.assertIn('3Hourly3.GustSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly3.GustSpeed'], '9')
        self.assertIn('3Hourly3.UVIndex', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly3.UVIndex'], '0')
        self.assertIn('3Hourly3.Precipitation', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly3.Precipitation'], '0')
        self.assertIn('3Hourly3.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly3.Outlook'], 'Clear')
        self.assertIn('3Hourly3.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly3.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/31.png')

        self.assertIn('3Hourly4.Day', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly4.Day'], 'Sun')
        self.assertIn('3Hourly4.Time', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly4.Time'], '00:00')
        self.assertIn('3Hourly4.ActualTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly4.ActualTemp'], '3')
        self.assertIn('3Hourly4.FeelsLikeTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly4.FeelsLikeTemp'], '1')
        self.assertIn('3Hourly4.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly4.WindSpeed'], '7')
        self.assertIn('3Hourly4.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly4.WindDirection'], 'ssw')
        self.assertIn('3Hourly4.GustSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly4.GustSpeed'], '16')
        self.assertIn('3Hourly4.UVIndex', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly4.UVIndex'], '0')
        self.assertIn('3Hourly4.Precipitation', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly4.Precipitation'], '3')
        self.assertIn('3Hourly4.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly4.Outlook'], 'Clear')
        self.assertIn('3Hourly4.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly4.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/31.png')

        self.assertIn('3Hourly5.Day', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly5.Day'], 'Sun')
        self.assertIn('3Hourly5.Time', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly5.Time'], '03:00')
        self.assertIn('3Hourly5.ActualTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly5.ActualTemp'], '3')
        self.assertIn('3Hourly5.FeelsLikeTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly5.FeelsLikeTemp'], '0')
        self.assertIn('3Hourly5.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly5.WindSpeed'], '9')
        self.assertIn('3Hourly5.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly5.WindDirection'], 'ssw')
        self.assertIn('3Hourly5.GustSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly5.GustSpeed'], '20')
        self.assertIn('3Hourly5.UVIndex', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly5.UVIndex'], '0')
        self.assertIn('3Hourly5.Precipitation', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly5.Precipitation'], '5')
        self.assertIn('3Hourly5.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly5.Outlook'], 'Clear')
        self.assertIn('3Hourly5.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly5.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/31.png')

        self.assertIn('3Hourly6.Day', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly6.Day'], 'Sun')
        self.assertIn('3Hourly6.Time', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly6.Time'], '06:00')
        self.assertIn('3Hourly6.ActualTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly6.ActualTemp'], '4')
        self.assertIn('3Hourly6.FeelsLikeTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly6.FeelsLikeTemp'], '0')
        self.assertIn('3Hourly6.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly6.WindSpeed'], '11')
        self.assertIn('3Hourly6.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly6.WindDirection'], 'ssw')
        self.assertIn('3Hourly6.GustSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly6.GustSpeed'], '25')
        self.assertIn('3Hourly6.UVIndex', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly6.UVIndex'], '0')
        self.assertIn('3Hourly6.Precipitation', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly6.Precipitation'], '8')
        self.assertIn('3Hourly6.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly6.Outlook'], 'Cloudy')
        self.assertIn('3Hourly6.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly6.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/26.png')

        self.assertIn('3Hourly7.Day', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly7.Day'], 'Sun')
        self.assertIn('3Hourly7.Time', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly7.Time'], '09:00')
        self.assertIn('3Hourly7.ActualTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly7.ActualTemp'], '6')
        self.assertIn('3Hourly7.FeelsLikeTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly7.FeelsLikeTemp'], '3')
        self.assertIn('3Hourly7.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly7.WindSpeed'], '13')
        self.assertIn('3Hourly7.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly7.WindDirection'], 'ssw')
        self.assertIn('3Hourly7.GustSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly7.GustSpeed'], '25')
        self.assertIn('3Hourly7.UVIndex', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly7.UVIndex'], '1')
        self.assertIn('3Hourly7.Precipitation', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly7.Precipitation'], '5')
        self.assertIn('3Hourly7.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly7.Outlook'], 'Cloudy')
        self.assertIn('3Hourly7.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly7.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/26.png')

        self.assertIn('3Hourly8.Day', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly8.Day'], 'Sun')
        self.assertIn('3Hourly8.Time', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly8.Time'], '12:00')
        self.assertIn('3Hourly8.ActualTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly8.ActualTemp'], '9')
        self.assertIn('3Hourly8.FeelsLikeTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly8.FeelsLikeTemp'], '5')
        self.assertIn('3Hourly8.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly8.WindSpeed'], '18')
        self.assertIn('3Hourly8.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly8.WindDirection'], 's')
        self.assertIn('3Hourly8.GustSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly8.GustSpeed'], '31')
        self.assertIn('3Hourly8.UVIndex', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly8.UVIndex'], '1')
        self.assertIn('3Hourly8.Precipitation', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly8.Precipitation'], '5')
        self.assertIn('3Hourly8.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly8.Outlook'], 'Cloudy')
        self.assertIn('3Hourly8.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly8.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/26.png')

        self.assertIn('3Hourly9.Day', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly9.Day'], 'Sun')
        self.assertIn('3Hourly9.Time', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly9.Time'], '15:00')
        self.assertIn('3Hourly9.ActualTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly9.ActualTemp'], '9')
        self.assertIn('3Hourly9.FeelsLikeTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly9.FeelsLikeTemp'], '5')
        self.assertIn('3Hourly9.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly9.WindSpeed'], '20')
        self.assertIn('3Hourly9.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly9.WindDirection'], 's')
        self.assertIn('3Hourly9.GustSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly9.GustSpeed'], '36')
        self.assertIn('3Hourly9.UVIndex', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly9.UVIndex'], '1')
        self.assertIn('3Hourly9.Precipitation', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly9.Precipitation'], '31')
        self.assertIn('3Hourly9.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly9.Outlook'], 'Cloudy')
        self.assertIn('3Hourly9.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly9.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/26.png')

        self.assertIn('3Hourly10.Day', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly10.Day'], 'Sun')
        self.assertIn('3Hourly10.Time', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly10.Time'], '18:00')
        self.assertIn('3Hourly10.ActualTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly10.ActualTemp'], '8')
        self.assertIn('3Hourly10.FeelsLikeTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly10.FeelsLikeTemp'], '3')
        self.assertIn('3Hourly10.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly10.WindSpeed'], '20')
        self.assertIn('3Hourly10.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly10.WindDirection'], 's')
        self.assertIn('3Hourly10.GustSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly10.GustSpeed'], '38')
        self.assertIn('3Hourly10.UVIndex', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly10.UVIndex'], '0')
        self.assertIn('3Hourly10.Precipitation', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly10.Precipitation'], '95')
        self.assertIn('3Hourly10.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly10.Outlook'], 'Heavy Rain')
        self.assertIn('3Hourly10.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly10.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/40.png')

        self.assertIn('3Hourly11.Day', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly11.Day'], 'Sun')
        self.assertIn('3Hourly11.Time', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly11.Time'], '21:00')
        self.assertIn('3Hourly11.ActualTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly11.ActualTemp'], '7')
        self.assertIn('3Hourly11.FeelsLikeTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly11.FeelsLikeTemp'], '3')
        self.assertIn('3Hourly11.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly11.WindSpeed'], '16')
        self.assertIn('3Hourly11.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly11.WindDirection'], 's')
        self.assertIn('3Hourly11.GustSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly11.GustSpeed'], '29')
        self.assertIn('3Hourly11.UVIndex', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly11.UVIndex'], '0')
        self.assertIn('3Hourly11.Precipitation', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly11.Precipitation'], '96')
        self.assertIn('3Hourly11.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly11.Outlook'], 'Heavy Rain')
        self.assertIn('3Hourly11.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly11.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/40.png')

        self.assertIn('3Hourly12.Day', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly12.Day'], 'Mon')
        self.assertIn('3Hourly12.Time', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly12.Time'], '00:00')
        self.assertIn('3Hourly12.ActualTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly12.ActualTemp'], '5')
        self.assertIn('3Hourly12.FeelsLikeTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly12.FeelsLikeTemp'], '1')
        self.assertIn('3Hourly12.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly12.WindSpeed'], '13')
        self.assertIn('3Hourly12.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly12.WindDirection'], 'wsw')
        self.assertIn('3Hourly12.GustSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly12.GustSpeed'], '22')
        self.assertIn('3Hourly12.UVIndex', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly12.UVIndex'], '0')
        self.assertIn('3Hourly12.Precipitation', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly12.Precipitation'], '54')
        self.assertIn('3Hourly12.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly12.Outlook'], 'Light Rain')
        self.assertIn('3Hourly12.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly12.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/11.png')

        self.assertIn('3Hourly13.Day', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly13.Day'], 'Mon')
        self.assertIn('3Hourly13.Time', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly13.Time'], '03:00')
        self.assertIn('3Hourly13.ActualTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly13.ActualTemp'], '3')
        self.assertIn('3Hourly13.FeelsLikeTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly13.FeelsLikeTemp'], '-1')
        self.assertIn('3Hourly13.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly13.WindSpeed'], '11')
        self.assertIn('3Hourly13.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly13.WindDirection'], 'ssw')
        self.assertIn('3Hourly13.GustSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly13.GustSpeed'], '18')
        self.assertIn('3Hourly13.UVIndex', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly13.UVIndex'], '0')
        self.assertIn('3Hourly13.Precipitation', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly13.Precipitation'], '13')
        self.assertIn('3Hourly13.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly13.Outlook'], 'Clear')
        self.assertIn('3Hourly13.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly13.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/31.png')

        self.assertIn('3Hourly14.Day', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly14.Day'], 'Mon')
        self.assertIn('3Hourly14.Time', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly14.Time'], '06:00')
        self.assertIn('3Hourly14.ActualTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly14.ActualTemp'], '2')
        self.assertIn('3Hourly14.FeelsLikeTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly14.FeelsLikeTemp'], '-2')
        self.assertIn('3Hourly14.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly14.WindSpeed'], '11')
        self.assertIn('3Hourly14.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly14.WindDirection'], 's')
        self.assertIn('3Hourly14.GustSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly14.GustSpeed'], '20')
        self.assertIn('3Hourly14.UVIndex', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly14.UVIndex'], '0')
        self.assertIn('3Hourly14.Precipitation', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly14.Precipitation'], '5')
        self.assertIn('3Hourly14.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly14.Outlook'], 'Cloudy')
        self.assertIn('3Hourly14.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly14.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/26.png')

        self.assertIn('3Hourly15.Day', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly15.Day'], 'Mon')
        self.assertIn('3Hourly15.Time', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly15.Time'], '09:00')
        self.assertIn('3Hourly15.ActualTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly15.ActualTemp'], '5')
        self.assertIn('3Hourly15.FeelsLikeTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly15.FeelsLikeTemp'], '1')
        self.assertIn('3Hourly15.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly15.WindSpeed'], '11')
        self.assertIn('3Hourly15.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly15.WindDirection'], 's')
        self.assertIn('3Hourly15.GustSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly15.GustSpeed'], '22')
        self.assertIn('3Hourly15.UVIndex', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly15.UVIndex'], '1')
        self.assertIn('3Hourly15.Precipitation', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly15.Precipitation'], '33')
        self.assertIn('3Hourly15.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly15.Outlook'], 'Light Rain')
        self.assertIn('3Hourly15.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly15.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/11.png')

        self.assertIn('3Hourly16.Day', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly16.Day'], 'Mon')
        self.assertIn('3Hourly16.Time', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly16.Time'], '12:00')
        self.assertIn('3Hourly16.ActualTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly16.ActualTemp'], '7')
        self.assertIn('3Hourly16.FeelsLikeTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly16.FeelsLikeTemp'], '4')
        self.assertIn('3Hourly16.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly16.WindSpeed'], '11')
        self.assertIn('3Hourly16.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly16.WindDirection'], 's')
        self.assertIn('3Hourly16.GustSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly16.GustSpeed'], '22')
        self.assertIn('3Hourly16.UVIndex', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly16.UVIndex'], '1')
        self.assertIn('3Hourly16.Precipitation', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly16.Precipitation'], '65')
        self.assertIn('3Hourly16.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly16.Outlook'], 'Heavy Rain')
        self.assertIn('3Hourly16.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly16.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/40.png')

        self.assertIn('3Hourly17.Day', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly17.Day'], 'Mon')
        self.assertIn('3Hourly17.Time', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly17.Time'], '15:00')
        self.assertIn('3Hourly17.ActualTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly17.ActualTemp'], '7')
        self.assertIn('3Hourly17.FeelsLikeTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly17.FeelsLikeTemp'], '4')
        self.assertIn('3Hourly17.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly17.WindSpeed'], '7')
        self.assertIn('3Hourly17.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly17.WindDirection'], 'ssw')
        self.assertIn('3Hourly17.GustSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly17.GustSpeed'], '16')
        self.assertIn('3Hourly17.UVIndex', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly17.UVIndex'], '1')
        self.assertIn('3Hourly17.Precipitation', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly17.Precipitation'], '48')
        self.assertIn('3Hourly17.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly17.Outlook'], 'Light Rain')
        self.assertIn('3Hourly17.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly17.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/11.png')

        self.assertIn('3Hourly18.Day', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly18.Day'], 'Mon')
        self.assertIn('3Hourly18.Time', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly18.Time'], '18:00')
        self.assertIn('3Hourly18.ActualTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly18.ActualTemp'], '5')
        self.assertIn('3Hourly18.FeelsLikeTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly18.FeelsLikeTemp'], '3')
        self.assertIn('3Hourly18.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly18.WindSpeed'], '7')
        self.assertIn('3Hourly18.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly18.WindDirection'], 'wsw')
        self.assertIn('3Hourly18.GustSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly18.GustSpeed'], '11')
        self.assertIn('3Hourly18.UVIndex', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly18.UVIndex'], '0')
        self.assertIn('3Hourly18.Precipitation', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly18.Precipitation'], '46')
        self.assertIn('3Hourly18.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly18.Outlook'], 'Light Rain')
        self.assertIn('3Hourly18.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly18.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/45.png')

        self.assertIn('3Hourly19.Day', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly19.Day'], 'Mon')
        self.assertIn('3Hourly19.Time', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly19.Time'], '21:00')
        self.assertIn('3Hourly19.ActualTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly19.ActualTemp'], '4')
        self.assertIn('3Hourly19.FeelsLikeTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly19.FeelsLikeTemp'], '1')
        self.assertIn('3Hourly19.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly19.WindSpeed'], '9')
        self.assertIn('3Hourly19.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly19.WindDirection'], 'w')
        self.assertIn('3Hourly19.GustSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly19.GustSpeed'], '13')
        self.assertIn('3Hourly19.UVIndex', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly19.UVIndex'], '0')
        self.assertIn('3Hourly19.Precipitation', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly19.Precipitation'], '13')
        self.assertIn('3Hourly19.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly19.Outlook'], 'Cloudy')
        self.assertIn('3Hourly19.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly19.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/26.png')

        self.assertIn('3Hourly20.Day', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly20.Day'], 'Tue')
        self.assertIn('3Hourly20.Time', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly20.Time'], '00:00')
        self.assertIn('3Hourly20.ActualTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly20.ActualTemp'], '3')
        self.assertIn('3Hourly20.FeelsLikeTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly20.FeelsLikeTemp'], '0')
        self.assertIn('3Hourly20.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly20.WindSpeed'], '9')
        self.assertIn('3Hourly20.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly20.WindDirection'], 'wnw')
        self.assertIn('3Hourly20.GustSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly20.GustSpeed'], '18')
        self.assertIn('3Hourly20.UVIndex', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly20.UVIndex'], '0')
        self.assertIn('3Hourly20.Precipitation', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly20.Precipitation'], '2')
        self.assertIn('3Hourly20.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly20.Outlook'], 'Partly Cloudy')
        self.assertIn('3Hourly20.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly20.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/29.png')

        self.assertIn('3Hourly21.Day', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly21.Day'], 'Tue')
        self.assertIn('3Hourly21.Time', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly21.Time'], '03:00')
        self.assertIn('3Hourly21.ActualTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly21.ActualTemp'], '3')
        self.assertIn('3Hourly21.FeelsLikeTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly21.FeelsLikeTemp'], '-1')
        self.assertIn('3Hourly21.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly21.WindSpeed'], '9')
        self.assertIn('3Hourly21.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly21.WindDirection'], 'wnw')
        self.assertIn('3Hourly21.GustSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly21.GustSpeed'], '18')
        self.assertIn('3Hourly21.UVIndex', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly21.UVIndex'], '0')
        self.assertIn('3Hourly21.Precipitation', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly21.Precipitation'], '2')
        self.assertIn('3Hourly21.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly21.Outlook'], 'Partly Cloudy')
        self.assertIn('3Hourly21.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly21.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/29.png')

        self.assertIn('3Hourly22.Day', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly22.Day'], 'Tue')
        self.assertIn('3Hourly22.Time', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly22.Time'], '06:00')
        self.assertIn('3Hourly22.ActualTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly22.ActualTemp'], '2')
        self.assertIn('3Hourly22.FeelsLikeTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly22.FeelsLikeTemp'], '-1')
        self.assertIn('3Hourly22.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly22.WindSpeed'], '9')
        self.assertIn('3Hourly22.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly22.WindDirection'], 'w')
        self.assertIn('3Hourly22.GustSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly22.GustSpeed'], '16')
        self.assertIn('3Hourly22.UVIndex', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly22.UVIndex'], '0')
        self.assertIn('3Hourly22.Precipitation', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly22.Precipitation'], '2')
        self.assertIn('3Hourly22.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly22.Outlook'], 'Partly Cloudy')
        self.assertIn('3Hourly22.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly22.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/29.png')

        self.assertIn('3Hourly23.Day', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly23.Day'], 'Tue')
        self.assertIn('3Hourly23.Time', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly23.Time'], '09:00')
        self.assertIn('3Hourly23.ActualTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly23.ActualTemp'], '5')
        self.assertIn('3Hourly23.FeelsLikeTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly23.FeelsLikeTemp'], '2')
        self.assertIn('3Hourly23.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly23.WindSpeed'], '11')
        self.assertIn('3Hourly23.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly23.WindDirection'], 'w')
        self.assertIn('3Hourly23.GustSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly23.GustSpeed'], '20')
        self.assertIn('3Hourly23.UVIndex', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly23.UVIndex'], '1')
        self.assertIn('3Hourly23.Precipitation', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly23.Precipitation'], '6')
        self.assertIn('3Hourly23.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly23.Outlook'], 'Partly Cloudy')
        self.assertIn('3Hourly23.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly23.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/30.png')

        self.assertIn('3Hourly24.Day', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly24.Day'], 'Tue')
        self.assertIn('3Hourly24.Time', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly24.Time'], '12:00')
        self.assertIn('3Hourly24.ActualTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly24.ActualTemp'], '8')
        self.assertIn('3Hourly24.FeelsLikeTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly24.FeelsLikeTemp'], '5')
        self.assertIn('3Hourly24.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly24.WindSpeed'], '11')
        self.assertIn('3Hourly24.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly24.WindDirection'], 'w')
        self.assertIn('3Hourly24.GustSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly24.GustSpeed'], '22')
        self.assertIn('3Hourly24.UVIndex', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly24.UVIndex'], '1')
        self.assertIn('3Hourly24.Precipitation', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly24.Precipitation'], '7')
        self.assertIn('3Hourly24.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly24.Outlook'], 'Partly Cloudy')
        self.assertIn('3Hourly24.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly24.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/30.png')

        self.assertIn('3Hourly25.Day', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly25.Day'], 'Tue')
        self.assertIn('3Hourly25.Time', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly25.Time'], '15:00')
        self.assertIn('3Hourly25.ActualTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly25.ActualTemp'], '8')
        self.assertIn('3Hourly25.FeelsLikeTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly25.FeelsLikeTemp'], '6')
        self.assertIn('3Hourly25.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly25.WindSpeed'], '11')
        self.assertIn('3Hourly25.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly25.WindDirection'], 'w')
        self.assertIn('3Hourly25.GustSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly25.GustSpeed'], '18')
        self.assertIn('3Hourly25.UVIndex', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly25.UVIndex'], '1')
        self.assertIn('3Hourly25.Precipitation', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly25.Precipitation'], '10')
        self.assertIn('3Hourly25.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly25.Outlook'], 'Sunny')
        self.assertIn('3Hourly25.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly25.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/32.png')

        self.assertIn('3Hourly26.Day', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly26.Day'], 'Tue')
        self.assertIn('3Hourly26.Time', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly26.Time'], '18:00')
        self.assertIn('3Hourly26.ActualTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly26.ActualTemp'], '6')
        self.assertIn('3Hourly26.FeelsLikeTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly26.FeelsLikeTemp'], '4')
        self.assertIn('3Hourly26.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly26.WindSpeed'], '7')
        self.assertIn('3Hourly26.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly26.WindDirection'], 'wsw')
        self.assertIn('3Hourly26.GustSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly26.GustSpeed'], '11')
        self.assertIn('3Hourly26.UVIndex', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly26.UVIndex'], '0')
        self.assertIn('3Hourly26.Precipitation', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly26.Precipitation'], '8')
        self.assertIn('3Hourly26.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly26.Outlook'], 'Clear')
        self.assertIn('3Hourly26.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly26.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/31.png')

        self.assertIn('3Hourly27.Day', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly27.Day'], 'Tue')
        self.assertIn('3Hourly27.Time', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly27.Time'], '21:00')
        self.assertIn('3Hourly27.ActualTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly27.ActualTemp'], '5')
        self.assertIn('3Hourly27.FeelsLikeTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly27.FeelsLikeTemp'], '2')
        self.assertIn('3Hourly27.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly27.WindSpeed'], '7')
        self.assertIn('3Hourly27.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly27.WindDirection'], 'sw')
        self.assertIn('3Hourly27.GustSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly27.GustSpeed'], '13')
        self.assertIn('3Hourly27.UVIndex', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly27.UVIndex'], '0')
        self.assertIn('3Hourly27.Precipitation', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly27.Precipitation'], '11')
        self.assertIn('3Hourly27.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly27.Outlook'], 'Partly Cloudy')
        self.assertIn('3Hourly27.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly27.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/29.png')

        self.assertIn('3Hourly28.Day', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly28.Day'], 'Wed')
        self.assertIn('3Hourly28.Time', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly28.Time'], '00:00')
        self.assertIn('3Hourly28.ActualTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly28.ActualTemp'], '4')
        self.assertIn('3Hourly28.FeelsLikeTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly28.FeelsLikeTemp'], '1')
        self.assertIn('3Hourly28.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly28.WindSpeed'], '9')
        self.assertIn('3Hourly28.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly28.WindDirection'], 'sw')
        self.assertIn('3Hourly28.GustSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly28.GustSpeed'], '16')
        self.assertIn('3Hourly28.UVIndex', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly28.UVIndex'], '0')
        self.assertIn('3Hourly28.Precipitation', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly28.Precipitation'], '16')
        self.assertIn('3Hourly28.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly28.Outlook'], 'Cloudy')
        self.assertIn('3Hourly28.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly28.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/26.png')

        self.assertIn('3Hourly29.Day', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly29.Day'], 'Wed')
        self.assertIn('3Hourly29.Time', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly29.Time'], '03:00')
        self.assertIn('3Hourly29.ActualTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly29.ActualTemp'], '4')
        self.assertIn('3Hourly29.FeelsLikeTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly29.FeelsLikeTemp'], '1')
        self.assertIn('3Hourly29.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly29.WindSpeed'], '9')
        self.assertIn('3Hourly29.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly29.WindDirection'], 'sw')
        self.assertIn('3Hourly29.GustSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly29.GustSpeed'], '16')
        self.assertIn('3Hourly29.UVIndex', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly29.UVIndex'], '0')
        self.assertIn('3Hourly29.Precipitation', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly29.Precipitation'], '23')
        self.assertIn('3Hourly29.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly29.Outlook'], 'Cloudy')
        self.assertIn('3Hourly29.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly29.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/26.png')

        self.assertIn('3Hourly30.Day', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly30.Day'], 'Wed')
        self.assertIn('3Hourly30.Time', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly30.Time'], '06:00')
        self.assertIn('3Hourly30.ActualTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly30.ActualTemp'], '4')
        self.assertIn('3Hourly30.FeelsLikeTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly30.FeelsLikeTemp'], '1')
        self.assertIn('3Hourly30.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly30.WindSpeed'], '11')
        self.assertIn('3Hourly30.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly30.WindDirection'], 'sw')
        self.assertIn('3Hourly30.GustSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly30.GustSpeed'], '20')
        self.assertIn('3Hourly30.UVIndex', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly30.UVIndex'], '0')
        self.assertIn('3Hourly30.Precipitation', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly30.Precipitation'], '24')
        self.assertIn('3Hourly30.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly30.Outlook'], 'Overcast')
        self.assertIn('3Hourly30.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly30.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/26.png')

        self.assertIn('3Hourly31.Day', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly31.Day'], 'Wed')
        self.assertIn('3Hourly31.Time', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly31.Time'], '09:00')
        self.assertIn('3Hourly31.ActualTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly31.ActualTemp'], '6')
        self.assertIn('3Hourly31.FeelsLikeTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly31.FeelsLikeTemp'], '2')
        self.assertIn('3Hourly31.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly31.WindSpeed'], '13')
        self.assertIn('3Hourly31.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly31.WindDirection'], 'wsw')
        self.assertIn('3Hourly31.GustSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly31.GustSpeed'], '29')
        self.assertIn('3Hourly31.UVIndex', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly31.UVIndex'], '1')
        self.assertIn('3Hourly31.Precipitation', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly31.Precipitation'], '55')
        self.assertIn('3Hourly31.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly31.Outlook'], 'Light Rain')
        self.assertIn('3Hourly31.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly31.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/11.png')

        self.assertIn('3Hourly32.Day', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly32.Day'], 'Wed')
        self.assertIn('3Hourly32.Time', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly32.Time'], '12:00')
        self.assertIn('3Hourly32.ActualTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly32.ActualTemp'], '8')
        self.assertIn('3Hourly32.FeelsLikeTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly32.FeelsLikeTemp'], '3')
        self.assertIn('3Hourly32.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly32.WindSpeed'], '18')
        self.assertIn('3Hourly32.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly32.WindDirection'], 'w')
        self.assertIn('3Hourly32.GustSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly32.GustSpeed'], '38')
        self.assertIn('3Hourly32.UVIndex', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly32.UVIndex'], '1')
        self.assertIn('3Hourly32.Precipitation', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly32.Precipitation'], '37')
        self.assertIn('3Hourly32.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly32.Outlook'], 'Light Rain')
        self.assertIn('3Hourly32.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly32.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/11.png')

        self.assertIn('3Hourly33.Day', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly33.Day'], 'Wed')
        self.assertIn('3Hourly33.Time', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly33.Time'], '15:00')
        self.assertIn('3Hourly33.ActualTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly33.ActualTemp'], '8')
        self.assertIn('3Hourly33.FeelsLikeTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly33.FeelsLikeTemp'], '3')
        self.assertIn('3Hourly33.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly33.WindSpeed'], '18')
        self.assertIn('3Hourly33.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly33.WindDirection'], 'w')
        self.assertIn('3Hourly33.GustSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly33.GustSpeed'], '36')
        self.assertIn('3Hourly33.UVIndex', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly33.UVIndex'], '1')
        self.assertIn('3Hourly33.Precipitation', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly33.Precipitation'], '14')
        self.assertIn('3Hourly33.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly33.Outlook'], 'Partly Cloudy')
        self.assertIn('3Hourly33.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly33.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/30.png')

        self.assertIn('3Hourly34.Day', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly34.Day'], 'Wed')
        self.assertIn('3Hourly34.Time', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly34.Time'], '18:00')
        self.assertIn('3Hourly34.ActualTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly34.ActualTemp'], '7')
        self.assertIn('3Hourly34.FeelsLikeTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly34.FeelsLikeTemp'], '2')
        self.assertIn('3Hourly34.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly34.WindSpeed'], '13')
        self.assertIn('3Hourly34.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly34.WindDirection'], 'w')
        self.assertIn('3Hourly34.GustSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly34.GustSpeed'], '27')
        self.assertIn('3Hourly34.UVIndex', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly34.UVIndex'], '0')
        self.assertIn('3Hourly34.Precipitation', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly34.Precipitation'], '6')
        self.assertIn('3Hourly34.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly34.Outlook'], 'Partly Cloudy')
        self.assertIn('3Hourly34.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly34.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/29.png')

        self.assertIn('3Hourly35.Day', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly35.Day'], 'Wed')
        self.assertIn('3Hourly35.Time', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly35.Time'], '21:00')
        self.assertIn('3Hourly35.ActualTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly35.ActualTemp'], '5')
        self.assertIn('3Hourly35.FeelsLikeTemp', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly35.FeelsLikeTemp'], '1')
        self.assertIn('3Hourly35.WindSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly35.WindSpeed'], '13')
        self.assertIn('3Hourly35.WindDirection', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly35.WindDirection'], 'wsw')
        self.assertIn('3Hourly35.GustSpeed', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly35.GustSpeed'], '25')
        self.assertIn('3Hourly35.UVIndex', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly35.UVIndex'], '0')
        self.assertIn('3Hourly35.Precipitation', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly35.Precipitation'], '7')
        self.assertIn('3Hourly35.Outlook', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly35.Outlook'], 'Partly Cloudy')
        self.assertIn('3Hourly35.OutlookIcon', self.window_properties)
        self.assertEqual(self.window_properties['3Hourly35.OutlookIcon'], 'special://xbmc/addons/resource.images.weathericons.default/resources/29.png')

        #Test exception handling when given json without proper keys
        mock_cache.return_value.__enter__.return_value.get = Mock(return_value=EMPTY_FILE)
        with self.assertRaises(KeyError) as cm:
            properties.threehourly()
        self.assertEqual(('Key Error in JSON File',
                          "Key 'SiteRep' not found while processing file from url:",
                          self.constants.THREEHOURLY_LOCATION_FORECAST_URL), cm.exception.args)

    @patch('metoffice.utilities.panelbusy')
    @patch('metoffice.urlcache.URLCache')
    def test_text(self, mock_cache, mock_panelbusy):
        mock_panelbusy.side_effect = self.mock_panelbusy
        mock_cache.return_value.__enter__.return_value.get = Mock(side_effect=self.mock_get)
        from metoffice import properties
        properties.text()
        self.assertIn('TextForecast.IssuedAt', self.window_properties)
        self.assertEqual(self.window_properties['TextForecast.IssuedAt'], '16:00 Mon 24 Feb 2014')

        self.assertIn('Text.Paragraph0.Title', self.window_properties)
        self.assertEqual(self.window_properties['Text.Paragraph0.Title'], 'Headline')
        self.assertIn('Text.Paragraph0.Content', self.window_properties)
        self.assertEqual(self.window_properties['Text.Paragraph0.Content'], 'Rain clearing eastwards, showers following, with increasing winds.')
        self.assertIn('Text.Paragraph1.Title', self.window_properties)
        self.assertEqual(self.window_properties['Text.Paragraph1.Title'], 'This Evening and Tonight')
        self.assertIn('Text.Paragraph1.Content', self.window_properties)
        self.assertEqual(self.window_properties['Text.Paragraph1.Content'], 'Rain arriving in the far west around dusk will clear eastwards overnight, this heaviest in the west and over high ground, where winds will also become strong. Mild, with clear spells and scattered showers following. Minimum Temperature 5C.')
        self.assertIn('Text.Paragraph2.Title', self.window_properties)
        self.assertEqual(self.window_properties['Text.Paragraph2.Title'], 'Tuesday')
        self.assertIn('Text.Paragraph2.Content', self.window_properties)
        self.assertEqual(self.window_properties['Text.Paragraph2.Content'], 'Some dry and bright weather is likely at times but also scattered blustery, heavy showers. Remaining windy, especially around exposed coasts and hills where gales are likely. Maximum Temperature 9C.')
        self.assertIn('Text.Paragraph3.Title', self.window_properties)
        self.assertEqual(self.window_properties['Text.Paragraph3.Title'], 'Wednesday to Friday')
        self.assertIn('Text.Paragraph3.Content', self.window_properties)
        self.assertEqual(self.window_properties['Text.Paragraph3.Content'], 'Sunny spells and lighter winds on Wednesday, some showers along the coast. Wet and windy overnight, turning showery on Thursday and Friday, becoming wintry over hills.')
        self.assertIn('Text.Paragraph4.Title', self.window_properties)
        self.assertEqual(self.window_properties['Text.Paragraph4.Title'], 'Saturday 1 Mar 2014 to Monday 10 Mar 2014')
        self.assertIn('Text.Paragraph4.Content', self.window_properties)
        self.assertEqual(self.window_properties['Text.Paragraph4.Content'], 'The weekend will start unsettled with showers or longer spells of rain, with some heavier bursts at first. This will be most persistent in the far southeast and far north, with a risk of hill snow in the north. There will be some drier slots too, especially on Sunday with a risk of local frost and icy surfaces. Temperatures near normal. Through the next week it will remain unsettled in northern parts, with further rain or showers, and some hill snow. It will be mainly dry but fairly cloudy towards the south with isolated patchy frost. During the middle part of the week rain may spread southwards for a time, before turning wet and windy in the northwest again later, with a risk of gales.')
        self.assertIn('Text.Paragraph5.Title', self.window_properties)
        self.assertEqual(self.window_properties['Text.Paragraph5.Title'], 'Tuesday 11 Mar 2014 to Tuesday 25 Mar 2014')
        self.assertIn('Text.Paragraph5.Content', self.window_properties)
        self.assertEqual(self.window_properties['Text.Paragraph5.Content'], 'Current indications suggest a more typically unsettled pattern across the United Kingdom through much of March. Through this period we can expect to see fairly average conditions, which would mean spells of wet and windy weather, mostly in the north and west, but still some decent sunny spells in between. The best of the drier, brighter conditions is most likely in the south and east of the UK. Temperatures are likely to be around average, which may lead to more frequent incidences of frost compared to recent weeks.')

        #Test exception handling when given json without proper keys
        mock_cache.return_value.__enter__.return_value.get = Mock(return_value=EMPTY_FILE)
        with self.assertRaises(KeyError) as cm:
            properties.text()
        self.assertEqual(('Key Error in JSON File',
                          "Key 'RegionalFcst' not found while processing file from url:",
                          self.constants.TEXT_FORECAST_URL), cm.exception.args)

    @patch('metoffice.utilities.panelbusy')
    @patch('metoffice.urlcache.URLCache')
    def test_forecastlayer(self, mock_cache, mock_panelbusy):
        mock_panelbusy.side_effect = self.mock_panelbusy
        mock_cache.return_value.__enter__.return_value.get = Mock(side_effect=self.mock_get)

        from metoffice import properties
        properties.forecastlayer()
        self.assertIn('ForecastMap.Surface', self.window_properties)
        self.assertEqual(self.window_properties['ForecastMap.Surface'], GOOGLE_SURFACE_IMAGE)
        self.assertIn('ForecastMap.Marker', self.window_properties)
        self.assertEqual(self.window_properties['ForecastMap.Marker'], GOOGLE_MARKER_IMAGE)
        self.assertIn('ForecastMap.Slider', self.window_properties)
        self.assertEqual(self.window_properties['ForecastMap.Slider'], '0')
        self.assertIn('ForecastMap.IssuedAt', self.window_properties)
        self.assertEqual(self.window_properties['ForecastMap.IssuedAt'], '09:00 Wed 19 Mar 2014')
        self.assertIn('ForecastMap.MapTime', self.window_properties)
        self.assertEqual(self.window_properties['ForecastMap.MapTime'], '0900 Wed')
        self.assertIn('ForecastMap.Layer', self.window_properties)
        self.assertEqual(self.window_properties['ForecastMap.Layer'], PRECIPITATION_LAYER_IMAGE)
        self.assertIn('ForecastMap.IsFetched', self.window_properties)
        self.assertEqual(self.window_properties['ForecastMap.IsFetched'], 'true')

        #Test exception handling when given json without proper keys
        mock_cache.return_value.__enter__.return_value.get = Mock(return_value=EMPTY_FILE)
        with self.assertRaises(KeyError) as cm:
            properties.forecastlayer()
        self.assertEqual(('Key Error in JSON File',
                          "Key 'Layers' not found while processing file from url:",
                          self.constants.FORECAST_LAYER_CAPABILITIES_URL), cm.exception.args)

        #Test exception handling when given corrupt BaseURL in json
        #(We have provide partially valid json so execution can drop to the exception under test.)
        mock_cache.return_value.__enter__.return_value.get = Mock(return_value=CORRUPTFORECASTLAYERCAPABILITIES)
        with self.assertRaises(KeyError) as cm:
            properties.forecastlayer()
        self.assertEqual(('Key Error in JSON File',
                          "Key '$' not found while processing file from url:",
                          self.constants.FORECAST_LAYER_CAPABILITIES_URL), cm.exception.args)

        mock_cache.return_value.__enter__.return_value.get = Mock(side_effect=self.mock_get)
        mock_cache.reset_mock()

        #Test valid url is used when requesting with an invalid slider position
        properties.FORECASTMAP_SLIDER = '-9'
        properties.forecastlayer()
        self.assertEqual(PRECIPITATION_LAYER_HOUR0_URL,
                         mock_cache.return_value.__enter__.return_value.get.call_args_list[3][0][0])

        mock_cache.reset_mock()
        properties.FORECASTMAP_SLIDER = '45'
        properties.forecastlayer()
        self.assertEqual(PRECIPITATION_LAYER_HOUR36_URL,
                         mock_cache.return_value.__enter__.return_value.get.call_args_list[3][0][0])

        #Test response when given unknown layer name
        properties.FORECASTMAP_LAYER_SELECTION = 'Unknown'
        with self.assertRaises(Exception) as cm:
            properties.forecastlayer()
        self.assertEqual(('Error', "Couldn't find layer 'Unknown'"), cm.exception.args)

    @patch('metoffice.utilities.panelbusy')
    @patch('metoffice.urlcache.URLCache')
    def test_observationlayer(self, mock_cache, mock_panelbusy):
        mock_panelbusy.side_effect = self.mock_panelbusy
        mock_cache.return_value.__enter__.return_value.get = Mock(side_effect=self.mock_get)

        from metoffice import properties
        properties.observationlayer()
        self.assertIn('ObservationMap.Surface', self.window_properties)
        self.assertEqual(self.window_properties['ObservationMap.Surface'], GOOGLE_SURFACE_IMAGE)
        self.assertIn('ObservationMap.Marker', self.window_properties)
        self.assertEqual(self.window_properties['ObservationMap.Marker'], GOOGLE_MARKER_IMAGE)
        self.assertIn('ObservationMap.Slider', self.window_properties)
        self.assertEqual(self.window_properties['ObservationMap.Slider'], '0')
        self.assertIn('ObservationMap.IssuedAt', self.window_properties)
        self.assertEqual(self.window_properties['ObservationMap.IssuedAt'], '17:30 Tue 01 Apr 2014')
        self.assertIn('ObservationMap.MapTime', self.window_properties)
        self.assertEqual(self.window_properties['ObservationMap.MapTime'], '1730 Tue')
        self.assertIn('ObservationMap.Layer', self.window_properties)
        self.assertEqual(self.window_properties['ObservationMap.Layer'], PRECIPITATION_LAYER_IMAGE)
        self.assertIn('ObservationMap.IsFetched', self.window_properties)
        self.assertEqual(self.window_properties['ObservationMap.IsFetched'], 'true')

        #Test exception handling when given json without proper keys
        mock_cache.return_value.__enter__.return_value.get = Mock(return_value=EMPTY_FILE)
        with self.assertRaises(KeyError) as cm:
            properties.observationlayer()
        self.assertEqual(('Key Error in JSON File',
                          "Key 'Layers' not found while processing file from url:",
                          self.constants.OBSERVATION_LAYER_CAPABILITIES_URL), cm.exception.args)

        #Test exception handling when given corrupt BaseURL in json
        #(We have provide partially valid json so execution can drop to the exception under test.)
        mock_cache.return_value.__enter__.return_value.get = Mock(return_value=CORRUPTOBSERVATIONLAYERCAPABILITIES)
        with self.assertRaises(KeyError) as cm:
            properties.observationlayer()
        self.assertEqual(('Key Error in JSON File',
                          "Key '$' not found while processing file from url:",
                          self.constants.OBSERVATION_LAYER_CAPABILITIES_URL), cm.exception.args)

        mock_cache.return_value.__enter__.return_value.get = Mock(side_effect=self.mock_get)
        mock_cache.reset_mock()

        #Test valid url is used when requesting with an invalid slider position
        properties.OBSERVATIONMAP_SLIDER = '-9'
        properties.observationlayer()
        self.assertEqual(OBSERVATION_LAYER0_URL,
                         mock_cache.return_value.__enter__.return_value.get.call_args_list[3][0][0])

        mock_cache.reset_mock()
        properties.OBSERVATIONMAP_SLIDER = '45'
        properties.observationlayer()
        self.assertEqual(OBSERVATION_LAYER1_URL,
                         mock_cache.return_value.__enter__.return_value.get.call_args_list[3][0][0])

        #Test response when given unknown layer name
        properties.OBSERVATIONMAP_LAYER_SELECTION = 'Unknown'
        with self.assertRaises(Exception) as cm:
            properties.observationlayer()
        self.assertEqual(('Error', "Couldn't find layer 'Unknown'"), cm.exception.args)

    @patch('metoffice.utilities.panelbusy')
    @patch('metoffice.urlcache.URLCache')
    def test_daily_expiry(self, mock_cache, mock_panelbusy):
        mock_panelbusy.side_effect = self.mock_panelbusy
        mock_cache.return_value.__enter__.return_value.get = Mock(side_effect=self.mock_get)
        from metoffice import properties
        result = properties.daily_expiry(FORECASTDAILY)
        self.assertEqual(datetime.datetime(2014, 2, 24, 15, 30), result)

    @patch('metoffice.utilities.panelbusy')
    @patch('metoffice.urlcache.URLCache')
    def test_threehourly_expiry(self, mock_cache, mock_panelbusy):
        mock_panelbusy.side_effect = self.mock_panelbusy
        mock_cache.return_value.__enter__.return_value.get = Mock(side_effect=self.mock_get)
        from metoffice import properties
        result = properties.threehourly_expiry(FORECAST3HOURLY)
        self.assertEqual(datetime.datetime(2014, 3, 1, 17, 30), result)

    @patch('metoffice.utilities.panelbusy')
    @patch('metoffice.urlcache.URLCache')
    def test_text_expiry(self, mock_cache, mock_panelbusy):
        mock_panelbusy.side_effect = self.mock_panelbusy
        mock_cache.return_value.__enter__.return_value.get = Mock(side_effect=self.mock_get)
        from metoffice import properties
        result = properties.text_expiry(FORECASTTEXT)
        self.assertEqual(datetime.datetime(2014, 2, 25, 4, 0), result)

    @patch('metoffice.utilities.panelbusy')
    @patch('metoffice.urlcache.URLCache')
    def test_observation_expiry(self, mock_cache, mock_panelbusy):
        mock_panelbusy.side_effect = self.mock_panelbusy
        mock_cache.return_value.__enter__.return_value.get = Mock(side_effect=self.mock_get)
        from metoffice import properties
        result = properties.observation_expiry(OBSERVATIONHOURLY)
        self.assertEqual(datetime.datetime(2014, 3, 6, 18, 30), result)

    @patch('metoffice.utilities.panelbusy')
    @patch('metoffice.urlcache.URLCache')
    def test_forecastlayer_capabilities_expiry(self, mock_cache, mock_panelbusy):
        mock_panelbusy.side_effect = self.mock_panelbusy
        mock_cache.return_value.__enter__.return_value.get = Mock(side_effect=self.mock_get)
        from metoffice import properties
        result = properties.forecastlayer_capabilities_expiry(FORECASTLAYERCAPABILITIES)
        self.assertEqual(datetime.datetime(2014, 3, 19, 18, 0), result)

    @patch('metoffice.utilities.panelbusy')
    @patch('metoffice.urlcache.URLCache')
    def test_observationlayer_capabilities_expiry(self, mock_cache, mock_panelbusy):
        mock_panelbusy.side_effect = self.mock_panelbusy
        mock_cache.return_value.__enter__.return_value.get = Mock(side_effect=self.mock_get)
        from metoffice import properties
        result = properties.observationlayer_capabilities_expiry(OBSERVATIONLAYERCAPABILITIES)
        self.assertEqual(datetime.datetime(2014, 4, 1, 17, 0), result)

    @patch('metoffice.utilities.panelbusy')
    @patch('metoffice.urlcache.URLCache')
    def test_layer_image_resize_callback(self, mock_cache, mock_panelbusy):
        mock_panelbusy.side_effect = self.mock_panelbusy
        mock_cache.return_value.__enter__.return_value.get = Mock(side_effect=self.mock_get)

        #Assert that the pretend image in cache has not been resized
        img = Image.open(PRECIPITATION_LAYER_IMAGE)
        (width, height) = img.size
        self.assertEqual(500, width)
        self.assertEqual(500, height)

        from metoffice import properties
        properties.image_resize(PRECIPITATION_LAYER_IMAGE)
        img = Image.open(PRECIPITATION_LAYER_IMAGE)
        (width, height) = img.size
        self.assertEqual(420, width)
        self.assertEqual(460, height)

    def tearDown(self):
        super(TestProperties, self).tearDown()
        shutil.rmtree(RESULTS_FOLDER)