package io

import (
	"encoding/json"
	"io"

	hcl2json "github.com/tmccombs/hcl2json/convert"
)

func init() {
	registerInputHandler("hcl", func() InputHandler { return &HCLInputHandler{} })
	registerInputHandler("tf", func() InputHandler { return &HCLInputHandler{} })
}

// Declare conformance with InputHandler interface.
var _ InputHandler = &HCLInputHandler{}

// HCLInputHandler handles parsing HCL data.
type HCLInputHandler struct{}

// Name implements InputHandler.Name().
func (h *HCLInputHandler) Name() string {
	return "hcl"
}

// Parse implements InputHandler.Parse().
func (h *HCLInputHandler) Parse(reader io.Reader) (interface{}, error) {
	inputBytes, err := io.ReadAll(reader)
	if err != nil {
		return nil, err
	}

	jsonBytes, err := hcl2json.Bytes(inputBytes, "input.hcl", hcl2json.Options{Simplify: false})
	if err != nil {
		return nil, err
	}

	var parsed interface{}
	err = json.Unmarshal(jsonBytes, &parsed)
	if err != nil {
		return nil, err
	}

	return parsed, nil
}

// SetOption implements InputHandler.SetOption().
func (h *HCLInputHandler) SetOption(name string, value string) error {
	// The HCL handler does not have any options.
	return nil
}
