package io

import (
	"fmt"
	"io"
	"strconv"

	"git.sr.ht/~charles/rq/util"

	"github.com/alecthomas/chroma/v2/quick"
	"github.com/clbanning/mxj/v2"
)

type XMLListStyle string

const XMLListStyleEnumerated XMLListStyle = "enumerated"
const XMLListStyleCommon XMLListStyle = "common"

func init() {
	registerOutputHandler("xml", func() OutputHandler { return &XMLOutputHandler{} })
}

// Declare conformance with OutputHandler interface.
var _ OutputHandler = &XMLOutputHandler{}

// XMLOutputHandler handles serializing XML data.
type XMLOutputHandler struct {
	indent      string
	style       string
	rootTag     string
	colorize    bool
	pretty      bool
	initialized bool
	listStyle   XMLListStyle
}

func (x *XMLOutputHandler) init() {
	if x.initialized {
		return
	}

	x.colorize = false
	x.pretty = false
	x.indent = "\t"
	x.style = "native"
	x.initialized = true
	x.rootTag = "doc"
	x.listStyle = XMLListStyleEnumerated
}

// Name implements OutputHandler.Name().
func (x *XMLOutputHandler) Name() string {
	return "xml"
}

// SetOption implements OutputHandler.SetOption().
func (x *XMLOutputHandler) SetOption(name string, value string) error {
	x.init()

	switch name {
	case "output.colorize":
		x.colorize = util.StringToValue(value).(bool)
	case "output.pretty":
		x.pretty = util.StringToValue(value).(bool)
	case "output.indent":
		x.indent = value
	case "output.style":
		x.style = value
	case "xml.root-tag":
		x.rootTag = value
	case "xml.list-style":
		switch value {
		case string(XMLListStyleEnumerated):
			x.listStyle = XMLListStyleEnumerated
		case string(XMLListStyleCommon):
			x.listStyle = XMLListStyleCommon
		default:
			return fmt.Errorf("unknown xml.list-style '%s'", string(value))
		}
	}

	return nil
}

// Format implements OutputHandler.Format()
func (x *XMLOutputHandler) Format(writer io.Writer, data interface{}) error {
	x.init()

	var mv mxj.Map

	if x.listStyle == XMLListStyleEnumerated {
		m, ok := data.(map[string]interface{})
		if !ok {
			if lm, ok2 := data.([]interface{}); ok2 {
				m = make(map[string]interface{})
				for i, v := range lm {
					nodeid := "list-element" + strconv.Itoa(i)
					m[nodeid] = v
				}
			} else if lm, ok2 := data.([]map[string]interface{}); ok2 {
				m = make(map[string]interface{})
				for i, v := range lm {
					nodeid := "list-element" + strconv.Itoa(i)
					m[nodeid] = v
				}
			} else {
				m = map[string]interface{}{"object": data}
			}
		}
		mv = mxj.Map(m)
	} else if x.listStyle == XMLListStyleCommon {
		switch data.(type) {
		case map[string]any:
			mv = mxj.Map(data.(map[string]any))
		case []any:
			mv = mxj.Map(map[string]any{"list": data})
		case []map[string]any:
			mv = mxj.Map(map[string]any{"list": data})
		default:
			mv = mxj.Map(map[string]any{"object": data})
		}
	} else {
		return fmt.Errorf("unknown xml.list-style '%s'", string(x.listStyle))
	}

	var b []byte
	var err error

	if x.pretty {
		b, err = mv.XmlIndent("", x.indent, x.rootTag)
		if err != nil {
			return err
		}
	} else {
		b, err = mv.Xml(x.rootTag)
		if err != nil {
			return err
		}
	}

	if x.colorize {
		err := quick.Highlight(writer, string(b), "xml", "terminal", x.style)
		if err != nil {
			return err
		}
	} else {
		_, err := writer.Write(b)
		if err != nil {
			return err
		}
	}

	return nil
}
