# Gizmo API

![Autogenerated file](https://img.shields.io/badge/file-generated-orange.svg)

## The `graph` object

Name: `graph`, Alias: `g`

This is the only special object in the environment, generates the query objects. Under the hood, they're simple objects that get compiled to a Go iterator tree when executed.

### `graph.addDefaultNamespaces()`

AddDefaultNamespaces register all default namespaces for automatic IRI resolution.

### `graph.addNamespace(pref, ns)`

AddNamespace associates prefix with a given IRI namespace.

### `graph.emit(*)`

Emit adds data programmatically to the JSON result list. Can be any JSON type.

```javascript
g.emit({ name: "bob" }); // push {"name":"bob"} as a result
```

### `graph.loadNamespaces()`

LoadNamespaces loads all namespaces saved to graph.

### `graph.M()`

M is a shorthand for Morphism.

### `graph.Morphism()`

Morphism creates a morphism path object. Unqueryable on it's own, defines one end of the path. Saving these to variables with

```javascript
var shorterPath = graph
  .Morphism()
  .out("foo")
  .out("bar");
```

is the common use case. See also: path.follow\(\), path.followR\(\).

### `graph.IRI(s)`

Uri creates an IRI values from a given string.

### `graph.V(*)`

V is a shorthand for Vertex.

### `graph.Vertex([nodeId],[nodeId]...)`

Vertex starts a query path at the given vertex/vertices. No ids means "all vertices".

Arguments:

* `nodeId` \(Optional\): A string or list of strings representing the starting vertices.

Returns: Path object

## Path object

Both `.Morphism()` and `.Vertex()` create path objects, which provide the following traversal methods. Note that `.Vertex()` returns a query object, which is a subclass of path object.

For these examples, suppose we have the following graph:

```text
+-------+                        +------+
| alice |-----                 ->| fred |<--
+-------+     \---->+-------+-/  +------+   \-+-------+
              ----->| #bob# |       |         |*emily*|
+---------+--/  --->+-------+       |         +-------+
| charlie |    /                    v
+---------+   /                  +--------+
  \---    +--------+             |*#greg#*|
      \-->| #dani# |------------>+--------+
          +--------+
```

Where every link is a `<follows>` relationship, and the nodes with an extra `#` in the name have an extra `<status>` link. As in,

```text
<dani> -- <status> --> "cool_person"
```

Perhaps these are the influencers in our community. So too are extra `*`s in the name -- these are our smart people, according to the `<smart_graph>` label, eg, the quad:

```text
<greg> <status> "smart_person" <smart_graph> .
```

### `path.all()`

All executes the query and adds the results, with all tags, as a string-to-string \(tag to node\) map in the output set, one for each path that a traversal could take.

### `path.and(path)`

And is an alias for Intersect.

### `path.as(tags)`

As is an alias for Tag.

### `path.back([tag])`

Back returns current path to a set of nodes on a given tag, preserving all constraints.

If still valid, a path will now consider their vertex to be the same one as the previously tagged one, with the added constraint that it was valid all the way here. Useful for traversing back in queries and taking another route for things that have matched so far.

Arguments:

* `tag`: A previous tag in the query to jump back to.

Example:

```javascript
// Start from all nodes, save them into start, follow any status links,
// jump back to the starting node, and find who follows them. Return the result.
// Results are:
//   {"id": "<alice>", "start": "<bob>"},
//   {"id": "<charlie>", "start": "<bob>"},
//   {"id": "<charlie>", "start": "<dani>"},
//   {"id": "<dani>", "start": "<bob>"},
//   {"id": "<dani>", "start": "<greg>"},
//   {"id": "<dani>", "start": "<greg>"},
//   {"id": "<fred>", "start": "<greg>"},
//   {"id": "<fred>", "start": "<greg>"}
g.V()
  .Tag("start")
  .out("<status>")
  .back("start")
  .in("<follows>")
  .all();
```

### `path.both([predicatePath], [tags])`

Both follow the predicate in either direction. Same as Out or In.

Example:

```javascript
// Find all followers/followees of fred. Returns bob, emily and greg
g.V("<fred>")
  .both("<follows>")
  .all();
```

### `path.count()`

Count returns a number of results and returns it as a value.

Example:

```javascript
// Save count as a variable
var n = g.V().count();
// Send it as a query result
g.emit(n);
```

### `path.difference(path)`

Difference is an alias for Except.

### `path.except(path)`

Except removes all paths which match query from current path.

In a set-theoretic sense, this is \(A - B\). While `g.V().except(path)` to achieve `U - B = !B` is supported, it's often very slow. Example:

```javascript
var cFollows = g.V("<charlie>").out("<follows>");
var dFollows = g.V("<dani>").out("<follows>");
// People followed by both charlie (bob and dani) and dani (bob and greg) -- returns bob.
cFollows.except(dFollows).all(); // The set (dani) -- what charlie follows that dani does not also follow.
// Equivalently, g.V("<charlie>").out("<follows>").except(g.V("<dani>").out("<follows>")).all()
```

### `path.filter(args)`

Filter applies constraints to a set of nodes. Can be used to filter values by range or match strings.

#### `path.filter(regex(expression, includeIRIs))`

Filters by match a regular expression \([syntax](https://github.com/google/re2/wiki/Syntax)\). By default works only on literals unless includeEntities is set to `true`.

### `path.follow(path)`

Follow is the way to use a path prepared with Morphism. Applies the path chain on the morphism object to the current path.

Starts as if at the g.M\(\) and follows through the morphism path.

Example:

```javascript
var friendOfFriend = g
  .Morphism()
  .out("<follows>")
  .out("<follows>");
// Returns the followed people of who charlie follows -- a simplistic "friend of my friend"
// and whether or not they have a "cool" status. Potential for recommending followers abounds.
// Returns bob and greg
g.V("<charlie>")
  .follow(friendOfFriend)
  .has("<status>", "cool_person")
  .all();
```

### `path.followR(path)`

FollowR is the same as Follow but follows the chain in the reverse direction. Flips "In" and "Out" where appropriate, the net result being a virtual predicate followed in the reverse direction.

Starts at the end of the morphism and follows it backwards \(with appropriate flipped directions\) to the g.M\(\) location.

Example:

```javascript
var friendOfFriend = g
  .Morphism()
  .out("<follows>")
  .out("<follows>");
// Returns the third-tier of influencers -- people who follow people who follow the cool people.
// Returns charlie (from bob), charlie (from greg), bob and emily
g.V()
  .has("<status>", "cool_person")
  .followR(friendOfFriend)
  .all();
```

### `path.followRecursive(*)`

FollowRecursive is the same as Follow but follows the chain recursively.

Starts as if at the g.M\(\) and follows through the morphism path multiple times, returning all nodes encountered.

Example:

```javascript
var friend = g.Morphism().out("<follows>");
// Returns all people in Charlie's network.
// Returns bob and dani (from charlie), fred (from bob) and greg (from dani).
g.V("<charlie>")
  .followRecursive(friend)
  .all();
```

### `path.forEach(callback) or (limit, callback)`

ForEach calls callback\(data\) for each result, where data is the tag-to-string map as in All case.

Arguments:

* `limit` \(Optional\): An integer value on the first `limit` paths to process.
* `callback`: A javascript function of the form `function(data)`

Example:

```javascript
// Simulate query.all().all()
graph.V("<alice>").ForEach(function(d) {
  g.emit(d);
});
```

### `path.getLimit(limit)`

GetLimit is the same as All, but limited to the first N unique nodes at the end of the path, and each of their possible traversals.

### `path.has(predicate, object)`

Has filters all paths which are, at this point, on the subject for the given predicate and object, but do not follow the path, merely filter the possible paths.

Usually useful for starting with all nodes, or limiting to a subset depending on some predicate/value pair.

Arguments:

* `predicate`: A string for a predicate node.
* `object`: A string for a object node or a set of filters to find it.

Example:

```javascript
// Start from all nodes that follow bob -- results in alice, charlie and dani
g.V()
  .has("<follows>", "<bob>")
  .all();
// People charlie follows who then follow fred. Results in bob.
g.V("<charlie>")
  .out("<follows>")
  .has("<follows>", "<fred>")
  .all();
// People with friends who have names sorting lower then "f".
g.V()
  .has("<follows>", gt("<f>"))
  .all();
```

### `path.hasR(*)`

HasR is the same as Has, but sets constraint in reverse direction.

### `path.in([predicatePath], [tags])`

In is inverse of Out. Starting with the nodes in `path` on the object, follow the quads with predicates defined by `predicatePath` to their subjects.

Arguments:

* `predicatePath` \(Optional\): One of:
  * null or undefined: All predicates pointing into this node
  * a string: The predicate name to follow into this node
  * a list of strings: The predicates to follow into this node
  * a query path object: The target of which is a set of predicates to follow.
* `tags` \(Optional\): One of:
  * null or undefined: No tags
  * a string: A single tag to add the predicate used to the output set.
  * a list of strings: Multiple tags to use as keys to save the predicate used to the output set.

Example:

```javascript
// Find the cool people, bob, dani and greg
g.V("cool_person")
  .in("<status>")
  .all();
// Find who follows bob, in this case, alice, charlie, and dani
g.V("<bob>")
  .in("<follows>")
  .all();
// Find who follows the people emily follows, namely, bob and emily
g.V("<emily>")
  .out("<follows>")
  .in("<follows>")
  .all();
```

### `path.inPredicates()`

InPredicates gets the list of predicates that are pointing in to a node.

Example:

```javascript
// bob only has "<follows>" predicates pointing inward
// returns "<follows>"
g.V("<bob>")
  .inPredicates()
  .all();
```

### `path.intersect(path)`

Intersect filters all paths by the result of another query path.

This is essentially a join where, at the stage of each path, a node is shared. Example:

```javascript
var cFollows = g.V("<charlie>").out("<follows>");
var dFollows = g.V("<dani>").out("<follows>");
// People followed by both charlie (bob and dani) and dani (bob and greg) -- returns bob.
cFollows.intersect(dFollows).all();
// Equivalently, g.V("<charlie>").out("<follows>").And(g.V("<dani>").out("<follows>")).all()
```

### `path.is(node, [node..])`

Filter all paths to ones which, at this point, are on the given node.

Arguments:

* `node`: A string for a node. Can be repeated or a list of strings.

Example:

```javascript
// Starting from all nodes in the graph, find the paths that follow bob.
// Results in three paths for bob (from alice, charlie and dani).all()
g.V()
  .out("<follows>")
  .is("<bob>")
  .all();
```

### `path.labelContext([labelPath], [tags])`

LabelContext sets \(or removes\) the subgraph context to consider in the following traversals. Affects all In\(\), Out\(\), and Both\(\) calls that follow it. The default LabelContext is null \(all subgraphs\).

Arguments:

* `predicatePath` \(Optional\): One of:
  * null or undefined: In future traversals, consider all edges, regardless of subgraph.
  * a string: The name of the subgraph to restrict traversals to.
  * a list of strings: A set of subgraphs to restrict traversals to.
  * a query path object: The target of which is a set of subgraphs.
* `tags` \(Optional\): One of:
  * null or undefined: No tags
  * a string: A single tag to add the last traversed label to the output set.
  * a list of strings: Multiple tags to use as keys to save the label used to the output set.

Example:

```javascript
// Find the status of people Dani follows
g.V("<dani>")
  .out("<follows>")
  .out("<status>")
  .all();
// Find only the statuses provided by the smart_graph
g.V("<dani>")
  .out("<follows>")
  .labelContext("<smart_graph>")
  .out("<status>")
  .all();
// Find all people followed by people with statuses in the smart_graph.
g.V()
  .labelContext("<smart_graph>")
  .in("<status>")
  .labelContext(null)
  .in("<follows>")
  .all();
```

### `path.labels()`

Labels gets the list of inbound and outbound quad labels

### `path.limit(limit)`

Limit limits a number of nodes for current path.

Arguments:

* `limit`: A number of nodes to limit results to.

Example:

```javascript
// Start from all nodes that follow bob, and limit them to 2 nodes -- results in alice and charlie
g.V()
  .has("<follows>", "<bob>")
  .limit(2)
  .all();
```

### `path.map(*)`

Map is a alias for ForEach.

### `path.or(path)`

Or is an alias for Union.

### `path.out([predicatePath], [tags])`

Out is the work-a-day way to get between nodes, in the forward direction. Starting with the nodes in `path` on the subject, follow the quads with predicates defined by `predicatePath` to their objects.

Arguments:

* `predicatePath` \(Optional\): One of:
  * null or undefined: All predicates pointing out from this node
  * a string: The predicate name to follow out from this node
  * a list of strings: The predicates to follow out from this node
  * a query path object: The target of which is a set of predicates to follow.
* `tags` \(Optional\): One of:
  * null or undefined: No tags
  * a string: A single tag to add the predicate used to the output set.
  * a list of strings: Multiple tags to use as keys to save the predicate used to the output set.

Example:

```javascript
// The working set of this is bob and dani
g.V("<charlie>")
  .out("<follows>")
  .all();
// The working set of this is fred, as alice follows bob and bob follows fred.
g.V("<alice>")
  .out("<follows>")
  .out("<follows>")
  .all();
// Finds all things dani points at. Result is bob, greg and cool_person
g.V("<dani>")
  .out()
  .all();
// Finds all things dani points at on the status linkage.
// Result is bob, greg and cool_person
g.V("<dani>")
  .out(["<follows>", "<status>"])
  .all();
// Finds all things dani points at on the status linkage, given from a separate query path.
// Result is {"id": "cool_person", "pred": "<status>"}
g.V("<dani>")
  .out(g.V("<status>"), "pred")
  .all();
```

### `path.outPredicates()`

OutPredicates gets the list of predicates that are pointing out from a node.

Example:

```javascript
// bob has "<follows>" and "<status>" edges pointing outwards
// returns "<follows>", "<status>"
g.V("<bob>")
  .outPredicates()
  .all();
```

### `path.save(predicate, tag)`

Save saves the object of all quads with predicate into tag, without traversal.

Arguments:

* `predicate`: A string for a predicate node.
* `tag`: A string for a tag key to store the object node.

Example:

```javascript
// Start from dani and bob and save who they follow into "target"
// Returns:
//   {"id" : "<bob>", "target": "<fred>" },
//   {"id" : "<dani>", "target": "<bob>" },
//   {"id" : "<dani>", "target": "<greg>" }
g.V("<dani>", "<bob>")
  .save("<follows>", "target")
  .all();
```

### `path.saveInPredicates(tag)`

SaveInPredicates tags the list of predicates that are pointing in to a node.

Example:

```javascript
// bob only has "<follows>" predicates pointing inward
// returns {"id":"<bob>", "pred":"<follows>"}
g.V("<bob>")
  .saveInPredicates("pred")
  .all();
```

### `path.saveOpt(*)`

SaveOpt is the same as Save, but returns empty tags if predicate does not exists.

### `path.saveOptR(*)`

SaveOptR is the same as SaveOpt, but tags values via reverse predicate.

### `path.saveOutPredicates(tag)`

SaveOutPredicates tags the list of predicates that are pointing out from a node.

Example:

```javascript
// bob has "<follows>" and "<status>" edges pointing outwards
// returns {"id":"<bob>", "pred":"<follows>"}
g.V("<bob>")
  .saveInPredicates("pred")
  .all();
```

### `path.saveR(*)`

SaveR is the same as Save, but tags values via reverse predicate.

### `path.skip(offset)`

Skip skips a number of nodes for current path.

Arguments:

* `offset`: A number of nodes to skip.

Example:

```javascript
// Start from all nodes that follow bob, and skip 2 nodes -- results in dani
g.V()
  .has("<follows>", "<bob>")
  .skip(2)
  .all();
```

### `path.tag(tags)`

Tag saves a list of nodes to a given tag.

In order to save your work or learn more about how a path got to the end, we have tags. The simplest thing to do is to add a tag anywhere you'd like to put each node in the result set.

Arguments:

* `tag`: A string or list of strings to act as a result key. The value for tag was the vertex the path was on at the time it reached "Tag"

  Example:

```javascript
// Start from all nodes, save them into start, follow any status links, and return the result.
// Results are:
//   {"id": "cool_person", "start": "<bob>"},
//   {"id": "cool_person", "start": "<dani>"},
//   {"id": "cool_person", "start": "<greg>"},
//   {"id": "smart_person", "start": "<emily>"},
//   {"id": "smart_person", "start": "<greg>"}
g.V()
  .tag("start")
  .out("<status>")
  .all();
```

### `path.tagArray(*)`

TagArray is the same as ToArray, but instead of a list of top-level nodes, returns an Array of tag-to-string dictionaries, much as All would, except inside the JS environment.

Example:

```javascript
// bobTags contains an Array of followers of bob (alice, charlie, dani).
var bobTags = g
  .V("<bob>")
  .tag("name")
  .in("<follows>")
  .tagArray();
// nameValue should be the string "<bob>"
var nameValue = bobTags[0]["name"];
```

### `path.tagValue()`

TagValue is the same as TagArray, but limited to one result node. Returns a tag-to-string map.

### `path.toArray(*)`

ToArray executes a query and returns the results at the end of the query path as an JS array.

Example:

```javascript
// bobFollowers contains an Array of followers of bob (alice, charlie, dani).
var bobFollowers = g
  .V("<bob>")
  .in("<follows>")
  .toArray();
```

### `path.toValue()`

ToValue is the same as ToArray, but limited to one result node.

### `path.union(path)`

Union returns the combined paths of the two queries.

Notice that it's per-path, not per-node. Once again, if multiple paths reach the same destination, they might have had different ways of getting there \(and different tags\). See also: `path.Tag()`

Example:

```javascript
var cFollows = g.V("<charlie>").out("<follows>");
var dFollows = g.V("<dani>").out("<follows>");
// People followed by both charlie (bob and dani) and dani (bob and greg) -- returns bob (from charlie), dani, bob (from dani), and greg.
cFollows.union(dFollows).all();
```

### `path.unique()`

Unique removes duplicate values from the path.

### `path.order()`

Order returns values from the path in ascending order.

