// Copyright ©2020 The go-fonts Authors. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Copyright 2016 The Go Authors. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// +build ignore

package main

import (
	"bytes"
	"flag"
	"fmt"
	"go/format"
	"io"
	"io/ioutil"
	"log"
	"net/http"
	"os"
	"path/filepath"
	"strings"
)

func main() {
	log.SetPrefix("stix-gen: ")
	log.SetFlags(0)

	var (
		tmpl = flag.String("src", "https://github.com/stipub/stixfonts/raw/master/OTF", "remote directory holding OTF files for STIX fonts")
	)

	flag.Parse()

	for _, fname := range []string{
		"STIX2Math.otf",
		"STIX2Text-Bold.otf",
		"STIX2Text-BoldItalic.otf",
		"STIX2Text-Italic.otf",
		"STIX2Text-Regular.otf",
	} {
		err := gen(*tmpl, fname)
		if err != nil {
			log.Fatalf("could not generate package %q: %+v", fname, err)
		}
	}
}

func gen(dir, fname string) error {
	log.Printf("generating fonts package for %q...", fname)

	resp, err := http.Get(dir + "/" + fname)
	if err != nil {
		return fmt.Errorf("could not GET input file: %w", err)
	}
	defer resp.Body.Close()

	raw := new(bytes.Buffer)

	_, err = io.Copy(raw, resp.Body)
	if err != nil {
		return fmt.Errorf("could not download TTF file: %w", err)
	}

	err = do(fname, raw.Bytes())
	if err != nil {
		return fmt.Errorf("could not generate package for %q: %w", fname, err)
	}

	return nil
}

func do(ttfName string, src []byte) error {
	fontName := fontName(ttfName)
	pkgName := pkgName(ttfName)
	if err := os.Mkdir(pkgName, 0777); err != nil && !os.IsExist(err) {
		return fmt.Errorf("could not create package dir %q: %w", pkgName, err)
	}

	desc := "a proportional-width, serif"
	b := new(bytes.Buffer)
	fmt.Fprintf(b, "// generated by go run gen-fonts.go; DO NOT EDIT\n\n")
	fmt.Fprintf(b, "// Package %s provides the %q TrueType font\n", pkgName, fontName)
	fmt.Fprintf(b, "// from the STIX2 font family. It is %s font.\n", desc)
	fmt.Fprintf(b, "package %[1]s // import \"github.com/go-fonts/stix/%[1]s\"\n\n", pkgName)
	fmt.Fprintf(b, "// TTF is the data for the %q TrueType font.\n", fontName)
	fmt.Fprintf(b, "var TTF = []byte{")
	for i, x := range src {
		if i&15 == 0 {
			b.WriteByte('\n')
		}
		fmt.Fprintf(b, "%#02x,", x)
	}
	fmt.Fprintf(b, "\n}\n")

	dst, err := format.Source(b.Bytes())
	if err != nil {
		return fmt.Errorf("could not format source: %w", err)
	}

	err = ioutil.WriteFile(filepath.Join(pkgName, "data.go"), dst, 0666)
	if err != nil {
		return fmt.Errorf("could not write package source file: %w", err)
	}

	return nil
}

const suffix = ".otf"

// fontName maps "Go-Regular.ttf" to "Go Regular".
func fontName(ttfName string) string {
	s := ttfName[:len(ttfName)-len(suffix)]
	s = strings.Replace(s, "-", " ", -1)
	return s
}

// pkgName maps "Go-Regular.ttf" to "goregular".
func pkgName(ttfName string) string {
	s := ttfName[:len(ttfName)-len(suffix)]
	s = strings.Replace(s, "-", "", -1)
	s = strings.ToLower(s)
	return s
}
