/*  Move

    Copyright (c) Joe Cosentino 1997-2002.
    All Rights Reserved.

    Some code borrowed from Replace by Rene Ableidinger.

*/

// I N C L U D E S //////////////////////////////////////////////////////////

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <io.h>
#include <dir.h>
#include <dos.h>
#include <sys/stat.h>

// D E F I N E S ////////////////////////////////////////////////////////////

#define DIR_SEPARATOR "\\"

#define F_OK 0                          // Exists.
#define R_OK 4                          // Readable.
#define W_OK 2                          // Writeable.
#define X_OK 1                          // Executable.

// G L O B A L S ////////////////////////////////////////////////////////////

char opt_verify=0,opt_help=0,dest_drive;

// P R O T O T Y P E S //////////////////////////////////////////////////////

void classify_args(char, char *[], char *, char *[], char *, char *[]);
void error(char *, char *, const char *);
void replace_files(char [], char [], char []);
void replace_file(char [], char []);
void copy_file(char [], char []);

// F U N C T I O N S ////////////////////////////////////////////////////////

void classify_args(char argc, char *argv[], char *fileargc, char *fileargv[], char *optargc, char *optargv[])
{
    char i, *argptr;

    *fileargc=0;
    *optargc=0;
    for (i=1;i<argc;i++)
        {
        argptr=argv[i];
        if (argptr[0] == '/')
            {
            optargv[*optargc]=argptr + 1;
            *optargc=*optargc + 1;
            } // end if.
        else
            {
            fileargv[*fileargc]=argptr;
            *fileargc=*fileargc + 1;
            } // end else.

        } // end for.

} // end classify_args.

/////////////////////////////////////////////////////////////////////////////

void error(char *source, char *dest, const char *error)
{
    printf("Error copying %s to %s [%s]\n", strupr(source), strupr(dest), error);

} // end error.

/////////////////////////////////////////////////////////////////////////////

void replace_files(char src_pathname[], char src_filemask[], char dest_pathname[])
{
    char filemask[MAXPATH],new_dest_pathname[MAXPATH],src_filename[MAXPATH],dest_filename[MAXPATH];
    struct ffblk fileblock;
    int fileattrib,done;

    fileattrib=FA_RDONLY + FA_ARCH;
    fileattrib=fileattrib + FA_HIDDEN + FA_SYSTEM;
    strcpy(filemask, src_pathname);
    strcat(filemask, src_filemask);
    done=findfirst(filemask, &fileblock, fileattrib);
    while (!done)
        {
        // Rebuild source filename.
        strcpy(src_filename, src_pathname);
        strcat(src_filename, fileblock.ff_name);

        // Build destination filename.
        strcpy(dest_filename, dest_pathname);
        strcat(dest_filename, fileblock.ff_name);

        replace_file(src_filename, dest_filename);
        done=findnext(&fileblock);
        } // end while.

} // end replace_files.

/////////////////////////////////////////////////////////////////////////////

void replace_file(char src_filename[], char dest_filename[])
{
    char ch,msg_prompt[255],*s;
    int dest_file_exists,fileattrib;
    struct stat src_statbuf,dest_statbuf;
    struct dfree disktable;
    unsigned long free_diskspace;

    // Check source file for read permission.
    if (access(src_filename, R_OK) != 0)
        {
        error(src_filename, dest_filename, "Access denied");
        exit(4);
        } // end if.

    // Get info of source and destination file.
    stat(src_filename, &src_statbuf);
    dest_file_exists=!stat(dest_filename, &dest_statbuf);

    // Get amount of free disk space in destination drive.
    getdfree(dest_drive, &disktable);
    free_diskspace=(unsigned long) disktable.df_avail * disktable.df_sclus * disktable.df_bsec;
    if (dest_file_exists)
        {
        return;
        } // end if.

    // Check free space on destination disk.
    if (src_statbuf.st_size > free_diskspace - dest_statbuf.st_size)
        {
        error(src_filename, dest_filename, "Insufficient disk space");
        exit(4);
        } // end if.

    copy_file(src_filename, dest_filename);

} // end replace_file.

/////////////////////////////////////////////////////////////////////////////

void copy_file(char src_filename[], char dest_filename[])
{
    FILE *src_file,*dest_file;
    char buffer[BUFSIZ];
    int buffersize,fileattrib;
    struct ftime filetime;

    printf("%s => %s ", src_filename, dest_filename);

    // Open source file.
    src_file=fopen(src_filename, "rb");
    if (src_file == NULL)
        {
        error(src_filename, dest_filename, "Cannot open source file");
        exit(4);
        } // end if.

    // Open destination file.
    dest_file=fopen(dest_filename, "wb");
    if (dest_file == NULL)
        {
        error(src_filename, dest_filename, "Cannot create destination file");
        fclose(src_file);
        exit(5);
        } // end if.

    // Copy file data.
    buffersize=fread(buffer, sizeof(char), BUFSIZ, src_file);
    while (buffersize > 0)
        {
        if (fwrite(buffer, sizeof(char), buffersize, dest_file) != buffersize)
            {
            error(src_filename, dest_filename, "Error writing destination file");
            fclose(src_file);
            fclose(dest_file);
            exit(5);
            } // end if.

        buffersize=fread(buffer, sizeof(char), BUFSIZ, src_file);
        } // end while.

    // Copy file timestamp.
    getftime(fileno(src_file), &filetime);
    setftime(fileno(dest_file), &filetime);

    // Close files.
    fclose(src_file);
    fclose(dest_file);

    // Copy file attributes.
    fileattrib=_chmod(src_filename, 0);
    _chmod(dest_filename, 1, fileattrib);
    unlink(src_filename);               // Delete file.
    printf("[ok]\n");

} // end copy_file.

/////////////////////////////////////////////////////////////////////////////

int main(int argc, char *argv[])
{
    char fileargc,*fileargv[255],optargc,*optargv[255],cur_pathname[MAXPATH]="",src_pathname[MAXPATH]="",src_filemask[MAXPATH]="",dest_pathname[MAXPATH]="",*ptr,option[255]="",i;

    classify_args(argc, argv, &fileargc, fileargv, &optargc, optargv);
    for (i=0;i<optargc;i++)             // Get options.
        {
        strcpy(option, optargv[i]);
        if (stricmp(option, "v") == 0)
            opt_verify=-1;
        else if (stricmp(option, "?") == 0)
            opt_help=-1;
        else
            {
            printf("Invalid parameter - %s\n", optargv[i]);
            exit(4);
            } // end else.

        } // end for.

    if (opt_help)
        {
        printf("Moves a file to another location.\n\n"
               "Syntax: MOVE source destination [/?]\n"
               "  source      The name of the file you want to move\n"
               "  destination Where you want to move the file to\n"
               "  /V          Verifies each file as it is written to the destination file\n"
               "              to make sure that the destination files are identical to the\n"
               "              source files\n"
               "  /?          Displays this help message\n");
        exit(0);
        } // end if.

    if (argc < 2 || fileargc < 1)
        {
        printf("Required parameter missing\n");
        return 1;
        } // end if.

    getcwd(cur_pathname, MAXPATH);
    _fullpath(src_pathname, fileargv[0], MAXPATH);
    ptr=strrchr(src_pathname, (char)* DIR_SEPARATOR);
    strcpy(src_filemask, ptr + 1);
    *ptr='\0';
    if (strlen(src_pathname) <= 2)
        {
        strcat(src_pathname, DIR_SEPARATOR);
        } // end if.

    if (chdir(src_pathname) != 0)       // Check source path.
        {
        error(src_pathname, dest_pathname, "Path not found");
        exit(1);
        } // end if.

    chdir(cur_pathname);                // Restore path.
    if (strlen(src_pathname) > 3)
        {
        // Source path is not a root directory -> add backslash.
        strcat(src_pathname, DIR_SEPARATOR);
        } // end if.

    if (fileargc < 2)                   // Get destination pathname.
        {
        // No destination path specified -> use current.
        strcpy(dest_pathname, cur_pathname);
        } // end if.
    else
        {
        _fullpath(dest_pathname, fileargv[1], MAXPATH);
        if (chdir(dest_pathname) != 0)
            {
            error(src_pathname, dest_pathname, "Destination path not found");
            exit(4);
            } // end if.

        chdir(cur_pathname);            // Restore path.
        } // end else.

    strcat(dest_pathname, DIR_SEPARATOR);
    if (strcmp(src_pathname, dest_pathname) == 0)
        {
        error(src_pathname, dest_pathname, "Source and destination path cannot be the same");
        exit(4);
        } // end if.

    dest_drive=toupper(dest_pathname[0]) - 64;
    if (opt_verify)
        setverify(1);

    replace_files(src_pathname, src_filemask, dest_pathname);
    return 0;

} // end main.
